package io.embrace.android.embracesdk;

import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

import java9.util.Sets;
import java9.util.stream.Collectors;
import java9.util.stream.StreamSupport;

class EmbraceUserService implements ActivityListener, UserService {


    static final Pattern VALID_PERSONA = Pattern.compile("^[a-zA-Z0-9_]{1,32}$");

    // Maximum number of allowed personas
    private static final int PERSONA_LIMIT = 10;

    private final PreferencesService preferencesService;

    private final ApiClient apiClient;

    private volatile UserInfo userInfo = UserInfo.ofEmpty();

    EmbraceUserService(
            ActivityService activityService,
            PreferencesService preferencesService,
            ApiClient apiClient) {

        this.preferencesService = preferencesService;
        this.apiClient = apiClient;
        this.userInfo = UserInfo.ofStored(preferencesService);
    }

    @Override
    public UserInfo getUserInfo() {
        return userInfo;
    }

    @Override
    public void setUserIdentifier(String userId) {
        UserInfo newUserInfo = UserInfo.newBuilder(userInfo)
                .withUserId(userId)
                .build();
        this.userInfo = newUserInfo;
        preferencesService.setUserIdentifier(userId);
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void clearUserIdentifier() {
        setUserIdentifier(null);
    }

    @Override
    public void setUserEmail(String email) {
        UserInfo newUserInfo = UserInfo.newBuilder(userInfo)
                .withEmail(email)
                .build();
        this.userInfo = newUserInfo;
        preferencesService.setUserEmailAddress(email);
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void clearUserEmail() {
        setUserEmail(null);
    }

    @Override
    public void setUserAsPayer() {
        setUserPersona(UserInfo.PERSONA_PAYER);
    }

    @Override
    public void clearUserAsPayer() {
        clearUserPersona(UserInfo.PERSONA_PAYER);
    }

    @Override
    public void setUserPersona(String persona) {
        if (!VALID_PERSONA.matcher(persona).matches()) {
            EmbraceLogger.logWarning("Ignoring persona " + persona + " as it does not match " + VALID_PERSONA.pattern());
            return;
        }
        if (userInfo.getPersonas().size() >= PERSONA_LIMIT) {
            EmbraceLogger.logWarning("Cannot set persona as the limit of " + PERSONA_LIMIT + " has been reached");
            return;
        }
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (builder.getPersonas() != null) {
            personas.addAll(builder.getPersonas());
        }
        personas.add(persona);
        UserInfo newUserInfo = builder
        .withPersonas(personas)
        .build();
        this.userInfo = newUserInfo;
        preferencesService.setUserPersonas(personas);
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void clearUserPersona(String persona) {
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (builder.getPersonas() != null) {
            personas.addAll(builder.getPersonas());
        }
        personas.remove(persona);
        UserInfo newUserInfo = builder
                .withPersonas(personas)
                .build();
        this.userInfo = newUserInfo;
        preferencesService.setUserPersonas(personas);
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void clearAllUserPersonas() {
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (preferencesService.getUserPayer()) {
            personas.add(UserInfo.PERSONA_PAYER);
        }
        if (preferencesService.isUsersFirstDay()) {
            personas.add(UserInfo.PERSONA_FIRST_DAY_USER);
        }
        
        UserInfo newUserInfo = builder
                .withPersonas(personas)
                .build();
        this.userInfo = newUserInfo;
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void setUsername(String username) {
        UserInfo newUserInfo = UserInfo.newBuilder(userInfo)
                .withUsername(username)
                .build();
        this.userInfo = newUserInfo;
        preferencesService.setUsername(username);
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void clearUsername() {
        setUsername(null);
    }
}
