package io.embrace.android.embracesdk;

import android.util.Log;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.Map;

/**
 * Wrapper for the Android {@link Log} utility. This class is part of the Embrace Public API.
 */
public class EmbraceLogger {
    public enum Severity {
        DEBUG,
        INFO,
        WARNING,
        ERROR,
        NONE
    }

    /** The default severity. */
    private static Severity threshold = Severity.INFO;

    private EmbraceLogger() {
        // Restricted constructor
    }

    /**
     * Sets the logging threshold. Anything below the threshold will not be logged.
     *
     * @param threshold the threshold
     */
    public static void setThreshold(Severity threshold) {
        EmbraceLogger.threshold = Preconditions.checkNotNull(threshold, "threshold must not be null");
    }

    /**
     * Logs a debug message.
     *
     * @param msg the message to log
     */
    public static void logDebug(String msg) {
        log(Severity.DEBUG, msg, null, false);
    }

    /**
     * Logs a debug message and sends exception to Embrace.
     *
     * @param msg the message to log
     * @param throwable the exception
     */
    public static void logDebug(String msg, Throwable throwable) {
        log(Severity.DEBUG, msg, throwable, null);
    }

    /**
     * Logs a debug message with a stack trace.
     *
     * @param msg the message to log
     * @param throwable the exception
     * @param logStacktrace include stacktrace in log
     */
    public static void logDebug(String msg, Throwable throwable, Boolean logStacktrace) {
        log(Severity.DEBUG, msg, throwable, logStacktrace);
    }

    /**
     * Logs an info message.
     *
     * @param msg the message to log
     */
    public static void logInfo(String msg) {
        log(Severity.INFO, msg, null, false);
    }

    /**
     * Logs an info message and sends exception to Embrace.
     *
     * @param msg the message to log
     * @param throwable the exception
     */
    public static void logInfo(String msg, Throwable throwable) {
        log(Severity.INFO, msg, throwable, false);
    }

    /**
     * Logs an info message with a stack trace.
     *
     * @param msg the message to log
     * @param throwable the exception
     */
    public static void logInfo(String msg, Throwable throwable, Boolean logStacktrace) {
        log(Severity.INFO, msg, throwable, logStacktrace);
    }

    /**
     * Logs a warning message.
     *
     * @param msg the message to log
     */
    public static void logWarning(String msg) {
        log(Severity.WARNING, msg, null, false);
    }

    /**
     * Logs a warning message and sends exception to Embrace.
     *
     * @param msg the message to log
     * @param throwable the exception
     */
    public static void logWarning(String msg, Throwable throwable) {
        log(Severity.WARNING, msg, throwable, false);
    }

    /**
     * Logs a warning message with a stack trace.
     *
     * @param msg the message to log
     * @param throwable the exception
     * @param logStacktrace include stacktrace in log
     */
    public static void logWarning(String msg, Throwable throwable, Boolean logStacktrace) {
        log(Severity.WARNING, msg, throwable, logStacktrace);
    }

    /**
     * Logs an error message.
     *
     * @param msg the message to log
     */
    public static void logError(String msg) {
        log(Severity.ERROR, msg, null, null);
    }

    /**
     * Logs an error and sends exception to Embrace.
     *
     * @param msg the message to log
     * @param throwable the exception
     */
    public static void logError(String msg, Throwable throwable) {
        log(Severity.ERROR, msg, throwable, false);
    }

    /**
     * Logs an error message with a stack trace.
     *
     * @param msg the message to log
     * @param throwable the exception
     * @param logStacktrace include stacktrace in log
     */
    public static void logError(String msg, Throwable throwable, Boolean logStacktrace) {
        log(Severity.ERROR, msg, throwable, logStacktrace);
    }

    /**
     * Logs a message with a stack trace at the specified severity. The 'data' attributes are
     * ignored.
     *
     * @param severity the severity of the message
     * @param msg the message to log
     * @param throwable the exception
     * @param logStacktrace include stacktrace in log
     */
    public static void log(Severity severity, String msg, Throwable throwable, Boolean logStacktrace) {
        if (severity == null) {
            throw new IllegalArgumentException("Severity is null or blank.");
        } else if (msg == null) {
            msg = "";
        }

        if (severity.ordinal() >= threshold.ordinal()) {
            if (logStacktrace) {
                Log.e("[Embrace]", String.format("%s", msg), throwable);
            } else {
                Log.e("[Embrace", String.format("%s", msg));
            }
        }

        if(throwable != null){
            try{
                Embrace.getInstance().getExceptionsService().handleExceptionError(throwable);
            } catch (Exception e){
                Log.e("[Embrace]", e.getLocalizedMessage());
            }
        }
    }
}
