package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;

/**
 * Logs messages remotely, so that they can be viewed as events during a user's session.
 */
class EmbraceRemoteLogger implements MemoryCleanerListener {

    private final MetadataService metadataService;

    private final ScreenshotService screenshotService;

    private final ApiClient apiClient;

    private final UserService userService;

    private final ConfigService configService;

    private final NavigableMap<Long, String> infoLogIds = new ConcurrentSkipListMap<>();
    private final NavigableMap<Long, String> warningLogIds = new ConcurrentSkipListMap<>();
    private final NavigableMap<Long, String> errorLogIds = new ConcurrentSkipListMap<>();


    EmbraceRemoteLogger(
            MetadataService metadataService,
            ScreenshotService screenshotService,
            ApiClient apiClient,
            UserService userService,
            ConfigService configService,
            MemoryCleanerService memoryCleanerService) {

        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.screenshotService = Preconditions.checkNotNull(screenshotService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.userService = Preconditions.checkNotNull(userService);
        this.configService = Preconditions.checkNotNull(configService);
        Preconditions.checkNotNull(memoryCleanerService).addListener(this);
    }

    /**
     * Gets the current thread's stack trace.
     *
     * @return the stack trace for the current thread
     */
    static Stacktraces getStackTraceForCurrentThread() {
        List<String> augmentedStackReturnAddresses = new ArrayList<>();
        for (StackTraceElement element : Thread.currentThread().getStackTrace()) {
            augmentedStackReturnAddresses.add(element.toString());
        }
        Stacktraces stacktraces = new Stacktraces(augmentedStackReturnAddresses);
        return stacktraces;
    }

    /**
     * Creates a remote log.
     *
     * @param message        the message to log
     * @param type           the type of message to log, which must be INFO_LOG, WARNING_LOG, or ERROR_LOG
     * @param takeScreenshot whether to take a screenshot when logging the event
     * @param properties     custom properties to send as part of the event
     */
    void log(String message,
             EmbraceEvent.Type type,
             boolean takeScreenshot,
             Map<String, Object> properties) {

        if (configService.isLogMessageDisabled(message)) {
            EmbraceLogger.logWarning(String.format("Log message disabled. Ignoring log with message %s", message));
            return;
        }

        if (configService.isMessageTypeDisabled(MessageType.LOG)) {
            EmbraceLogger.logWarning("Log message disabled. Ignoring all Logs.");
            return;
        }

        String id = Uuid.getEmbUuid();
        long timestamp = System.currentTimeMillis();
        if (type.equals(EmbraceEvent.Type.INFO_LOG)) {
            infoLogIds.put(timestamp, id);
        } else if (type.equals(EmbraceEvent.Type.WARNING_LOG)) {
            warningLogIds.put(timestamp, id);
        } else if (type.equals(EmbraceEvent.Type.ERROR_LOG)) {
            errorLogIds.put(timestamp, id);
        } else {
            EmbraceLogger.logWarning("Unknown log level " + type.toString());
            return;
        }
        Event.Builder builder = Event.newBuilder()
                .withType(type)
                .withName(message)
                .withTimestamp(System.currentTimeMillis())
                .withAppState(metadataService.getAppState())
                .withMessageId(id)
                .withCustomProperties(properties);
        Optional<String> optionalSessionId = metadataService.getActiveSessionId();
        if (optionalSessionId.isPresent()) {
            builder = builder.withSessionId(optionalSessionId.get());
        }
        if (takeScreenshot && !configService.isScreenshotDisabledForEvent(message)) {
            boolean screenshotTaken = screenshotService.takeScreenshotLogEvent(id);
            builder = builder.withScreenshotTaken(screenshotTaken);
        }
        Event event = builder.build();
        EventMessage eventMessage = EventMessage.newBuilder()
                .withEvent(event)
                .withStacktraces(getStackTraceForCurrentThread())
                .withDeviceInfo(metadataService.getDeviceInfo())
                .withTelephonyInfo(metadataService.getTelephonyInfo())
                .withAppInfo(metadataService.getAppInfo())
                .withUserInfo(userService.getUserInfo())
                .build();
        apiClient.sendLogs(eventMessage);
    }

    /**
     * Finds all IDs of log events at info level within the given time window.
     *
     * @param startTime the beginning of the time window
     * @param endTime   the end of the time window
     * @return the list of log IDs within the specified range
     */
    List<String> findInfoLogIds(long startTime, long endTime) {
        return new ArrayList<>(this.infoLogIds.subMap(startTime, endTime).values());
    }

    /**
     * Finds all IDs of log events at warning level within the given time window.
     *
     * @param startTime the beginning of the time window
     * @param endTime   the end of the time window
     * @return the list of log IDs within the specified range
     */
    List<String> findWarningLogIds(long startTime, long endTime) {
        return new ArrayList<>(this.warningLogIds.subMap(startTime, endTime).values());
    }

    /**
     * Finds all IDs of log events at error level within the given time window.
     *
     * @param startTime the beginning of the time window
     * @param endTime   the end of the time window
     * @return the list of log IDs within the specified range
     */
    List<String> findErrorLogIds(long startTime, long endTime) {
        return new ArrayList<>(this.errorLogIds.subMap(startTime, endTime).values());
    }

    @Override
    public void cleanCollections() {
        this.infoLogIds.clear();
        this.warningLogIds.clear();
        this.errorLogIds.clear();
    }
}
