package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.util.concurrent.TimeUnit;

import io.embrace.android.embracesdk.utils.exceptions.Unchecked;

import static io.embrace.android.embracesdk.EmbracePreferencesService.SDK_STARTUP_COMPLETED;
import static io.embrace.android.embracesdk.Session.Builder;
import static io.embrace.android.embracesdk.Session.SessionLifeEventType;
import static io.embrace.android.embracesdk.Session.newBuilder;

/**
 * Handles the lifecycle of an Embrace session.
 * <p>
 * A session encapsulates all metrics gathered between the app being foregrounded and the app being
 * backgrounded. A caching service runs on an interval of every
 * {@value EmbraceSessionService#SESSION_CACHING_INTERVAL} seconds which saves the current session
 * to disk, so if the app is terminated, the session can resume from where it left off.
 */
final class EmbraceSessionService implements SessionService, ActivityListener {

    /**
     * Signals to the API that the application was in the foreground.
     */
    public static final String APPLICATION_STATE_ACTIVE = "active";
    /**
     * Signals to the API that the application was in the background.
     */
    public static final String APPLICATION_STATE_BACKGROUND = "background";
    /**
     * Signals to the API the start of a session.
     */
    private static final String SESSION_START_TYPE = "st";
    /**
     * Signals to the API the end of a session.
     */
    private static final String SESSION_END_TYPE = "en";
    /**
     * The name of the file containing the cached session.
     */
    private static final String SESSION_FILE_NAME = "last_session.json";
    /**
     * Sesion caching interval in seconds.
     */
    private static final int SESSION_CACHING_INTERVAL = 2;
    /**
     * ApiClient timeout for the active session send service
     */
    private static final int SEND_SESSION_API_CLIENT_TIMEOUT = 2;
    /**
     * Synchronization lock.
     */
    private final Object lock = new Object();
    /**
     * Embrace service dependencies of the session service.
     */
    private final PreferencesService preferencesService;
    private final PerformanceInfoService performanceInfoService;
    private final MetadataService metadataService;
    private final BreadcrumbService breadcrumbService;
    private final PowerService powerService;
    private final ActivityService activityService;
    private final ApiClient apiClient;
    private final EventService eventService;
    private final EmbraceRemoteLogger remoteLogger;
    private final UserService userService;
    private final ConfigService configService;
    private final CacheService cacheService;
    private final EmbraceExceptionService exceptionService;
    private final BuildInfo buildInfo;

    /**
     * Asynchronous workers.
     */
    private final BackgroundWorker sessionBackgroundWorker;
    private volatile ScheduledWorker sessionCacheWorker;
    private volatile ScheduledWorker automaticSessionCloser;

    /**
     * The currently active session.
     */
    private volatile Session activeSession;

    EmbraceSessionService(
            PreferencesService preferencesService,
            PerformanceInfoService performanceInfoService,
            MetadataService metadataService,
            BreadcrumbService breadcrumbService,
            PowerService powerService,
            ActivityService activityService,
            ApiClient apiClient,
            EventService eventService,
            EmbraceRemoteLogger remoteLogger,
            UserService userService,
            ConfigService configService,
            CacheService cacheService,
            EmbraceExceptionService exceptionService,
            BuildInfo buildInfo) {

        this.preferencesService = Preconditions.checkNotNull(preferencesService);
        this.performanceInfoService = Preconditions.checkNotNull(performanceInfoService);
        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.breadcrumbService = Preconditions.checkNotNull(breadcrumbService);
        this.powerService = Preconditions.checkNotNull(powerService);
        this.activityService = Preconditions.checkNotNull(activityService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.eventService = Preconditions.checkNotNull(eventService);
        this.remoteLogger = Preconditions.checkNotNull(remoteLogger);
        this.userService = Preconditions.checkNotNull(userService);
        this.configService = Preconditions.checkNotNull(configService);
        this.cacheService = Preconditions.checkNotNull(cacheService);
        this.exceptionService = Preconditions.checkNotNull(exceptionService);
        this.activityService.addListener(this, true);
        this.sessionBackgroundWorker = BackgroundWorker.ofSingleThread("Session");
        this.buildInfo = Preconditions.checkNotNull(buildInfo);
    }

    /**
     * @return session number incremented by 1
     */
    private int incrementAndGetSessionNumber() {
        if (preferencesService.getSessionNumber().isPresent()) {
            preferencesService.setSessionNumber(preferencesService.getSessionNumber().get() + 1);
        } else {
            preferencesService.setSessionNumber(1);
        }
        return preferencesService.getSessionNumber().get();
    }

    private void startCaching() {
        this.sessionCacheWorker = ScheduledWorker.ofSingleThread("Session Caching Service");
        this.sessionCacheWorker.scheduleAtFixedRate(
                this::cacheActiveSession,
                0,
                SESSION_CACHING_INTERVAL,
                TimeUnit.SECONDS);
    }

    private void stopCaching() {
        if (this.sessionCacheWorker != null) {
            this.sessionCacheWorker.close();
        }
    }

    private void startSessionCloser(int maxSessionSeconds) {
        this.automaticSessionCloser = ScheduledWorker.ofSingleThread("Session Closer Service");
        this.automaticSessionCloser.scheduleAtFixedRate(
                () -> {
                    try {
                        synchronized (lock) {
                            EmbraceLogger.logInfo("Automatic session closing triggered.");
                            triggerStatelessSessionEnd(SessionLifeEventType.TIMED);
                        }
                    } catch (Exception ex) {
                        EmbraceLogger.logError("Error while trying to close the session " +
                                "automatically", ex);
                    }
                },
                maxSessionSeconds, maxSessionSeconds, TimeUnit.SECONDS);
    }

    private void stopSessionCloser() {
        if (this.automaticSessionCloser != null) {
            this.automaticSessionCloser.close();
        }
    }

    @Override
    public void startSession(boolean coldStart, SessionLifeEventType startType) {

        if (configService.isMessageTypeDisabled(MessageType.SESSION)) {
            EmbraceLogger.logWarning("Session messages disabled. Ignoring all sessions.");
            return;
        }

        String sessionId = Uuid.getEmbUuid();
        SessionMessage message = sessionStartMessage(coldStart, sessionId, startType);
        this.activeSession = message.getSession();
        metadataService.setActiveSessionId(sessionId);
        apiClient.sendSession(message);

        //If getMaxSessionSeconds is not null, schedule the session closer.
        Optional<BuildInfo.SessionConfig> sessionConfig = buildInfo.getSessionConfig();
        if (sessionConfig.isPresent() && sessionConfig.get().getMaxSessionSeconds().isPresent()) {
            startSessionCloser(sessionConfig.get().getMaxSessionSeconds().get());
        }

        addViewBreadcrumbForResumedSession();
        startCaching();
    }

    @Override
    public boolean sdkStartupFailedLastSession() {
        Optional<String> lastSessionStartupProgress = preferencesService.getSDKStartupStatus();
        return lastSessionStartupProgress.isPresent() && !lastSessionStartupProgress.get().equals(SDK_STARTUP_COMPLETED);
    }

    @Override
    public void handleCrash(String stacktrace) {
        synchronized (lock) {
            if (this.activeSession != null) {
                SessionMessage sessionMessage = sessionEndMessage(
                        this.activeSession,
                        false,
                        false,
                        true,
                        stacktrace,
                        SessionLifeEventType.STATE);
                Unchecked.wrap(() -> apiClient.sendSession(sessionMessage).get());
                cacheService.deleteObject(SESSION_FILE_NAME);
            }
        }
    }

    private SessionMessage sessionStartMessage(boolean coldStart, String id, SessionLifeEventType startType) {

        Builder builder = newBuilder();

        Session session = builder
                .withSessionId(id)
                .withStartTime(System.currentTimeMillis())
                .withNumber(incrementAndGetSessionNumber())
                .withColdStart(coldStart)
                .withStartType(startType)
                .withSessionType(SESSION_START_TYPE)
                .build();

        if (powerService.getLatestBatteryLevel().isPresent()) {
            builder.withStartingBatteryLevel(powerService.getLatestBatteryLevel().get());
        }

        return SessionMessage.newBuilder()
                .withSession(session)
                .withDeviceInfo(metadataService.getDeviceInfo())
                .withAppInfo(metadataService.getAppInfo())
                .build();
    }

    /**
     * Caches the session, with performance information generated up to the current point.
     */
    private void cacheActiveSession() {
        try {
            synchronized (lock) {
                if (activeSession != null) {
                    SessionMessage sessionMessage = sessionEndMessage(activeSession, false, true, SessionLifeEventType.STATE);
                    cacheService.cacheObject(SESSION_FILE_NAME, sessionMessage, SessionMessage.class);
                }
            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Error while caching active session", ex);
        }
    }

    private SessionMessage sessionEndMessage(Session session,
                                             boolean endedCleanly,
                                             boolean forceQuit,
                                             SessionLifeEventType endType) {
        return this.sessionEndMessage(session, endedCleanly, forceQuit, false, null, endType);
    }

    private SessionMessage sessionEndMessage(Session session,
                                             boolean endedCleanly,
                                             boolean forceQuit,
                                             boolean crashed,
                                             String stacktrace,
                                             SessionLifeEventType endType) {

        Long startTime = session.getStartTime();
        long endTime = System.currentTimeMillis();
        Builder builder = newBuilder(session)
                .withEndedCleanly(endedCleanly)
                .withLastState(getApplicationState())
                .withSessionType(SESSION_END_TYPE)
                .withEventIds(eventService.findEventIdsForSession(startTime, endTime))
                .withInfoLogIds(remoteLogger.findInfoLogIds(startTime, endTime))
                .withWarningLogIds(remoteLogger.findWarningLogIds(startTime, endTime))
                .withErrorLogIds(remoteLogger.findErrorLogIds(startTime, endTime))
                .withExceptionErrors(exceptionService.getCurrentExceptionError())
                .withCrashed(crashed)
                .withLastHeartbeatTime(System.currentTimeMillis())
                .withEndType(endType);

        if (crashed) {
            // if it's a crash session, then add the stacktrace to the session payload
            builder = builder
                    .withExceptionStacktrace(stacktrace);
        }
        if (forceQuit) {
            builder = builder
                    .withTerminationTime(System.currentTimeMillis())
                    .withReceivedTermination(true);
        } else {
            // We don't set end time for force-quit, as the API interprets this to be a clean
            // termination
            builder = builder
                    .withEndTime(endTime);
        }

        return SessionMessage.newBuilder()
                .withUserInfo(userService.getUserInfo())
                .withAppInfo(metadataService.getAppInfo())
                .withDeviceInfo(metadataService.getDeviceInfo())
                .withPerformanceInfo(performanceInfoService.getSessionPerformanceInfo(startTime, endTime))
                .withBreadcrumbs(breadcrumbService.getBreadcrumbs(startTime, endTime))
                .withSession(builder.build())
                .build();
    }

    private String getApplicationState() {
        return activityService.isInBackground() ? APPLICATION_STATE_BACKGROUND : APPLICATION_STATE_ACTIVE;
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        synchronized (lock) {
            if (coldStart) {
                sendCachedSession();
            }
            startSession(coldStart, SessionLifeEventType.STATE);
        }
    }

    @Override
    public void onBackground() {
        endSession(SessionLifeEventType.STATE);
    }

    @Override
    public void triggerStatelessSessionEnd(SessionLifeEventType endType) {

        if (!configService.isSessionControlEnabled()) {
            EmbraceLogger.logWarning("Session control disabled from remote configuration.");
            return;
        }

        if (activeSession == null) {
            EmbraceLogger.logError("There's no active session to end.");
            return;
        }

        // If less than 5 seconds, then the session cannot be finished manually.
        if (System.currentTimeMillis() - activeSession.getStartTime() < 5000) {
            EmbraceLogger.logError("The session has to be of at least 5 seconds to be ended manually.");
            return;
        }

        // Ends active session.
        endSession(endType);

        // Starts a new session.
        if (!activityService.isInBackground()) {
            startSession(false, endType);
        }

        EmbraceLogger.logInfo("Session successfully closed.");
    }

    /**
     * This will trigger all necessary events to end the current session and send it to the server.
     *
     * @param endType the origin of the event that ends the session.
     */
    private synchronized void endSession(SessionLifeEventType endType) {

        stopCaching();
        stopSessionCloser();

        if (activeSession != null) {
            if (configService.isMessageTypeDisabled(MessageType.SESSION)) {
                EmbraceLogger.logWarning("Session messages disabled. Ignoring all Sessions.");
                return;
            }
            // Cache the active session to ensure that the cache agrees with the data we are
            // about to send. We do this since:
            //
            // 1. If the session data is sent, but the completion of the send is not registered
            //    by the SDK, then we will re-send an incomplete set of session data later.
            // 2. If the app crashes before we send the session message, the data since the last
            //    cache interval is lost.
            cacheActiveSession();
            SessionMessage sessionMessage = sessionEndMessage(activeSession, true, false, endType);
            activeSession = null;
            metadataService.setActiveSessionId(null);
            exceptionService.resetExceptionErrorObject();
            try {
                // During power saving mode, requests are hanged forever causing an ANR when
                // Foregrounded, so the Timeout prevents this scenario from happening.
                apiClient.sendSession(sessionMessage).get(SEND_SESSION_API_CLIENT_TIMEOUT, TimeUnit.SECONDS);
                cacheService.deleteObject(SESSION_FILE_NAME);
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to send session end message", ex);
            }
        }
    }

    /**
     * As when resuming the app a new session is created, the screen would be the same as the
     * one the app had when was backrounded. In this case, the SDK skips the "duplicate view
     * breadcrumb" scenario. This method, forces the repetition of the view breadcrumb when the
     * app's being resumed.
     */
    private void addViewBreadcrumbForResumedSession() {
        Optional<String> screen = breadcrumbService.getLastViewBreadcrumbScreenName();
        if (screen.isPresent()) {
            breadcrumbService.forceLogView(screen.orNull(), System.currentTimeMillis());
        }
    }

    /**
     * Send the cached session on a background thread, so that the main thread does not get
     * blocked during foreground.
     */
    private void sendCachedSession() {
        try {
            sessionBackgroundWorker.submit(() -> {
                try {
                    Optional<SessionMessage> optionalSession = cacheService.loadObject(SESSION_FILE_NAME, SessionMessage.class);
                    if (optionalSession.isPresent()) {
                        apiClient.sendSession(optionalSession.get());
                        cacheService.deleteObject(SESSION_FILE_NAME);
                    }
                } catch (Exception ex) {
                    EmbraceLogger.logDebug("Failed to send cached session message on resume", ex);
                }
                return null;
            });
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to create background worker to send cached session during resume", ex);
        }
    }

    @Override
    public void close() {
        EmbraceLogger.logInfo("Shutting down EmbraceSessionService");
        stopCaching();
        stopSessionCloser();
        sessionBackgroundWorker.close();
    }
}