package io.embrace.android.embracesdk;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;

/**
 * Checks whether the main thread is still responding by using the following strategy:
 * <ol>
 *     <li>Creating two {@link Handler}s, one on the main thread, and one on a new thread</li>
 *     <li>Using the 'monitoring' thread to message the main thread with a heartbeat</li>
 *     <li>Determining whether the main thread responds in time, and if not logging an ANR</li>
 * </ol>
 */
final class EmbraceAnrService implements AnrService {

    private static final int MONITORING_INTERVAL = 200;

    private static final long ANR_THRESHOLD_INTERVAL = 1000L;

    static final int HEALTHCHECK_EXECUTE = 34592;

    static final int HEALTHCHECK_REQUEST = 34593;

    static final int HEALTHCHECK_RESPONSE = 34594;

    private final NavigableMap<Long, AnrInterval> anrIntervals;

    private final MainThreadHandler mainThreadHandler;

    private final MonitoringThreadHandler monitoringThreadHandler;

    private final HandlerThread monitoringThread;

    private volatile long lastAlive;

    private volatile boolean anrInProgress;

    EmbraceAnrService() {
        this.anrIntervals = new ConcurrentSkipListMap<>();
        this.monitoringThread = new HandlerThread("Embrace ANR Healthcheck");
        this.monitoringThread.start();
        Looper callingThreadLooper = this.monitoringThread.getLooper();
        this.monitoringThreadHandler = new MonitoringThreadHandler(callingThreadLooper);
        this.mainThreadHandler = new MainThreadHandler(Looper.getMainLooper());
        this.monitoringThreadHandler.runHealthcheck();
    }

    @Override
    public List<AnrInterval> getAnrIntervals(long startTime, long endTime) {
        synchronized (this) {
            Collection<AnrInterval> intervals = anrIntervals.subMap(startTime, endTime).values();
            ArrayList<AnrInterval> results = new ArrayList<>();
            results.addAll(intervals);
            if (anrInProgress) {
                results.add(new AnrInterval(lastAlive, System.currentTimeMillis(), null, AnrInterval.Type.UI));
            }
            return results;
        }
    }

    @Override
    public void close() {
        try {
            monitoringThread.quit();
            monitoringThread.join();
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Failed to cleanly shut down EmbraceAnrService");
        }
    }

    /**
     * Performs the monitoring by submitting heartbeats to the {@link MainThreadHandler}, then
     * re-scheduling itself for another heartbeat by sending a message to itself with a delay.
     */
    class MonitoringThreadHandler extends Handler {

        public MonitoringThreadHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            try {
                long timestamp = System.currentTimeMillis();
                boolean anrThresholdExceeded = lastAlive > 0 && (timestamp - lastAlive) > ANR_THRESHOLD_INTERVAL;
                if (msg.what == EmbraceAnrService.HEALTHCHECK_RESPONSE) {
                    if (anrThresholdExceeded) {
                        // Application was not responding, but recovered
                        EmbraceLogger.logDebug("Main thread recovered from not responding for > 1s");
                        synchronized (EmbraceAnrService.this) {
                            anrIntervals.put(timestamp, new AnrInterval(
                                    lastAlive,
                                    timestamp,
                                    timestamp,
                                    AnrInterval.Type.UI));
                        }
                    }
                    lastAlive = timestamp;
                    anrInProgress = false;
                } else if (msg.what == EmbraceAnrService.HEALTHCHECK_EXECUTE) {
                    // Application is not responding
                    if (anrThresholdExceeded && !anrInProgress) {
                        EmbraceLogger.logDebug("Main thread not responding for > 1s");
                        anrInProgress = true;
                    }
                    runHealthcheck();
                }
            } catch (Exception ex) {
                EmbraceLogger.logDebug("ANR healthcheck failed in monitoring thread", ex);
            }
        }

        void runHealthcheck() {
            mainThreadHandler.sendMessage(obtainMessage(EmbraceAnrService.HEALTHCHECK_REQUEST));
            monitoringThreadHandler.sendMessageDelayed(
                    obtainMessage(EmbraceAnrService.HEALTHCHECK_EXECUTE),
                    MONITORING_INTERVAL);
        }
    }

    /**
     * Handles healthcheck messages sent to the main thread. Responds with an acknowledgement to the
     * calling thread.
     */
    class MainThreadHandler extends Handler {

        public MainThreadHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            try {
                if (msg.what == EmbraceAnrService.HEALTHCHECK_REQUEST) {
                    monitoringThreadHandler.sendMessage(obtainMessage(EmbraceAnrService.HEALTHCHECK_RESPONSE));
                }
            } catch (Exception ex) {
                EmbraceLogger.logDebug("ANR healthcheck failed in main (monitored) thread", ex);
            }
            super.handleMessage(msg);
        }
    }
}
