package io.embrace.android.embracesdk;

import android.app.Activity;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.view.View;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.io.ByteArrayOutputStream;
import java.util.concurrent.Future;

/**
 * Takes screenshots and submits them to the Embrace API.
 */
final class EmbraceScreenshotService implements ScreenshotService {
    private static final long SCREENSHOT_COOLDOWN = 3000;

    private static final int SCREENSHOT_JPEG_COMPRESSION_VALUE = 70;

    private final ActivityService activityService;

    private final ConfigService configService;

    private final ApiClient apiClient;

    private volatile long lastScreenshot;

    public EmbraceScreenshotService(ActivityService activityService, ConfigService configService, ApiClient apiClient) {
        this.activityService = Preconditions.checkNotNull(activityService);
        this.configService = Preconditions.checkNotNull(configService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
    }

    @Override
    public boolean takeScreenshotLogEvent(String logId) {
        return takeScreenshot(logId, null);
    }

    @Override
    public boolean takeScreenshotMoment(String eventId) {
        return takeScreenshot(null, eventId);
    }

    private boolean takeScreenshot(String logId, String eventId) {
        synchronized (this) {
            long timestamp = System.currentTimeMillis();
            if (activityService.isInBackground() ||
                    !configService.getConfig().getScreenshotsEnabled() ||
                    (timestamp - lastScreenshot) < SCREENSHOT_COOLDOWN) {

                EmbraceLogger.logDebug("Screenshots are disabled, app is backgrounded, or cooling down");
                return false;
            }
            Optional<Activity> optionalActivity = activityService.getForegroundActivity();
            if (optionalActivity.isPresent()) {
                byte[] screenshot = screenshot(optionalActivity.get());
                lastScreenshot = timestamp;
                Future<String> future;
                if (eventId != null) {
                    future = apiClient.sendMomentScreenshot(screenshot, eventId);
                } else if (logId != null) {
                    future = apiClient.sendLogScreenshot(screenshot, logId);
                } else {
                    return false;
                }
                return true;
            } else {
                EmbraceLogger.logDebug("Screenshot cannot be taken as there is no active activity");
                return false;
            }
        }
    }

    private byte[] screenshot(Activity activity) {
        Bitmap screenshot = takeScreenshot(activity);
        if (screenshot == null) {
            EmbraceLogger.logWarning("Screenshot returned was null");
            return null;
        }

        EmbraceLogger.logDebug("Compressing screenshot");
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        screenshot.compress(Bitmap.CompressFormat.JPEG, SCREENSHOT_JPEG_COMPRESSION_VALUE, stream);
        screenshot.recycle();
        return stream.toByteArray();
    }

    private Bitmap takeScreenshot(Activity activity) {
        if (activity != null) {
            View view = activity.getWindow().getDecorView();
            if (view != null && view.getWidth() > 0 && view.getHeight() > 0) {
                Bitmap screenshot = Bitmap.createBitmap(view.getWidth(),
                        view.getHeight(), Bitmap.Config.ARGB_8888);
                Canvas canvas = new Canvas(screenshot);
                view.draw(canvas);
                return screenshot;
            }
        }
        return null;
    }


}
