package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.optional.Optional;
import com.google.gson.annotations.SerializedName;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import java9.util.Maps;
import java9.util.Sets;

/**
 * Configuration of the SDK set by the Embrace API.
 */
class Config {
    /** The default percentage of devices for which the SDK is enabled. */
    private static final int DEFAULT_THRESHOLD = 100;
    /** The default percentage offset of devices for which the SDK is enabled. */
    private static final int DEFAULT_OFFSET = 0;

    /**
     * Used to determine whether or not the SDK should be activated for this device. The threshold
     * identifies the percentage of devices for which the SDK is enabled. A threshold of 100 means
     * that the SDK is enabled for all devices, whilst 0 means it is disabled for all devices.
     */
    @SerializedName("threshold")
    private final Integer threshold;

    /** Used to shift the offset of devices for which the SDK is enabled/disabled. */
    @SerializedName("offset")
    private final Integer offset;

    /** Whether screenshots should be taken for slow moments or error logs. */
    @SerializedName("screenshots_enabled")
    private final Boolean screenshotsEnabled;

    /** Whether gathering of signal strength by the {@link EmbraceSignalQualityService} is enabled. */
    @SerializedName("signal_strength_enabled")
    private final Boolean signalStrengthEnabled;

    /** The time in milliseconds after which a particular event ID is considered 'late'. */
    @SerializedName("event_limits")
    private final Map<String, Long> eventLimits;

    /** The list of {@link MessageType} which are disabled. */
    @SerializedName("disabled_message_types")
    private final Set<String> disabledMessageTypes;

    /** List of regular expressions matching event names and log messages which should be disabled. */
    @SerializedName("disabled_event_and_log_patterns")
    private final Set<String> disabledEventAndLogPatterns;

    /** List of regular expressions of URLs which should not be logged. */
    @SerializedName("disabled_url_patterns")
    private final Set<String> disabledUrlPatterns;

    /** List of regular expressions matching event names for which screenshots should be disabled. */
    @SerializedName("disabled_screenshot_patterns")
    private final Set<String> disabledScreenshotPatterns;

    /** Settings relating to the user interface, such as the breadcrumb limits. */
    @SerializedName("ui")
    private final UiConfig uiConfig;

    /** Settings defining the capture limits for network calls. */
    @SerializedName("network")
    private final NetworkConfig networkConfig;

    Config(
            Integer threshold,
            Integer offset,
            Boolean screenshotsEnabled,
            Boolean signalStrengthEnabled,
            Map<String, Long> eventLimits,
            Set<String> disabledMessageTypes,
            Set<String> disabledEventAndLogPatterns,
            Set<String> disabledUrlPatterns,
            Set<String> disabledScreenshotPatterns,
            UiConfig uiConfig,
            NetworkConfig networkConfig) {

        this.threshold = threshold;
        this.offset = offset;
        this.screenshotsEnabled = screenshotsEnabled;
        this.signalStrengthEnabled = signalStrengthEnabled;
        this.eventLimits = eventLimits;
        this.disabledMessageTypes = disabledMessageTypes;
        this.disabledEventAndLogPatterns = disabledEventAndLogPatterns;
        this.disabledUrlPatterns = disabledUrlPatterns;
        this.disabledScreenshotPatterns = disabledScreenshotPatterns;
        this.uiConfig = uiConfig;
        this.networkConfig = networkConfig;
    }

    /**
     * Creates SDK configuration based on default settings (threshold of 100, offset of 0, with
     * screenshots enabled).
     *
     * @return the default SDK configuration
     */
    static Config ofDefault() {
        return new Config(
                DEFAULT_THRESHOLD,
                DEFAULT_OFFSET,
                true,
                false,
                Maps.of(),
                Sets.of(),
                Sets.of(),
                Sets.of(),
                Sets.of(),
                new UiConfig(null, null, null, null),
                new NetworkConfig(null, null));
    }

    Integer getThreshold() {
        return threshold != null ? threshold : DEFAULT_THRESHOLD;
    }

    Integer getOffset() {
        return offset != null ? offset : DEFAULT_OFFSET;
    }

    Boolean getScreenshotsEnabled() {
        return screenshotsEnabled != null ? screenshotsEnabled : true;
    }

    Boolean getSignalStrengthEnabled() {
        return signalStrengthEnabled != null ? signalStrengthEnabled : false;
    }

    Map<String, Long> getEventLimits() {
        return eventLimits != null ? eventLimits : new HashMap<>();
    }

    Set<String> getDisabledMessageTypes() {
        return disabledMessageTypes != null ? disabledMessageTypes : new HashSet<>();
    }

    Set<String> getDisabledEventAndLogPatterns() {
        return disabledEventAndLogPatterns != null ? disabledEventAndLogPatterns : new HashSet<>();
    }

    Set<String> getDisabledUrlPatterns() {
        return disabledUrlPatterns != null ? disabledUrlPatterns : new HashSet<>();
    }

    Set<String> getDisabledScreenshotPatterns() {
        return disabledScreenshotPatterns != null ? disabledScreenshotPatterns : new HashSet<>();
    }

    Optional<Integer> getCustomBreadcrumbLimit() {
        return uiConfig == null ? Optional.absent() : uiConfig.getBreadcrumbs();
    }

    Optional<Integer> getViewBreadcrumbLimit() {
        return uiConfig == null ? Optional.absent() : uiConfig.getViews();
    }

    Optional<Integer> getTapBreadcrumbLimit() {
        return uiConfig == null ? Optional.absent() : uiConfig.getTaps();
    }

    Optional<Integer> getWebViewBreadcrumbLimit() {
        return uiConfig == null ? Optional.absent() : uiConfig.getWebViews();
    }

    Optional<Integer> getDefaultNetworkCallLimit() {
        return networkConfig == null ? Optional.absent() : networkConfig.getDefaultCaptureLimit();
    }

    Map<String, Integer> getNetworkCallLimitsPerDomain() {
        return networkConfig == null ? new HashMap<>() : networkConfig.getDomainLimits();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Config config = (Config) o;

        if (threshold != null ? !threshold.equals(config.threshold) : config.threshold != null)
            return false;
        if (offset != null ? !offset.equals(config.offset) : config.offset != null) return false;
        if (screenshotsEnabled != null ? !screenshotsEnabled.equals(config.screenshotsEnabled) : config.screenshotsEnabled != null)
            return false;
        if (signalStrengthEnabled != null ? !signalStrengthEnabled.equals(config.signalStrengthEnabled) : config.signalStrengthEnabled != null)
            return false;
        if (eventLimits != null ? !eventLimits.equals(config.eventLimits) : config.eventLimits != null)
            return false;
        if (disabledMessageTypes != null ? !disabledMessageTypes.equals(config.disabledMessageTypes) : config.disabledMessageTypes != null)
            return false;
        if (disabledEventAndLogPatterns != null ? !disabledEventAndLogPatterns.equals(config.disabledEventAndLogPatterns) : config.disabledEventAndLogPatterns != null)
            return false;
        if (disabledUrlPatterns != null ? !disabledUrlPatterns.equals(config.disabledUrlPatterns) : config.disabledUrlPatterns != null)
            return false;
        if (disabledScreenshotPatterns != null ? !disabledScreenshotPatterns.equals(config.disabledScreenshotPatterns) : config.disabledScreenshotPatterns != null)
            return false;
        if (uiConfig != null ? !uiConfig.equals(config.uiConfig) : config.uiConfig != null)
            return false;
        return networkConfig != null ? networkConfig.equals(config.networkConfig) : config.networkConfig == null;
    }

    @Override
    public int hashCode() {
        int result = threshold != null ? threshold.hashCode() : 0;
        result = 31 * result + (offset != null ? offset.hashCode() : 0);
        result = 31 * result + (screenshotsEnabled != null ? screenshotsEnabled.hashCode() : 0);
        result = 31 * result + (signalStrengthEnabled != null ? signalStrengthEnabled.hashCode() : 0);
        result = 31 * result + (eventLimits != null ? eventLimits.hashCode() : 0);
        result = 31 * result + (disabledMessageTypes != null ? disabledMessageTypes.hashCode() : 0);
        result = 31 * result + (disabledEventAndLogPatterns != null ? disabledEventAndLogPatterns.hashCode() : 0);
        result = 31 * result + (disabledUrlPatterns != null ? disabledUrlPatterns.hashCode() : 0);
        result = 31 * result + (disabledScreenshotPatterns != null ? disabledScreenshotPatterns.hashCode() : 0);
        result = 31 * result + (uiConfig != null ? uiConfig.hashCode() : 0);
        result = 31 * result + (networkConfig != null ? networkConfig.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return "Config{" +
                "threshold=" + threshold +
                ", offset=" + offset +
                ", screenshotsEnabled=" + screenshotsEnabled +
                ", signalStrengthEnabled=" + signalStrengthEnabled +
                ", eventLimits=" + eventLimits +
                ", disabledMessageTypes=" + disabledMessageTypes +
                ", disabledEventAndLogPatterns=" + disabledEventAndLogPatterns +
                ", disabledUrlPatterns=" + disabledUrlPatterns +
                ", disabledScreenshotPatterns=" + disabledScreenshotPatterns +
                ", uiConfig=" + uiConfig +
                ", networkConfig=" + networkConfig +
                '}';
    }

    /**
     * Configuration values relating to the user interface of the app.
     */
    static class UiConfig {
        /** The maximum number of custom breadcrumbs to send per session. */
        private final Integer breadcrumbs;

        private final Integer taps;

        private final Integer views;

        @SerializedName("web_views")
        private final Integer webViews;

        UiConfig(Integer breadcrumbs, Integer taps, Integer views, Integer webViews) {
            this.breadcrumbs = breadcrumbs;
            this.taps = taps;
            this.views = views;
            this.webViews = webViews;
        }

        public Optional<Integer> getBreadcrumbs() {
            return Optional.fromNullable(breadcrumbs);
        }

        public Optional<Integer> getTaps() {
            return Optional.fromNullable(taps);
        }

        public Optional<Integer> getViews() {
            return Optional.fromNullable(views);
        }

        public Optional<Integer> getWebViews() {
            return Optional.fromNullable(webViews);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            UiConfig uiConfig = (UiConfig) o;

            if (breadcrumbs != null ? !breadcrumbs.equals(uiConfig.breadcrumbs) : uiConfig.breadcrumbs != null)
                return false;
            if (taps != null ? !taps.equals(uiConfig.taps) : uiConfig.taps != null) return false;
            if (views != null ? !views.equals(uiConfig.views) : uiConfig.views != null)
                return false;
            return webViews != null ? webViews.equals(uiConfig.webViews) : uiConfig.webViews == null;
        }

        @Override
        public int hashCode() {
            int result = breadcrumbs != null ? breadcrumbs.hashCode() : 0;
            result = 31 * result + (taps != null ? taps.hashCode() : 0);
            result = 31 * result + (views != null ? views.hashCode() : 0);
            result = 31 * result + (webViews != null ? webViews.hashCode() : 0);
            return result;
        }

        @Override
        public String toString() {
            return "UiConfig{" +
                    "breadcrumbs=" + breadcrumbs +
                    ", taps=" + taps +
                    ", views=" + views +
                    ", webViews=" + webViews +
                    '}';
        }
    }

    /**
     * Configures limit of number of requests for network calls per domain.
     * <p>
     * If the default capture limit is specified as zero, then the config operates in whitelist
     * mode, meaning only specified domains will be tracked.
     */
    static class NetworkConfig {
        /** The default request capture limit for non-specified domains. */
        private final Integer defaultCaptureLimit;

        /** Map of domain suffix to maximum number of requests. */
        @SerializedName("domains")
        private final Map<String, Integer> domainLimits;

        NetworkConfig(Integer defaultCaptureLimit, Map<String, Integer> domainLimits) {
            this.defaultCaptureLimit = defaultCaptureLimit;
            this.domainLimits = domainLimits;
        }

        Map<String, Integer> getDomainLimits() {
            return domainLimits == null ? new HashMap<>() : domainLimits;
        }

        Optional<Integer> getDefaultCaptureLimit() {
            return Optional.fromNullable(defaultCaptureLimit);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            NetworkConfig that = (NetworkConfig) o;

            if (defaultCaptureLimit != null ? !defaultCaptureLimit.equals(that.defaultCaptureLimit) : that.defaultCaptureLimit != null)
                return false;
            return domainLimits != null ? domainLimits.equals(that.domainLimits) : that.domainLimits == null;
        }

        @Override
        public int hashCode() {
            int result = defaultCaptureLimit != null ? defaultCaptureLimit.hashCode() : 0;
            result = 31 * result + (domainLimits != null ? domainLimits.hashCode() : 0);
            return result;
        }

        @Override
        public String toString() {
            return "NetworkConfig{" +
                    "defaultCaptureLimit=" + defaultCaptureLimit +
                    ", domainLimits=" + domainLimits +
                    '}';
        }
    }
}
