package io.embrace.android.embracesdk;

import android.content.Context;
import android.content.res.AssetManager;

import com.fernandocejas.arrow.optional.Optional;
import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;
import com.google.gson.stream.JsonReader;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Collections;
import java.util.List;

/**
 * Specifies the application ID and build ID.
 */
final class BuildInfo {

    /**
     * Build Info json file name to be retrieved from the assets.
     */
    private static final String FILE_BUILD_INFO = "build-info.json";

    /**
     * The Embrace app ID. This is used to identify the app within the database.
     */
    @SerializedName("app-id")
    private final String appId;

    /**
     * The ID of the particular build, generated at compile-time.
     */
    @SerializedName("build-id")
    private final String buildId;

    /**
     * The BuildType name of the particular build, extracted at compile-time.
     */
    @SerializedName("build-type")
    private final String buildType;

    /**
     * The Flavor name of the particular build, extracted at compile-time.
     */
    @SerializedName("build-flavor")
    private final String buildFlavor;

    /**
     * The Network configuration provided on the gradle's extension.
     */
    @SerializedName("network-config")
    private final Network networkConfig;

    BuildInfo(String appId, String buildId, String buildType, String buildFlavor, Network networkConfig) {
        this.appId = appId;
        this.buildId = buildId;
        this.networkConfig = networkConfig;
        this.buildType = buildType;
        this.buildFlavor = buildFlavor;
    }

    /**
     * Loads the build information from a JSON file packaged within the application by Gradle at
     * build-time.
     *
     * @return the build information
     */
    static BuildInfo fromFile(Context context) {

        InputStream buildInfoStream = null;
        InputStreamReader inputStreamReader = null;
        JsonReader reader = null;
        try {
            AssetManager assets = context.getApplicationContext().getAssets();
            buildInfoStream = assets.open(FILE_BUILD_INFO);
            inputStreamReader = new InputStreamReader(buildInfoStream);
            reader = new JsonReader(inputStreamReader);
            return new Gson().fromJson(reader, BuildInfo.class);
        } catch (Exception ex) {
            throw new RuntimeException("Failed to read build-info.json", ex);
        } finally {
            try {
                if (buildInfoStream != null) {
                    buildInfoStream.close();
                }
                if (reader != null) {
                    reader.close();
                }
                if (inputStreamReader != null) {
                    inputStreamReader.close();
                }
            } catch (IOException ex) {
                EmbraceLogger.logWarning("Failed to close build-info.json", ex);
            }
        }
    }

    String getAppId() {
        return appId;
    }

    String getBuildId() {
        return buildId;
    }

    String getBuildType() {
        return buildType;
    }

    String getBuildFlavor() {
        return buildFlavor;
    }

    public Optional<Network> getNetworkConfig() {
        return Optional.fromNullable(networkConfig);
    }

    /**
     * Represents the network configuration element specified in the embrace extension.
     */
    class Network {

        /**
         * The default capture limit for the specified domains.
         */
        @SerializedName("default-capture-limit")
        private final Integer defaultCaptureLimit;

        /**
         * List of domains to be limited for tracking.
         */
        @SerializedName("domains")
        private final List<Domain> domains;

        Network(Integer defaultCaptureLimit, List<Domain> domains) {
            this.defaultCaptureLimit = defaultCaptureLimit;
            this.domains = domains;
        }

        public Optional<Integer> getDefaultCaptureLimit() {
            return Optional.fromNullable(defaultCaptureLimit);
        }

        public List<Domain> getDomains() {
            return domains != null ? domains : Collections.emptyList();
        }

        /**
         * Represents each domain element specified in the embrace extension.
         */
        class Domain {

            /**
             * Url for the domain.
             */
            @SerializedName("domain-name")
            private final String domain;

            /**
             * Limit for the number of requests to be tracked.
             */
            @SerializedName("domain-limit")
            private final Integer limit;

            Domain(String domain, Integer limit) {
                this.domain = domain;
                this.limit = limit;
            }

            public String getDomain() {
                return domain;
            }

            public Integer getLimit() {
                return limit;
            }
        }
    }
}
