package io.embrace.android.embracesdk;

import android.text.TextUtils;

import java.util.Map;

/**
 * This class is responsible for tracking app performance during subscription purchase flows.
 * <p>
 * This class is thread-safe.
 */
public final class SubscriptionPurchaseFlow extends CustomFlow {

    static final String MOMENT_SUBSCRIPTION_PURCHASE = "_subscription-purchase";

    static final String PROP_AMOUNT = "amount";
    static final String PROP_ORDER_ID = "order-id";
    static final String PROP_PAYMENT_TYPE = "payment-type";
    static final String PROP_SUBSCRIPTION_TYPE = "subscription-type";

    private volatile String purchaseMomentId;
    private Map<String, Object> purchaseProps;

    /**
     * Starts a subscription purchase moment.
     * <p>
     * This method should be called as soon as the user indicates an intent to purchase a subscription. This means that
     * all information pertaining to the purchase (e.g. billing, payment, shipping) should already be known prior to
     * invoking this method.
     *
     * @param orderId          The ID that represents the subscription purchase order. This value is optional and, if present, will
     *                         associate the value as a property of the moment.
     * @param subscriptionType The recurrence factor (e.g. monthly, annual) of the subscription purchase. This value is
     *                         optional and, if present, will associate the value as a property of the moment.
     * @param amount           The total amount of the subscription purchase. This value is optional and, if present, will associate
     *                         the value as a property of the moment.
     * @param paymentType      The payment system that will be fulfilling the subscription purchase (e.g. Google IAB, PayPal,
     *                         BrainTree). This value is optional and, if present, will associate the value as a property of
     *                         the moment.
     * @param properties       A map of Strings to Objects that represent additional properties to associate with the moment.
     *                         This value is optional. A maximum of 10 properties (not including the ones set via arguments to
     *                         this method) may be set.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean subscriptionPurchaseStart(String orderId,
                                             String subscriptionType,
                                             Number amount,
                                             String paymentType,
                                             Map<String, Object> properties) {

        Map<String, Object> normalizedProperties = PropertyUtils.sanitizeProperties(properties);

        if (orderId != null) {
            normalizedProperties.put(PROP_ORDER_ID, orderId);
        }
        if (subscriptionType != null) {
            normalizedProperties.put(PROP_SUBSCRIPTION_TYPE, subscriptionType);
        }
        if (amount != null) {
            normalizedProperties.put(PROP_AMOUNT, amount);
        }
        if (paymentType != null) {
            normalizedProperties.put(PROP_PAYMENT_TYPE, paymentType);
        }

        this.purchaseProps = normalizedProperties;
        this.purchaseMomentId = Uuid.getEmbUuid();

        sendMomentStartEvent(MOMENT_SUBSCRIPTION_PURCHASE, this.purchaseMomentId, false, normalizedProperties);
        return true;
    }

    /**
     * Ends the subscription purchase moment and generates an info log message that indicates that the subscription
     * purchase completed.
     * <p>
     * This method should be called once the subscription purchase has been confirmed.
     *
     * @return True if the operation was successful; false otherwise.
     */
    public boolean subscriptionPurchaseComplete() {
        if (this.purchaseMomentId == null) {
            EmbraceLogger.logError("Subscription purchase wasn't started.");
            return false;
        }

        sendMomentEndEvent(MOMENT_SUBSCRIPTION_PURCHASE, this.purchaseMomentId);
        sendLogInfo("Subscription purchase was completed.", this.purchaseProps);

        // Since the user has completed a purchase, automatically mark them as being a payer.
        // TODO: Review which service is originally responsible for tracking this
        Embrace.getInstance().setUserAsPayer();

        this.purchaseMomentId = null;
        return true;
    }

    /**
     * Ends the subscription purchase moment and generates an error log message that indicates that the subscription
     * purchase failed.
     * <p>
     * This method should be called once the subscription purchase has been confirmed.
     *
     * @param msg A message that explains the reason for why this operation failed. This value is optional and, if
     *            provided, will associate the value as a property of the error log message.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean subscriptionPurchaseFail(String msg) {
        if (this.purchaseMomentId == null) {
            EmbraceLogger.logError("Subscription purchase wasn't started.");
            return false;
        } else if (msg != null) {
            this.purchaseProps.put(PROP_MESSAGE, msg);
        }

        String errorLogMsg;

        if (TextUtils.isEmpty(msg)) {
            errorLogMsg = "A failure occurred during subscription purchase.";
        } else {
            errorLogMsg = "A failure occurred during subscription purchase: " + msg;
        }

        sendMomentEndEvent(MOMENT_SUBSCRIPTION_PURCHASE, this.purchaseMomentId);
        sendLogError(errorLogMsg, true, this.purchaseProps);

        this.purchaseMomentId = null;
        return true;
    }
}
