package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

import java.util.Map;

/**
 * Immutable object which represents an event within the app.
 */
final class Event {
    @SerializedName("n")
    private final String name;

    /** Unique identifier used to refer to a remote log item. */
    @SerializedName("li")
    private final String messageId;

    /** Unique identifier used to refer to all other types of events. */
    @SerializedName("id")
    private final String storyId;

    @SerializedName("si")
    private final String sessionId;

    @SerializedName("t")
    private final EmbraceEvent.Type type;

    @SerializedName("ts")
    private final Long timestamp;

    @SerializedName("th")
    private final Long lateThreshold;

    @SerializedName("sc")
    private final boolean screenshotTaken;

    @SerializedName("du")
    private final Long duration;

    @SerializedName("st")
    private final String appState;

    @SerializedName("pr")
    private final Map<String, Object> customProperties;

    private Event(Builder builder) {
        name = builder.name;
        storyId = builder.storyId;
        messageId = builder.messageId;
        sessionId = builder.sessionId;
        type = builder.type;
        timestamp = builder.timestamp;
        lateThreshold = builder.lateThreshold;
        screenshotTaken = builder.screenshotTaken;
        duration = builder.duration;
        appState = builder.appState;
        customProperties = builder.customProperties;
    }

    public String getName() {
        return name;
    }

    public String getStoryId() {
        return storyId;
    }

    public String getMessageId() {
        return messageId;
    }

    public String getSessionId() {
        return sessionId;
    }

    public EmbraceEvent.Type getType() {
        return type;
    }

    public Long getTimestamp() {
        return timestamp;
    }

    public Long getLateThreshold() {
        return lateThreshold;
    }

    public boolean isScreenshotTaken() {
        return screenshotTaken;
    }

    public Long getDuration() {
        return duration;
    }

    public String getAppState() {
        return appState;
    }

    public Map<String, Object> getCustomProperties() {
        return customProperties;
    }

    public static Builder newBuilder() {
        return new Builder();
    }


    public static final class Builder {
        private String name;
        private String storyId = Uuid.getEmbUuid();
        private String messageId;
        private String sessionId;
        private EmbraceEvent.Type type;
        private Long timestamp = System.currentTimeMillis();
        private Long lateThreshold;
        private boolean screenshotTaken;
        private Long duration;
        private String appState;
        private Map<String, Object> customProperties;

        private Builder() {
        }

        public Builder withName(String name) {
            this.name = name;
            return this;
        }

        public Builder withStoryId(String storyId) {
            this.storyId = storyId;
            return this;
        }

        public Builder withMessageId(String messageId) {
            this.messageId = messageId;
            return this;
        }

        public Builder withSessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        public Builder withType(EmbraceEvent.Type type) {
            this.type = type;
            return this;
        }

        public Builder withTimestamp(Long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public Builder withLateThreshold(Long lateThreshold) {
            this.lateThreshold = lateThreshold;
            return this;
        }

        public Builder withScreenshotTaken(boolean screenshotTaken) {
            this.screenshotTaken = screenshotTaken;
            return this;
        }

        public Builder withDuration(Long duration) {
            this.duration = duration;
            return this;
        }

        public Builder withAppState(String appState) {
            this.appState = appState;
            return this;
        }

        public Builder withCustomProperties(Map<String, Object> customProperties) {
            this.customProperties = customProperties;
            return this;
        }

        public Event build() {
            return new Event(this);
        }
    }
}
