package io.embrace.android.embracesdk;

import android.content.Context;
import android.telephony.PhoneStateListener;
import android.telephony.TelephonyManager;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.ArrayList;
import java.util.List;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;

/**
 * Monitors changes in signal strength of the device, if enabled in {@link Config}.
 */
final class EmbraceSignalQualityService extends PhoneStateListener implements SignalQualityService {
    /** Minimum time between samples in ms. */
    private static final long MIN_TIME_BETWEEN_SAMPLES = 10000L;

    private final NavigableMap<Long, SignalStrength> signalStrength;

    private final TelephonyManager telephonyManager;

    private final ConfigService configService;

    /**
     * True if listening for signal strength changes is enabled, false otherwise.
     */
    private volatile boolean signalStrengthListening = false;

    EmbraceSignalQualityService(Context context, ConfigService configService, ActivityService activityService) {
        signalStrength = new ConcurrentSkipListMap<>();
        Preconditions.checkNotNull(context, "context must not be null");
        this.telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
        this.configService = Preconditions.checkNotNull(configService, "configService must not be null");
        configService.addListener(this);
        activityService.addListener(this);
    }



    @Override
    public List<SignalStrength> getSignalMeasurements(long startTime, long endTime) {
        return new ArrayList<>(this.signalStrength.subMap(startTime, endTime).values());
    }


    @Override
    public void onSignalStrengthsChanged(android.telephony.SignalStrength signalStrength) {
        long now = System.currentTimeMillis();
        // Don't allow more than one sample every MIN_TIME_BETWEEN_SAMPLES milliseconds
        if (this.signalStrength.isEmpty() || (now - this.signalStrength.lastKey() >= MIN_TIME_BETWEEN_SAMPLES)) {
            SignalStrength result = SignalStrength.of(signalStrength);
            this.signalStrength.put(now, result);
        }
    }

    @Override
    public void onConfigChange(Config previousConfig, Config newConfig) {
        if (!previousConfig.getSignalStrengthEnabled().equals(newConfig.getSignalStrengthEnabled())) {
            if (newConfig.getSignalStrengthEnabled()) {
                registerPhoneStateListener();
                EmbraceLogger.logInfo("Starting signal strength listener as config was changed");
            } else {
                deregisterPhoneStateListener();
                EmbraceLogger.logInfo("Stopping signal strength listener as config was changed");

            }
        }
    }

    @Override
    public void close() {
        EmbraceLogger.logInfo("Stopping EmbraceSignalQualityService");
        deregisterPhoneStateListener();
    }

    @Override
    public void onBackground() {
        EmbraceLogger.logDebug("Pausing signal strength collection as app going to background");
        deregisterPhoneStateListener();
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        EmbraceLogger.logDebug("Resuming signal strength collection as app in foreground");
        registerPhoneStateListener();
    }

    private void registerPhoneStateListener() {
        if (telephonyManager != null && configService.getConfig().getSignalStrengthEnabled() && !signalStrengthListening) {
            try {
                telephonyManager.listen(this, PhoneStateListener.LISTEN_SIGNAL_STRENGTHS);
                signalStrengthListening = true;
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to register signal strength listener", ex);
            }
        }
    }

    private void deregisterPhoneStateListener() {
        if (telephonyManager != null && signalStrengthListening) {
            try {
                telephonyManager.listen(this, PhoneStateListener.LISTEN_NONE);
                signalStrengthListening = false;
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to deregister signal strength listener", ex);
            }
        }
    }
}
