package io.embrace.android.embracesdk;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.text.TextUtils;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentSkipListMap;

import java9.util.Maps;

class EmbraceNetworkService extends BroadcastReceiver implements NetworkService {

    /**
     * Threshold at which a network call is considered slow, in milliseconds.
     */
    private static final int SLOW_REQUEST_THRESHOLD = 500;
    private final IntentFilter intentFilter = new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION);
    private final NavigableMap<Long, NetworkStatus> networkReachable;
    private final NavigableMap<Long, NetworkCall> networkCalls;
    private final ConnectivityManager connectivityManager;
    private final Context context;
    private final ConnectionClassService connectionClassService;

    public EmbraceNetworkService(Context context, ConnectionClassService signalQualityService) {
        this.context = Preconditions.checkNotNull(context, "context must not be null");
        this.connectionClassService = Preconditions.checkNotNull(
                signalQualityService,
                "connectionClassService must not be null");
        this.connectivityManager =
                (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        this.networkReachable = new TreeMap<>();
        this.networkCalls = new ConcurrentSkipListMap<>();
        try {
            context.registerReceiver(this, intentFilter);
        } catch (Exception ex) {
            EmbraceLogger.logError("Failed to register EmbraceNetworkService broadcast receiver. Connectivity status will be unavailable.", ex);
        }
    }

    @Override
    public void onReceive(Context context, Intent intent) {
        try {
            NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();
            if (networkInfo != null && networkInfo.isConnected()) {
                // Network is reachable
                if (networkInfo.getType() == ConnectivityManager.TYPE_WIFI) {
                    saveStatus(NetworkStatus.WIFI);
                } else if (networkInfo.getType() == ConnectivityManager.TYPE_MOBILE) {
                    saveStatus(NetworkStatus.WAN);
                }
            } else {
                // Network is not reachable
                saveStatus(NetworkStatus.NOT_REACHABLE);
            }
        } catch (Exception ex) {
            EmbraceLogger.logError("Failed to handle CONNECTIVITY_ACTION broadcast", ex);
        }
    }

    @Override
    public List<Interval> getNetworkInterfaceIntervals(long startTime, long endTime) {
        synchronized (this) {
            List<Interval> results = new ArrayList<>();
            for (Map.Entry<Long, NetworkStatus> entry : networkReachable.subMap(startTime, endTime).entrySet()) {
                NetworkStatus value = entry.getValue();
                if (!value.equals(NetworkStatus.NOT_REACHABLE)) {
                    long currentTime = entry.getKey();
                    Long next = networkReachable.higherKey(currentTime);
                    results.add(new Interval(currentTime, next != null ? next : currentTime, value.getName()));
                }
            }
            return results;
        }
    }

    @Override
    public List<NetworkCall> getNetworkCalls(long startTime, long endTime) {
        return new ArrayList<>(this.networkCalls.subMap(startTime, endTime).values());
    }

    @Override
    public NetworkSession getNetworkSession(long startTime, long lastKnownTime) {
        Map<String, NetworkTimeline> timelines = new HashMap<>();
        Map<String, List<NetworkTimeline.NetworkRequest>> results = new HashMap<>();
        List<NetworkCall> problematicCalls = new ArrayList<>();

        List<NetworkCall> calls = getNetworkCalls(startTime, lastKnownTime);

        for (NetworkCall call : calls) {
            String method = call.getHttpMethod();
            String url = stripUrl(call.getUrl());
            Integer statusCode = call.getStatusCode();

            if (call.isDidClientError() ||
                    call.getStatusCode() >= 300 ||
                    call.getEndTime() - call.getStartTime() >= SLOW_REQUEST_THRESHOLD) {
                problematicCalls.add(call);
            }

            if (!TextUtils.isEmpty(method) && !TextUtils.isEmpty(url) && statusCode != null) {
                String methodUrl = String.format("%s_%s", method, url);
                Maps.putIfAbsent(results, methodUrl, new ArrayList<>());
                results.get(methodUrl).add(new NetworkTimeline.NetworkRequest(
                        call.getStatusCode(),
                        call.getStartTime(),
                        call.getDuration()));
            }
        }

        for (Map.Entry<String, List<NetworkTimeline.NetworkRequest>> entry : results.entrySet()) {
            timelines.put(entry.getKey(), new NetworkTimeline(entry.getValue(), entry.getValue().size()));
        }
        return new NetworkSession(timelines, problematicCalls);
    }

    @Override
    public void logNetworkCall(
            String url,
            String httpMethod,
            Integer statusCode,
            long startTime,
            long endTime,
            Long bytesSent,
            Long bytesReceived) {

        long duration = Math.max(endTime - startTime, 0);
        NetworkCall networkCall = NetworkCall.newBuilder()
                .withUrl(url)
                .withHttpMethod(httpMethod)
                .withStatusCode(statusCode)
                .withStartTime(startTime)
                .withDuration(duration)
                .withEndTime(endTime)
                .withBytesSent(bytesSent)
                .withBytesReceived(bytesReceived)
                .build();
        if (networkCall.isValidNetworkCall()) {
            networkCalls.put(startTime, networkCall);
            connectionClassService.logBandwidth(bytesReceived, duration);
        }
    }

    @Override
    public void logNetworkError(
            String url,
            String httpMethod,
            long startTime,
            long endTime,
            String errorType,
            String errorMessage) {
        NetworkCall networkCall = NetworkCall.newBuilder()
                .withUrl(url)
                .withHttpMethod(httpMethod)
                .withStartTime(startTime)
                .withEndTime(endTime)
                .withDuration(Math.max(endTime - startTime, 0))
                .withErrorType(errorType)
                .withErrorMessage(errorMessage)
                .withDidClientError(true)
                .build();
        if (networkCall.isValidNetworkCall()) {
            networkCalls.put(startTime, networkCall);
        }
    }


    private void saveStatus(NetworkStatus networkStatus) {
        synchronized (this) {
            if (networkReachable.isEmpty() || !networkReachable.lastEntry().getValue().equals(networkStatus)) {
                networkReachable.put(System.currentTimeMillis(), networkStatus);
            }
        }
    }

    @Override
    public void close() {
        context.unregisterReceiver(this);
    }

    private enum NetworkStatus {
        NOT_REACHABLE("none"),
        WIFI("wifi"),
        WAN("wan");

        private String name;

        NetworkStatus(String name) {
            this.name = name;
        }

        String getName() {
            return this.name;
        }
    }

    /**
     * Strips off the query string and hash fragment from a URL.
     *
     * @param url the URL to parse
     * @return the URL with the hash fragment and query string parameters removed
     */
    private static String stripUrl(String url) {
        if (url == null) {
            return null;
        }

        int pathPos = url.lastIndexOf('/');
        String suffix = pathPos < 0 ? url : url.substring(pathPos);

        int queryPos = suffix.indexOf('?');
        int fragmentPos = suffix.indexOf('#');
        int terminalPos = Math.min(queryPos < 0 ? Integer.MAX_VALUE : queryPos,
                fragmentPos < 0 ? Integer.MAX_VALUE : fragmentPos);

        return url.substring(0, (pathPos < 0 ? 0 : pathPos) + Math.min(suffix.length(), terminalPos));
    }
}
