package io.embrace.android.embracesdk;

import android.util.Pair;

import com.fernandocejas.arrow.optional.Optional;

import java.util.List;

import io.embrace.android.embracesdk.TapBreadcrumb.TapBreadcrumbType;

/**
 * Service which stores breadcrumbs for the application.
 */
interface BreadcrumbService {
    /**
     * Gets the view breadcrumbs in the specified time window.
     * If the number of elements exceeds the limit, this will return the newest (latest) ones.
     *
     * @param start the start time
     * @param end the end time
     * @return the list of Breadcrumbs
     */
    List<ViewBreadcrumb> getViewBreadcrumbsForSession(long start, long end);

    /**
     * Gets the Taps breadcrumbs in the specified time window.
     * If the number of elements exceeds the limit, this will return the newest (latest) ones.
     *
     * @param start the start time
     * @param end the end time
     * @return the list of Breadcrumbs
     */
    List<TapBreadcrumb> getTapBreadcrumbsForSession(long start, long end);

    /**
     * Gets the Custom breadcrumbs in the specified time window.
     * If the number of elements exceeds the limit, this will return the newest (latest) ones.
     *
     * @param start the start time
     * @param end the end time
     * @return the list of Breadcrumbs
     */
    List<CustomBreadcrumb> getCustomBreadcrumbsForSession(long start, long end);

    /**
     * Gets all breadcrumbs within the specified time window. If the number of elements exceeds the
     * limit for each breadcrumb type, only the latest will be returned.
     *
     * @param start the start time
     * @param end the end time
     * @return the breadcrumbs
     */
    Breadcrumbs getBreadcrumbs(long start, long end);

    /**
     * Registers a view breadcrumb.
     * The view breadcrumb will not be registered if the last view breadcrumb registry has the same
     * screen name.
     *
     * @param screen    name of the screen.
     * @param timestamp time of occurrence of the tap event.
     */
    void logView(String screen, long timestamp);

    /**
     * Unlike {@link EmbraceBreadcrumbService#logView(String screen, long timestamp)}
     * this function logs the view despite the previous one in the queue has the same screen name.
     *
     * @param screen    name of the screen.
     * @param timestamp time of occurrence of the tap event.
     */
    void forceLogView(String screen, long timestamp);

    /**
     * Registers a tap event as a breadcrumb.
     *
     * @param point     coordinates of the tapped element.
     * @param element   tapped element view.
     * @param timestamp time of occurrence of the tap event.
     * @param type      type of tap event
     */
    void logTap(Pair<Float, Float> point, String element, long timestamp, TapBreadcrumbType type);

    /**
     * Registers a custom event as a breadcrumb.
     *
     * @param message message for the custom breadcrumb.
     */
    void logCustom(String message, long timestamp);

    /**
     * This function can be used to recover the last view breadcrumb registered.
     *
     * @return the last registered view breadcrumb in the queue.
     */
    Optional<String> getLastViewBreadcrumbScreenName();
}
