package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.optional.Optional;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;

/**
 * Retries failed API calls for a particular message type every 10 seconds.
 * <p>
 * This service is called by the {@link ApiClient} in the event that an API call fails.
 *
 * @param <T> the message type to retry
 */
class ApiClientRetryWorker <T> {
    private final Class<T> clazz;

    private final CacheService cacheService;

    private final Queue<FailedApiCall<T>> failedApiCalls = new ConcurrentLinkedQueue<>();

    private final String cacheKey;

    private final ApiClient apiClient;

    private final ScheduledWorker worker;

    ApiClientRetryWorker(Class<T> clazz, CacheService cacheService, ApiClient apiClient, ScheduledWorker worker) {
        this.clazz = clazz;
        this.cacheService = cacheService;
        this.apiClient = apiClient;
        this.cacheKey = "network_failed_" + clazz.getSimpleName().toLowerCase(Locale.ENGLISH);
        this.worker = worker;
        start();
    }

    /**
     * Reattempt the API call every 10 seconds.
     *
     * @param request the API request to retry
     * @param payload the payload to retry
     */
    void retryCall(ApiRequest request, T payload) {
        failedApiCalls.add(new FailedApiCall<>(request, payload));
    }

    private void start() {
        // Load any cached failed API calls
        Optional<FailedApiCalls> optionalCalls = cacheService.loadObject(cacheKey, FailedApiCalls.class);
        if (optionalCalls.isPresent()) {
            failedApiCalls.addAll(optionalCalls.get().failedApiCalls);
        }

        // Re-attempt any failed API calls every 10 seconds.
        worker.scheduleAtFixedRate(() -> {
            try {
                List<FailedApiCall<T>> reattempt = new ArrayList<>();
                FailedApiCall<T> lastCall = failedApiCalls.poll();
                while (lastCall != null) {
                    reattempt.add(lastCall);
                    lastCall = failedApiCalls.poll();
                }
                // Cache the failed API calls, in case the app is terminated
                cacheService.cacheObject(
                        cacheKey,
                        new FailedApiCalls<>(reattempt),
                        FailedApiCalls.class);
                for (FailedApiCall<T> apiCall : reattempt) {
                    apiClient.jsonPost(apiCall.apiRequest, apiCall.payload, clazz, this::retryCall);
                }
            } catch (Exception ex) {
                EmbraceLogger.logError("Error in ApiClient retry worker", ex);
            }
        }, 0, 10, TimeUnit.SECONDS);
    }

    private static class FailedApiCall <P> {
        private final ApiRequest apiRequest;

        private final P payload;

        FailedApiCall(ApiRequest apiRequest, P payload) {
            this.apiRequest = apiRequest;
            this.payload = payload;
        }
    }

    private static class FailedApiCalls<P> {
        private final List<FailedApiCall<P>> failedApiCalls;

        FailedApiCalls(List<FailedApiCall<P>> failedApiCalls) {
            this.failedApiCalls = failedApiCalls;
        }
    }

}
