package io.embrace.android.embracesdk;

import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import java9.util.Maps;
import java9.util.stream.Collectors;
import java9.util.stream.StreamSupport;

/**
 * Handles the lifecycle of events (moments).
 * <p>
 * An event is started, timed, and then ended. If the event takes longer than a specified period of
 * time, then the event is considered late, and a screenshot is taken.
 */
final class EmbraceEventService implements EventService, ActivityListener {
    public static final String STARTUP_EVENT_NAME = "_startup";

    /**
     * The time default period after which an event is considered 'late'.
     */
    private static final long DEFAULT_LATE_THRESHOLD_MILLIS = 5000L;

    private final ApiClient apiClient;
    private final ConfigService configService;
    private final MetadataService metadataService;
    private final PerformanceInfoService performanceInfoService;
    private final UserService userService;
    private final ScreenshotService screenshotService;

    /**
     * Timeseries of event IDs, keyed on the start time of the event.
     */
    private final NavigableMap<Long, String> eventIds = new ConcurrentSkipListMap<>();
    /**
     * Map of active events, keyed on their event ID (event name + identifier).
     */
    private final ConcurrentMap<String, EventDescription> activeEvents = new ConcurrentHashMap<>();
    /**
     * Posts a 'LATE' event when the timer expires.
     */
    private final ScheduledWorker lateEventWorker;

    public EmbraceEventService(
            ApiClient apiClient,
            ConfigService configService,
            MetadataService metadataService,
            PerformanceInfoService performanceInfoService,
            UserService userService,
            ScreenshotService screenshotService,
            EmbraceActivityService activityService) {

        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.configService = Preconditions.checkNotNull(configService);
        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.performanceInfoService = Preconditions.checkNotNull(performanceInfoService);
        this.userService = Preconditions.checkNotNull(userService);
        this.screenshotService = Preconditions.checkNotNull(screenshotService);
        Preconditions.checkNotNull(activityService).addListener(this);
        this.lateEventWorker = ScheduledWorker.ofSingleThread("Late Event Handler");
    }

    private static String getInternalEventKey(String eventName, String identifier) {
        if (TextUtils.isEmpty(identifier)) {
            return eventName;
        }
        return String.format("%s%s%s", eventName, "#", identifier);
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        if (coldStart) {
            startEvent(STARTUP_EVENT_NAME, null, true, null, startupTime);
        }
    }

    @Override
    public void applicationStartupComplete() {
        endEvent(STARTUP_EVENT_NAME);
    }

    @Override
    public void startEvent(String name) {
        startEvent(name, null, true);
    }

    @Override
    public void startEvent(String name, String identifier) {
        startEvent(name, identifier, true);
    }

    @Override
    public void startEvent(String name, String identifier, boolean allowScreenshot) {
        startEvent(name, identifier, allowScreenshot, null, null);
    }

    @Override
    public void startEvent(String name, String identifier, boolean allowScreenshot, Map<String, Object> properties) {
        startEvent(name, identifier, allowScreenshot, properties, null);
    }

    @Override
    public void startEvent(String name, String identifier, boolean allowScreenshot, Map<String, Object> properties, Long startTime) {
        try {
            Preconditions.checkArgument(!TextUtils.isEmpty(name), "Event name must be specified");

            if (configService.isEventDisabled(name)) {
                EmbraceLogger.logWarning(String.format("Event disabled. Ignoring event with name %s", name));
                return;
            }

            if (configService.isMessageTypeDisabled(MessageType.EVENT)) {
                EmbraceLogger.logWarning("Event message disabled. Ignoring all Events.");
                return;
            }

            if (lateEventWorker.isShutdown()) {
                EmbraceLogger.logError("Cannot start event as service is shut down");
                return;
            }

            String eventKey = getInternalEventKey(name, identifier);

            if (activeEvents.containsKey(eventKey)) {
                endEvent(name, identifier, false, null);
            }
            long now = System.currentTimeMillis();
            if (startTime == null) {
                startTime = now;
            }
            Long threshold = calculateLateThreshold(eventKey);

            String eventId = Uuid.getEmbUuid();
            eventIds.put(now, eventId);
            Event.Builder builder = Event.newBuilder()
                    .withEventId(eventId)
                    .withType(EmbraceEvent.Type.START)
                    .withAppState(metadataService.getAppState())
                    .withName(name)
                    .withLateThreshold(threshold)
                    .withTimestamp(startTime);

            if (metadataService.getActiveSessionId().isPresent()) {
                builder = builder.withSessionId(metadataService.getActiveSessionId().get());
            }
            if (properties != null) {
                builder = builder.withCustomProperties(properties);
            }

            Event event = builder.build();
            ScheduledFuture<?> timer = lateEventWorker.scheduleWithDelay(
                    () -> endEvent(name, identifier, true, null),
                    threshold - calculateOffset(startTime, threshold),
                    TimeUnit.MILLISECONDS);
            activeEvents.put(eventId, new EventDescription(timer, event, allowScreenshot));

            EventMessage eventMessage = EventMessage.newBuilder()
                    .withUserInfo(userService.getUserInfo())
                    .withAppInfo(metadataService.getAppInfo())
                    .withDeviceInfo(metadataService.getDeviceInfo())
                    .withTelephonyInfo(metadataService.getTelephonyInfo())
                    .withEvent(event)
                    .build();

            apiClient.sendEvent(eventMessage);
        } catch (Exception ex) {
            EmbraceLogger.logError(
                    String.format(
                            "Cannot start event with name: %s, identifier: %s due to an exception",
                            name,
                            identifier),
                    ex);
        }
    }

    @NonNull
    private Long calculateLateThreshold(String eventId) {
        // Check whether a late threshold has been configured, otherwise use the default
        Map<String, Long> limits = configService.getConfig().getEventLimits();
        return Maps.getOrDefault(limits, eventId, DEFAULT_LATE_THRESHOLD_MILLIS);
    }

    @NonNull
    private Long calculateOffset(Long startTime, Long threshold) {
        // Ensure we adjust the threshold to take into account backdated events
        return Math.min(threshold, Math.max(0, System.currentTimeMillis() - startTime));
    }

    @Override
    public void endEvent(String name) {
        endEvent(name, null);
    }

    @Override
    public void endEvent(String name, String identifier) {
        endEvent(name, identifier, false, null);
    }

    @Override
    public void endEvent(String name, String identifier, Map<String, Object> properties) {
        endEvent(name, identifier, false, properties);
    }

    private void endEvent(String name, String identifier, boolean late, Map<String, Object> properties) {

        try {
            if (configService.isMessageTypeDisabled(MessageType.EVENT)) {
                EmbraceLogger.logWarning("Event message disabled. Ignoring all Events.");
                return;
            }

            String eventKey = getInternalEventKey(name, identifier);
            EventDescription eventDescription;
            if (late) {
                eventDescription = activeEvents.get(eventKey);
            } else {
                eventDescription = activeEvents.remove(eventKey);
            }
            if (eventDescription == null) {
                EmbraceLogger.logError(
                        String.format("No start event found when ending an event with name: %s, identifier: %s",
                                name,
                                identifier));
                return;
            }


            Event event = eventDescription.getEvent();
            long startTime = event.getTimestamp();
            long endTime = System.currentTimeMillis();
            long duration = Math.max(0, endTime - startTime);
            boolean screenshotTaken = false;
            if (late && eventDescription.isAllowScreenshot() && !configService.isScreenshotDisabledForEvent(name)) {
                try {
                    screenshotTaken = screenshotService.takeScreenshotMoment(event.getEventId());
                } catch (Exception ex) {
                    EmbraceLogger.logWarning("Failed to take screenshot for event " + name, ex);
                }
            } else {
                eventDescription.getLateTimer().cancel(false);
            }
            Event.Builder builder = Event.newBuilder()
                    .withEventId(event.getEventId())
                    .withAppState(metadataService.getAppState())
                    .withTimestamp(endTime)
                    .withDuration(duration)
                    .withName(name)
                    .withScreenshotTaken(screenshotTaken)
                    .withCustomProperties(properties)
                    .withType(late ? EmbraceEvent.Type.LATE : EmbraceEvent.Type.END);
            if (metadataService.getActiveSessionId().isPresent()) {
                builder = builder.withSessionId(metadataService.getActiveSessionId().get());
            }
            EventMessage eventMessage = EventMessage.newBuilder()
                    .withPerformanceInfo(performanceInfoService.getPerformanceInfo(startTime, endTime))
                    .withUserInfo(userService.getUserInfo())
                    .withEvent(builder.build())
                    .build();
            apiClient.sendEvent(eventMessage);
        } catch (Exception ex) {
            EmbraceLogger.logError(
                    String.format(
                            "Cannot end event with name: %s, identifier: %s due to an exception",
                            name,
                            identifier),
                    ex);
        }
    }

    @Override
    public List<String> findEventIdsForSession(long startTime, long endTime) {
        return new ArrayList<>(this.eventIds.subMap(startTime, endTime).values());
    }

    @Override
    public List<String> getActiveEventIds() {
        return StreamSupport.stream(this.activeEvents.values())
                .map(event -> event.getEvent().getEventId())
                .collect(Collectors.toList());
    }

    @Override
    public void close() {
        lateEventWorker.close();
    }

    private class EventDescription {
        private final Future<?> lateTimer;
        private final Event event;
        private final boolean allowScreenshot;

        EventDescription(Future<?> lateTimer, Event event, boolean allowScreenshot) {
            this.lateTimer = lateTimer;
            this.event = event;
            this.allowScreenshot = allowScreenshot;
        }

        public Future<?> getLateTimer() {
            return lateTimer;
        }

        public Event getEvent() {
            return event;
        }

        public boolean isAllowScreenshot() {
            return allowScreenshot;
        }
    }
}
