package io.embrace.android.embracesdk;

import android.os.Process;
import android.os.SystemClock;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import io.embrace.android.embracesdk.utils.exceptions.Unchecked;

/**
 * Polls the process and system uptime every 100ms to determine current CPU usage. CPU usage is
 * calculated as the ratio of the process's CPU time vs the total uptime.
 * <p>
 * If CPU usage is above 70%, this is recorded in a time series.
 * <p>
 * This depends on accessing the /proc virtual file system. Since Android Nougat a new SELinux
 * policy on the device prevents access to this filesystem, so CPU monitoring is not possible and
 * the service will fail to start.
 */
final class EmbraceCpuService implements CpuService {
    private static final double HIGH_CPU_USAGE_THRESHOLD = 0.7;

    private final ScheduledWorker cpuWorker;

    private final NavigableMap<Long, Boolean> cpuPegging;

    private RandomAccessFile procFile;

    private RandomAccessFile pidProcFile;

    private final AtomicInteger failureCount = new AtomicInteger(0);

    private volatile Long lastUptime;

    private volatile Long lastCpuTime;

    EmbraceCpuService() {
        this.cpuWorker = ScheduledWorker.ofSingleThread("CPU Service");
        this.cpuWorker.scheduleAtFixedRate(() -> cpuSample(), 0, 100, TimeUnit.MILLISECONDS);
        this.cpuPegging = new TreeMap<>();
        try {
            this.procFile = new RandomAccessFile("/proc/stat", "r");
            this.pidProcFile = new RandomAccessFile("/proc/" + Process.myPid() + "/stat", "r");
        } catch (FileNotFoundException ex) {
            EmbraceLogger.logInfo("CPU monitoring is currently not supported on this device.");
        } catch (Exception ex) {
            EmbraceLogger.logWarning("EmbraceCpuService failed to start. CPU monitoring unavailable.", ex);
        } finally {
            close();
        }
    }

    private long getTotalUpTime() {
        if (this.procFile != null) {
            try {
                procFile.seek(0);
                String[] stats = this.procFile.readLine().split("[ ]+", 9);
                long userTime = Long.parseLong(stats[1]);// user: normal processes executing in user mode
                long niceTime = Long.parseLong(stats[2]);// nice: niced processes executing in user mode
                long systemTime = Long.parseLong(stats[3]); // system: processes executing in kernel mode

                long idleTime = Long.parseLong(stats[4]); // idle
                long ioWaitTime = Long.parseLong(stats[5]);// iowait: waiting for I/O to complete
                long irqTime = Long.parseLong(stats[6]);  // irq: servicing interrupts
                long softIrqTime = Long.parseLong(stats[7]);// softirq: servicing softirqs
                return userTime + niceTime + systemTime + idleTime + ioWaitTime + irqTime + softIrqTime;
            } catch (Exception ex) {
                throw new RuntimeException(ex);
            }
        } else {
            throw new IllegalStateException("proc file reader not initialized");
        }
    }

    private long getTotalProcessCpuTime() {
        if (this.pidProcFile != null) {
            // we split first row of file, all cpu's together stats
            try {
                pidProcFile.seek(0);
                String[] stats = this.pidProcFile.readLine().split("[ ]+", 18); // we split first row of file, all cpu's together stats
                long userTime = Long.parseLong(stats[13]);// utime;/** user mode jiffies **/
                long systemTime = Long.parseLong(stats[14]);// stime;/** kernel mode jiffies **/
                long userChildTime = Long.parseLong(stats[15]); // cutime; /** user mode jiffies with childs **/
                long systemChildTime = Long.parseLong(stats[16]); // cstime; /** kernel mode jiffies with childs **/
                return userTime + systemTime + userChildTime + systemChildTime;
            } catch (IOException ex) {
                throw new RuntimeException(ex);
            }
        } else {
            throw new IllegalStateException("PID proc file reader not initialized");
        }
    }

    private void cpuSample() {
        try {
            long uptime = getTotalUpTime();
            long cpuTime = getTotalProcessCpuTime();
            if (lastUptime != null && lastCpuTime != null) {
                long uptimeDiff = uptime - lastUptime;
                long cpuTimeDiff = cpuTime - lastCpuTime;
                if (uptimeDiff > 0) {
                    // Cap at 100% CPU usage
                    double result = Math.max(1.0, (cpuTimeDiff / (double) uptimeDiff));
                    saveResult(result);
                }
                lastUptime = uptime;
                lastCpuTime = cpuTime;
            }

        } catch (Exception ex) {
            // Avoid writing to the logs too frequently if CPU sampling is failing
            if (failureCount.incrementAndGet() > 50) {
                EmbraceLogger.logWarning("Failed to sample CPU usage", ex);
                failureCount.set(0);
            }
        }
    }

    private void saveResult(double result) {
        synchronized (this) {
            boolean pegging = result > HIGH_CPU_USAGE_THRESHOLD;
            if (cpuPegging.isEmpty() || cpuPegging.lastEntry().getValue() != pegging) {
                cpuPegging.put(System.currentTimeMillis(), pegging);
            }
        }
    }

    @Override
    public List<Interval> getCpuCriticalIntervals(long startTime, long endTime) {
        synchronized (this) {
            List<Interval> results = new ArrayList<>();
            for (Map.Entry<Long, Boolean> entry : cpuPegging.subMap(startTime, endTime).entrySet()) {
                if (entry.getValue()) {
                    long currentTime = entry.getKey();
                    // The next key above should be false, since we only write when we stop pegging
                    Long next = cpuPegging.higherKey(currentTime);
                    results.add(new Interval(currentTime, next != null ? next : currentTime));
                }
            }
            return results;
        }
    }

    @Override
    public void close() {
        cpuWorker.close();
        if (procFile != null) {
            Unchecked.wrap(() -> procFile.close());
        }
        if (pidProcFile != null) {
            Unchecked.wrap(() -> pidProcFile.close());
        }
    }
}
