package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.io.IOException;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import io.embrace.android.embracesdk.utils.exceptions.Unchecked;

import static io.embrace.android.embracesdk.EmbracePreferencesService.SDK_STARTUP_COMPLETED;

/**
 * Handles the lifecycle of an Embrace session.
 * <p>
 * A session encapsulates all metrics gathered between the app being foregrounded and the app being
 * backgrounded. A caching service runs on an interval of every
 * {@value EmbraceSessionService#SESSION_CACHING_INTERVAL} seconds which saves the current session
 * to disk, so if the app is terminated, the session can resume from where it left off.
 */
final class EmbraceSessionService implements SessionService, ActivityListener {

    /** Signals to the API that the application was in the foreground. */
    public static final String APPLICATION_STATE_ACTIVE = "active";
    /** Signals to the API that the application was in the background. */
    public static final String APPLICATION_STATE_BACKGROUND = "background";
    /** Signals to the API the start of a session. */
    private static final String SESSION_START_TYPE = "st";
    /** Signals to the API the end of a session. */
    private static final String SESSION_END_TYPE = "en";
    /** The name of the file containing the cached session. */
    private static final String SESSION_FILE_NAME = "last_session.json";
    /** Sesion caching interval in seconds. */
    private static final int SESSION_CACHING_INTERVAL = 2;
    /** ApiClient timeout for the active session send service */
    private static final int SEND_SESSION_API_CLIENT_TIMEOUT = 2;
    /** Synchronization lock. */
    private final Object lock = new Object();

    /** Embrace service dependencies of the session service. */
    private final PreferencesService preferencesService;
    private final PerformanceInfoService performanceInfoService;
    private final MetadataService metadataService;
    private final BreadcrumbService breadcrumbService;
    private final PowerService powerService;
    private final ActivityService activityService;
    private final ApiClient apiClient;
    private final EventService eventService;
    private final EmbraceRemoteLogger remoteLogger;
    private final UserService userService;
    private final ConfigService configService;
    private final CacheService cacheService;

    /** Sequence number for the session used by the API for message ordering. */
    private final AtomicInteger sessionNumber = new AtomicInteger(0);

    /** Asynchronous workers. */
    private final BackgroundWorker sessionBackgroundWorker;
    private volatile ScheduledWorker sessionCacheWorker;

    /** The currently active session. */
    private volatile Session activeSession;

    EmbraceSessionService(
            PreferencesService preferencesService,
            PerformanceInfoService performanceInfoService,
            MetadataService metadataService,
            BreadcrumbService breadcrumbService,
            PowerService powerService,
            ActivityService activityService,
            ApiClient apiClient,
            EventService eventService,
            EmbraceRemoteLogger remoteLogger,
            UserService userService,
            ConfigService configService,
            CacheService cacheService) {

        this.preferencesService = Preconditions.checkNotNull(preferencesService);
        this.performanceInfoService = Preconditions.checkNotNull(performanceInfoService);
        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.breadcrumbService = Preconditions.checkNotNull(breadcrumbService);
        this.powerService = Preconditions.checkNotNull(powerService);
        this.activityService = Preconditions.checkNotNull(activityService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.eventService = Preconditions.checkNotNull(eventService);
        this.remoteLogger = Preconditions.checkNotNull(remoteLogger);
        this.userService = Preconditions.checkNotNull(userService);
        this.configService = Preconditions.checkNotNull(configService);
        this.cacheService = Preconditions.checkNotNull(cacheService);
        this.activityService.addListener(this, true);
        this.sessionBackgroundWorker = BackgroundWorker.ofSingleThread("Session");
    }

    private void startCaching() {
        this.sessionCacheWorker = ScheduledWorker.ofSingleThread("Session Caching Service");
        this.sessionCacheWorker.scheduleAtFixedRate(
                this::cacheActiveSession,
                0,
                SESSION_CACHING_INTERVAL,
                TimeUnit.SECONDS);
    }

    private void stopCaching() {
        this.sessionCacheWorker.close();
    }

    @Override
    public void startSession(boolean coldStart) {
        if (configService.isMessageTypeDisabled(MessageType.SESSION)) {
            EmbraceLogger.logWarning("Session messages disabled. Ignoring all Sessions.");
            return;
        }
        String sessionId = Uuid.getEmbUuid();
        SessionMessage message = sessionStart(coldStart, sessionNumber.incrementAndGet(), sessionId);
        this.activeSession = message.getSession();
        metadataService.setActiveSessionId(sessionId);
        apiClient.sendSession(message);
    }

    @Override
    public boolean sdkStartupFailedLastSession() {
        Optional<String> lastSessionStartupProgress = preferencesService.getSDKStartupStatus();
        return lastSessionStartupProgress.isPresent() && !lastSessionStartupProgress.get().equals(SDK_STARTUP_COMPLETED);
    }

    @Override
    public void handleCrash() {
        synchronized (lock) {
            if (this.activeSession != null) {
                SessionMessage sessionMessage = sessionEnd(this.activeSession, false, false, true);
                Unchecked.wrap(() -> apiClient.sendSession(sessionMessage).get());
                cacheService.deleteObject(SESSION_FILE_NAME);
            }
        }
    }

    private SessionMessage sessionStart(boolean coldStart, int number, String id) {
        Session.Builder builder = Session.newBuilder();
        if (powerService.getLatestBatteryLevel().isPresent()) {
            builder.withStartingBatteryLevel(powerService.getLatestBatteryLevel().get());
        }
        Session session = builder
                .withSessionId(id)
                .withStartTime(System.currentTimeMillis())
                .withNumber(number)
                .withColdStart(coldStart)
                .withSessionType(SESSION_START_TYPE)
                .build();
        return SessionMessage.newBuilder()
                .withSession(session)
                .withDeviceInfo(metadataService.getDeviceInfo())
                .withAppInfo(metadataService.getAppInfo())
                .build();
    }

    /**
     * Saves a cacheable version of the session, with performance information generated up to
     * the current point.
     */
    private void cacheActiveSession() {
        try {
            synchronized (lock) {
                if (activeSession != null) {
                    SessionMessage sessionMessage = sessionEnd(activeSession, false, true, false);
                    cacheService.cacheObject(SESSION_FILE_NAME, sessionMessage, SessionMessage.class);
                }
            }
        } catch (Exception ex) {
            EmbraceLogger.logError("Error whilst caching active session", ex);
        }
    }

    private SessionMessage sessionEnd(Session session,
                                      boolean endedCleanly,
                                      boolean forceQuit,
                                      boolean crashed) {

        Long startTime = session.getStartTime();
        long endTime = System.currentTimeMillis();
        Session.Builder builder = Session.newBuilder(session)
                .withEndedCleanly(endedCleanly)
                .withLastState(getApplicationState())
                .withSessionType(SESSION_END_TYPE)
                .withStoryIds(eventService.findStoryIds(startTime, endTime))
                .withInfoLogIds(remoteLogger.findInfoLogIds(startTime, endTime))
                .withWarningLogIds(remoteLogger.findWarningLogIds(startTime, endTime))
                .withErrorLogIds(remoteLogger.findErrorLogIds(startTime, endTime))
                .withCrashed(crashed)
                .withLastHeartbeatTime(System.currentTimeMillis());
        if (forceQuit) {
            builder = builder
                    .withTerminationTime(System.currentTimeMillis())
                    .withReceivedTermination(true);
        } else {
            // We don't set end time for force-quit, as the API interprets this to be a clean
            // termination
            builder = builder
                    .withEndTime(endTime);
        }
        return SessionMessage.newBuilder()
                .withUserInfo(userService.getUserInfo())
                .withAppInfo(metadataService.getAppInfo())
                .withDeviceInfo(metadataService.getDeviceInfo())
                .withPerformanceInfo(performanceInfoService.getSessionPerformanceInfo(startTime, endTime))
                .withBreadcrumbs(breadcrumbService.getBreadcrumbs(startTime, endTime))
                .withSession(builder.build())
                .build();
    }

    private String getApplicationState() {
        if (activityService.isInBackground()) {
            return APPLICATION_STATE_BACKGROUND;
        } else {
            return APPLICATION_STATE_ACTIVE;
        }
    }

    @Override
    public void onBackground() {
        synchronized (lock) {
            stopCaching();
            if (activeSession != null) {
                if (configService.isMessageTypeDisabled(MessageType.SESSION)) {
                    EmbraceLogger.logWarning("Session messages disabled. Ignoring all Sessions.");
                    return;
                }
                SessionMessage sessionMessage = sessionEnd(activeSession, true, false, false);
                activeSession = null;
                metadataService.setActiveSessionId(null);
                try {
                    // During power saving mode, requests are hanged forever causing an ANR when
                    // Foregrounded, so the Timeout prevents this scenario from happening.
                    apiClient.sendSession(sessionMessage).get(SEND_SESSION_API_CLIENT_TIMEOUT, TimeUnit.SECONDS);
                    cacheService.deleteObject(SESSION_FILE_NAME);
                } catch (Exception ex) {
                    EmbraceLogger.logError("Failed to send session end message", ex);
                }
            }
        }
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        synchronized (lock) {
            sendCachedSession();
            startCaching();
            startSession(coldStart);
            addViewBreadcrumbForResumedSession();
        }
    }

    /**
     * As when resuming the app a new session is created, the screen would be the same as the
     * one the app had when was backrounded. In this case, the SDK skips the "duplicate view
     * breadcrumb" scenario. This method, forces the repetition of the view breadcrumb when the
     * app's being resumed.
     */
    private void addViewBreadcrumbForResumedSession() {
        Optional<String> screen = breadcrumbService.getLastViewBreadcrumbScreenName();
        if (screen.isPresent()) {
            breadcrumbService.forceLogView(screen.orNull(), System.currentTimeMillis());
        }
    }

    /**
     * Send the cached session on a background thread, so that the main thread does not get
     * blocked during foreground.
     */
    private void sendCachedSession() {
        sessionBackgroundWorker.submit(() -> {
            Optional<SessionMessage> optionalSession =
                    cacheService.loadObject(SESSION_FILE_NAME, SessionMessage.class);
            if (optionalSession.isPresent()) {
                apiClient.sendSession(optionalSession.get());
            }
            return null;
        });
    }

    @Override
    public void close() {
        EmbraceLogger.logInfo("Shutting down EmbraceSessionService");
        sessionCacheWorker.close();
        sessionBackgroundWorker.close();
    }
}
