package io.embrace.android.embracesdk;

import android.text.TextUtils;

import java.util.Map;

/**
 * This class is responsible for tracking app performance during registration flows.
 * <p>
 * This class is thread-safe.
 */
public final class RegistrationFlow extends CustomFlow {
    public static final long FLAG_SET_USER_PAYER = 1 << 0;

    static final String MOMENT_INTERNAL_REGISTRATION = "_internal-registration";
    static final String MOMENT_EXTERNAL_REGISTRATION = "_external-registration";

    static final String PROP_EMAIL = "email";
    static final String PROP_REGISTRATION_SOURCE = "registration-source";
    static final String PROP_USERNAME = "username";
    static final String PROP_USER_ID = "user-id";

    private Map<String, Object> registrationProps;
    private volatile String registrationMomentName;
    private volatile String registrationMomentId;

    /**
     * Starts a registration moment.
     * <p>
     * This method should be called as soon as a user wishes to submit their user registration info. This method is
     * designed specifically for cases where the registration occurs through an internally managed service as opposed to
     * an external service (e.g. Google, Facebook, GitHub Login).
     *
     * @param userId     An identifier that uniquely represents the user (e.g. frequent flyer number). This value is optional
     *                   and, if present, will associate the value as a property of the moment.
     * @param username   The username of the user. This value is optional and, if present, will associate the value as a
     *                   property of the moment.
     * @param email      The email address of the user. This value is optional and, if present, will associate the value as a
     *                   property of the moment.
     * @param properties A map of Strings to Objects that represent additional properties to associate with the moment.
     *                   This value is optional. A maximum of 10 properties (not including the ones set via arguments to
     *                   this method) may be set.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean registrationStart(String userId,
                                     String username,
                                     String email,
                                     Map<String, Object> properties) {

        Map<String, Object> normalizedProperties = PropertyUtils.sanitizeProperties(properties);

        if (userId != null) {
            normalizedProperties.put(PROP_USER_ID, userId);
        }
        if (username != null) {
            normalizedProperties.put(PROP_USERNAME, username);
        }
        if (email != null) {
            normalizedProperties.put(PROP_EMAIL, email);
        }

        return sendRegistrationStartMoment(MOMENT_INTERNAL_REGISTRATION, normalizedProperties);
    }

    /**
     * Starts a registration moment.
     * <p>
     * This method should be called as soon as a user wishes to submit their user registration info. This method is
     * designed specifically for cases where the registration occurs through an external authentication service (e.g.
     * Google, Facebook, GitHub Login).
     *
     * @param source     The registration system that will be authenticating the user. This value is optional and,
     *                   if present, will associate the value as a property of the moment.
     * @param properties A map of Strings to Objects that represent additional properties to associate with the moment.
     *                   This value is optional. A maximum of 10 properties (not including the ones set via arguments to
     *                   this method) may be set.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean registrationStart(String source,
                                     Map<String, Object> properties) {

        Map<String, Object> normalizedProperties = PropertyUtils.sanitizeProperties(properties);

        if (source != null) {
            normalizedProperties.put(PROP_REGISTRATION_SOURCE, source);
        }

        return sendRegistrationStartMoment(MOMENT_EXTERNAL_REGISTRATION, normalizedProperties);
    }

    boolean sendRegistrationStartMoment(String momentName,
                                        Map<String, Object> properties) {
        this.registrationMomentName = momentName;
        this.registrationMomentId = Uuid.getEmbUuid();
        this.registrationProps = PropertyUtils.sanitizeProperties(properties);

        sendMomentStartEvent(this.registrationMomentName, this.registrationMomentId, false, this.registrationProps);
        return true;
    }

    /**
     * Ends the registration moment and generates an info log message that indicates that the registration completed.
     * <p>
     * This method should be called once the registration information has been submitted. If any of the following values
     * were defined when the registration moment was initially started, the SDK will register the device ID to all valid
     * user information.
     * - Email
     * - Username
     * - User Identifier
     *
     * @return True if the operation was successful; false otherwise.
     */
    public boolean registrationComplete() {
        return registrationComplete(null);
    }

    /**
     * Ends the registration moment and generates an info log message that indicates that the registration completed.
     * <p>
     * This method should be called once the registration information has been submitted. If any of the following values
     * were defined when the registration moment was initially started, the SDK will register the device ID to all valid
     * user information.
     * - Email
     * - Username
     * - User Identifier
     *
     * @param isPayer An optional value that indicates whether the user is a payer or not. If this value is null, then it
     *                will not modify the payer persona status of the user.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean registrationComplete(Boolean isPayer) {
        if (this.registrationMomentId == null) {
            EmbraceLogger.logError("Registration wasn't started.");
            return false;
        }

        sendMomentEndEvent(this.registrationMomentName, this.registrationMomentId);
        sendLogInfo("Registration was completed.", this.registrationProps);

        Embrace embrace = Embrace.getInstance();

        // TODO: Review which service is originally responsible for tracking this
        if (isPayer != null) {
            if (isPayer) {
                embrace.setUserAsPayer();
            } else {
                embrace.clearUserAsPayer();
            }
        }

        String email = (String) this.registrationProps.get(PROP_EMAIL);
        String username = (String) this.registrationProps.get(PROP_USERNAME);
        String userId = (String) this.registrationProps.get(PROP_USER_ID);

        if (!TextUtils.isEmpty(email)) {
            embrace.setUserEmail(email);
        }
        if (!TextUtils.isEmpty(username)) {
            embrace.setUsername(username);
        }
        if (!TextUtils.isEmpty(userId)) {
            embrace.setUserIdentifier(userId);
        }

        this.registrationMomentId = null;
        return true;
    }

    /**
     * Ends the registration moment and generates a log error message that indicates that the registration failed.
     * <p>
     * This method should be called once the registration information has been submitted.
     *
     * @param msg A message that explains the reason for why this operation failed. This value is optional and, if
     *            provided, will associate the value as a property of the error log message.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean registrationFail(String msg) {
        if (this.registrationMomentId == null) {
            EmbraceLogger.logError("Registration wasn't started.");
            return false;
        } else if (msg != null) {
            this.registrationProps.put(PROP_MESSAGE, msg);
        }

        String errorLogMsg;

        if (TextUtils.isEmpty(msg)) {
            errorLogMsg = "A failure occurred during registration.";
        } else {
            errorLogMsg = "A failure occurred during registration: " + msg;
        }

        sendMomentEndEvent(this.registrationMomentName, this.registrationMomentId);
        sendLogError(errorLogMsg, false, this.registrationProps);

        this.registrationMomentId = null;
        return true;
    }
}
