package io.embrace.android.embracesdk;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.CopyOnWriteArraySet;

import java9.util.stream.StreamSupport;

/**
 * Captures all network calls above 20kb and uses this to calculate the current connection class
 * of the device.
 * <p>
 * Consumers can register as listeners of this service to be notified of connection quality changes.
 */
final class EmbraceConnectionClassService implements ConnectionClassService {
    /**
     * Minimum initial number of samples before we will measure initial network quality.
     */
    private static final int MINIMUM_SAMPLES_THRESHOLD = 2;

    /**
     * Minimum number of samples before we will change the network quality from the existing value.
     */
    private static final int MINIMUM_SAMPLES_CONNECTION_CHANGE = 5;

    /**
     * Bandwidth in kilobits / sec which is considered poor bandwidth.
     */
    private static final int POOR_BANDWIDTH = 150;

    /**
     * Bandwidth in kilobits / sec which is considered moderate bandwidth.
     */
    private static final int MODERATE_BANDWIDTH = 550;

    /**
     * Bandwidth in kilobits / sec which is considered good bandwidth.
     */
    private static final int GOOD_BANDWIDTH = 2000;

    /**
     * Minimum bandwidth in kilobits / sec for a call to be accepted into the calculation.
     */
    private static final int MINIMUM_BANDWIDTH = 10;

    /**
     * Minimum number of bytes for a sample to be accepted.
     */
    private static final long MINIMUM_CONTENT_LENGTH = 5000;

    /**
     * The last connection quality measurement.
     */
    private volatile ConnectionQuality connectionQuality = ConnectionQuality.UNKNOWN;

    /**
     * The bandwidth across the currently tracked samples.
     */
    private volatile int currentBandwidthSampling = 0;

    /**
     * The number of samples taken since the last connection quality change.
     */
    private volatile int numberOfSamples = 0;

    /**
     * The bandwidth at the last connection quality change.
     */
    private volatile int currentBandwidth = 0;

    private final Object lock = new Object();
    private final Set<ConnectionQualityListener> listeners = new CopyOnWriteArraySet<>();

    private final NavigableMap<Long, ConnectionStat> qualityIntervals;

    EmbraceConnectionClassService() {
        qualityIntervals = new TreeMap<>();
    }


    @Override
    public void logBandwidth(long bytes, long timeMs) {
        // Only log requests with at least 5k bytes + a valid duration
        if (bytes < MINIMUM_CONTENT_LENGTH || timeMs <= 0) {
            return;
        }
        // Only log requests above a minimum bandwidth
        double bandwidth = (bytes * 1.0 / timeMs) * 8;
        if (bandwidth < MINIMUM_BANDWIDTH) {
            return;
        }

        synchronized (lock) {
            currentBandwidthSampling = (int) ((currentBandwidthSampling *
                    numberOfSamples + bandwidth) / (numberOfSamples + 1));

            numberOfSamples++;

            boolean initialMeasurement =
                    (connectionQuality == ConnectionQuality.UNKNOWN &&
                            numberOfSamples == MINIMUM_SAMPLES_THRESHOLD);

            if (numberOfSamples == MINIMUM_SAMPLES_CONNECTION_CHANGE || initialMeasurement) {
                final ConnectionQuality lastConnectionQuality = connectionQuality;
                currentBandwidth = currentBandwidthSampling;
                if (currentBandwidthSampling <= 0) {
                    connectionQuality = ConnectionQuality.UNKNOWN;
                } else if (currentBandwidthSampling < POOR_BANDWIDTH) {
                    connectionQuality = ConnectionQuality.POOR;
                } else if (currentBandwidthSampling < MODERATE_BANDWIDTH) {
                    connectionQuality = ConnectionQuality.MODERATE;
                } else if (currentBandwidthSampling < GOOD_BANDWIDTH) {
                    connectionQuality = ConnectionQuality.GOOD;
                } else if (currentBandwidthSampling > GOOD_BANDWIDTH) {
                    connectionQuality = ConnectionQuality.EXCELLENT;
                }
                if (numberOfSamples == MINIMUM_SAMPLES_CONNECTION_CHANGE) {
                    currentBandwidthSampling = 0;
                    numberOfSamples = 0;
                }
                if (connectionQuality != lastConnectionQuality) {
                    // Connection quality change
                    qualityIntervals.put(System.currentTimeMillis(), new ConnectionStat(connectionQuality, currentBandwidth));
                    EmbraceLogger.logDebug("Connection quality change to " + connectionQuality);
                    notifyListeners(lastConnectionQuality, connectionQuality, currentBandwidth);
                }
            }
        }
    }

    @Override
    public void addListener(ConnectionQualityListener listener) {
        listeners.add(listener);
        try {
            // Notify listener of the current state upon registration
            listener.onConnectionQualityChange(connectionQuality, connectionQuality, currentBandwidthSampling);
        } catch (Exception ex) {
            EmbraceLogger.logWarning("Failed to notify ConnectionQualityListener", ex);
        }
    }

    @Override
    public void removeListener(ConnectionQualityListener listener) {
        listeners.remove(listener);
    }

    @Override
    public List<ConnectionQualityInterval> getQualityIntervals(long startTime, long endTime) {
        synchronized (lock) {
            List<ConnectionQualityInterval> results = new ArrayList<>();
            for (Map.Entry<Long, ConnectionStat> entry : qualityIntervals.subMap(startTime, endTime).entrySet()) {
                long currentTime = entry.getKey();
                Long next = qualityIntervals.higherKey(currentTime);
                ConnectionStat stats = entry.getValue();
                results.add(new ConnectionQualityInterval(
                        currentTime,
                        next != null ? next : currentTime,
                        stats.connectionQuality,
                        stats.bandwidth));
            }
            return results;
        }
    }

    private void notifyListeners(
            ConnectionQuality previousQuality,
            ConnectionQuality newQuality,
            int bandwidth) {

        StreamSupport.stream(listeners).forEach(listener -> {
            try {
                listener.onConnectionQualityChange(previousQuality, newQuality, bandwidth);
            } catch (Exception ex) {
                EmbraceLogger.logWarning("Failed to notify ConnectionQualityListener", ex);
            }
        });
    }

    private class ConnectionStat {
        private final ConnectionQuality connectionQuality;

        private final int bandwidth;

        private ConnectionStat(ConnectionQuality connectionQuality, int bandwidth) {
            this.connectionQuality = connectionQuality;
            this.bandwidth = bandwidth;
        }
    }
}
