package io.embrace.android.embracesdk;

import android.text.TextUtils;

import java.util.HashMap;
import java.util.Map;

import java9.util.Maps;

/**
 * Base class for creating custom domain-specific flows that are essentially convenience wrappers around existing SDK
 * functionality.
 */
abstract class CustomFlow {

    protected static final String PROP_MOMENT_ID = "moment-id";
    protected static final String PROP_MOMENT_NAME = "moment-name";
    protected static final String PROP_MESSAGE = "message";

    protected final Map<String, Map<String, Moment>> customMoments = new HashMap<>();

    private EventService eventService;
    private EmbraceRemoteLogger embraceRemoteLogger;

    protected CustomFlow() {
        this.eventService = Embrace.getInstance().getEventService();
        this.embraceRemoteLogger = Embrace.getInstance().getRemoteLogger();
    }

    /**
     * Starts a custom moment.
     *
     * @param momentName          The name of the moment.
     * @param doesAllowScreenshot If true, a screenshot will be taken if the moment exceeds the late threshold. If this
     *                            value is false, a screenshot will be not be taken regardless of the moment duration.
     * @param properties          A map of Strings to Objects that represent additional properties to associate with the moment.
     *                            This value is optional. A maximum of 10 properties may be set.
     * @return A moment identifier that uniquely identifies the newly started moment instance.
     */
    public String momentStart(String momentName, boolean doesAllowScreenshot, Map<String, Object> properties) {
        if (TextUtils.isEmpty(momentName)) {
            sendLogError("Moment name is null or blank.", false, null);
            return null;
        } else if (momentName.startsWith("_")) {
            sendLogError("Moment name may not start with '_'.", false, null);
            return null;
        }

        Maps.putIfAbsent(customMoments, momentName, new HashMap<>());
        String momentId = Uuid.getEmbUuid();
        Map<String, Moment> moments = this.customMoments.get(momentName);

        sendMomentStartEvent(momentName, momentId, doesAllowScreenshot, PropertyUtils.sanitizeProperties(properties));

        // Register the custom moment instance so that it can be accessed later.
        moments.put(momentId, new Moment(momentName, momentId, doesAllowScreenshot, properties));
        return momentId;
    }

    /**
     * Completes all started instances of the specified custom moment.
     * <p>
     * Note that only moment instances managed by this Flow object will be completed. In other words, if another Flow
     * instance starts a moment with the same name, completing the moment on this instance will not affect it.
     *
     * @param momentName The name of the moment.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean momentComplete(String momentName) {
        return momentComplete(momentName, null);
    }

    /**
     * Completes a started instance of the custom moment specified by the moment identifier.
     * <p>
     * Note that only moment instances managed by this Flow object will be completed. In other words, if another Flow
     * instance starts a moment with the same name, completing the moment on this instance will not affect it.
     *
     * @param momentName The name of the moment.
     * @param momentId   The optional moment identifier returned by the `momentStart` method. This moment identifier must be
     *                   an identifier produced by this particular Flow instance that has not already been completed or
     *                   failed. This value can also be null, in which case all instances of the given moment name
     *                   registered with this Flow instance will be completed.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean momentComplete(String momentName, String momentId) {

        if (TextUtils.isEmpty(momentName)) {
            sendLogError("Moment name is null or blank.", false, null);
            return false;
        }

        // Get all the moment instances for the requested moment.
        Map<String, Moment> moments = this.customMoments.get(momentName);

        if (moments == null) {
            EmbraceLogger.logError(String.format("Cannot fail %s because moment name is not recognized.", momentName));
            return false;
        }

        if (momentId == null) {
            // Complete all known moment instances for the requested moment.
            for (Map.Entry<String, Moment> entry : moments.entrySet()) {
                sendMomentEndEvent(momentName, entry.getKey());
            }
            moments.clear();

        } else {
            // Complete only the specific moment instance that was requested.
            Moment moment = moments.get(momentId);
            if (moment == null) {
                EmbraceLogger.logError(String.format("Cannot fail %s because moment identifier is not recognized.", momentName));
                return false;
            }
            sendMomentEndEvent(momentName, momentId);
            moments.remove(momentId);
        }

        if (moments.isEmpty()) {
            // No instances of the moment name is running so remove the entire entry in the custom moment map.
            this.customMoments.remove(momentName);
        }

        return true;
    }

    /**
     * Fails all started instances of the specified custom moment and generates an error log message for each failed
     * moment instance.
     * <p>
     * Note that only moment instances managed by this Flow object will be failed. In other words, if another Flow
     * instance fails a moment with the same name, failing the moment on this instance will not affect it.
     *
     * @param momentName The name of the moment.
     * @param msg        A message that explains the reason for why this operation failed. This value is optional and, if
     *                   provided, will associate the value as a property of the error log message.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean momentFail(String momentName, String msg) {
        return momentFail(momentName, null, msg);
    }

    /**
     * Fails a started instance of the custom moment specified by the moment identifier and sends an error log message for
     * the failed moment instance.
     * <p>
     * Note that only moment instances managed by this Flow object will be failed. In other words, if another Flow
     * instance fails a moment with the same name, failing the moment on this instance will not affect it.
     *
     * @param momentName The name of the moment.
     * @param momentId   The optional moment identifier returned by the `momentStart` method. This moment identifier must be
     *                   an identifier produced by this particular Flow instance that has not already been completed or
     *                   failed. This value can also be null, in which case all instances of the given moment name
     *                   registered with this Flow instance will be completed.
     * @param msg        A message that explains the reason for why this operation failed. This value is optional and, if
     *                   provided, will associate the value as a property of the error log message.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean momentFail(String momentName, String momentId, String msg) {
        if (TextUtils.isEmpty(momentName)) {
            sendLogError("Moment name is null or blank.", false, null);
            return false;
        }

        String errorLogMsg;

        if (TextUtils.isEmpty(msg)) {
            errorLogMsg = String.format("A failure occurred during the %s moment.", momentName);
        } else {
            errorLogMsg = String.format("A failure occurred during the %s moment: %s", momentName, msg);
        }

        // Get all the moment instances for the requested moment.
        Map<String, Moment> moments = this.customMoments.get(momentName);

        if (moments == null) {
            EmbraceLogger.logError(String.format("Cannot fail %s because moment name is not recognized.", momentName));
            return false;
        }

        if (momentId == null) {
            // Fail all known moment instances for the requested moment.

            for (Map.Entry<String, Moment> entry : moments.entrySet()) {
                Moment moment = entry.getValue();

                if (msg != null) {
                    moment.properties.put(PROP_MESSAGE, msg);
                }

                sendMomentEndEvent(momentName, entry.getKey());
                sendLogError(errorLogMsg, moment.doesAllowScreenshot, moment.properties);
            }

            moments.clear();
        } else {
            // Fail only the specific moment instance that was requested.

            Moment moment = moments.get(momentId);

            if (moment == null) {
                EmbraceLogger.logError(String.format(
                        "Cannot fail %s because moment identifier is not recognized.", momentName));
                return false;
            } else if (msg != null) {
                moment.properties.put(PROP_MESSAGE, msg);
            }

            sendMomentEndEvent(momentName, momentId);
            sendLogError(errorLogMsg, moment.doesAllowScreenshot, moment.properties);
            moments.remove(momentId);
        }

        if (moments.isEmpty()) {
            // No instances of the moment name is running so remove the entire entry in the custom moment map.
            this.customMoments.remove(momentName);
        }

        return true;
    }

    protected void sendMomentStartEvent(String momentName,
                                        String momentId,
                                        boolean doesAllowScreenshot,
                                        Map<String, Object> properties) {
        try {
            if (this.eventService != null) {
                this.eventService.startEvent(momentName, momentId, doesAllowScreenshot, properties);
            } else {
                throw new Exception("Event service is null. Embrace SDK might not be started.");
            }
        } catch (Exception e) {
            EmbraceLogger.logError(String.format("An error occurred trying to start moment: %s - %s.", momentName, momentId),
                    e);
        }
    }

    protected void sendMomentEndEvent(String momentName,
                                      String momentId) {

        try {
            if (this.eventService != null) {
                this.eventService.endEvent(momentName, momentId);
            } else {
                throw new Exception("Event service is null. Embrace SDK might not be started.");
            }
        } catch (Exception e) {
            EmbraceLogger.logError(String.format("An error occurred trying to end moment: %s - %s.", momentName, momentId), e);
        }
    }

    protected void sendLogInfo(String msg,
                               Map<String, Object> properties) {

        try {
            if (this.embraceRemoteLogger != null) {
                this.embraceRemoteLogger.log(msg, EmbraceEvent.Type.INFO_LOG, false, properties);
            } else {
                throw new Exception("Remote Logger is null. Embrace SDK might not be started.");
            }
        } catch (Exception e) {
            EmbraceLogger.logError(String.format("An error occurred sending log info message: %s.", msg), e);
        }
    }

    protected void sendLogError(String msg,
                                boolean doesAllowScreenshot,
                                Map<String, Object> properties) {

        try {
            if (this.embraceRemoteLogger != null) {
                this.embraceRemoteLogger.log(msg, EmbraceEvent.Type.ERROR_LOG, doesAllowScreenshot, properties);
            } else {
                throw new Exception("Remote Logger is null. Embrace SDK might not be started.");
            }
        } catch (Exception e) {
            EmbraceLogger.logError(String.format("An error occurred sending log error message: %s.", msg), e);
        }
    }

    private static class Moment {

        final String name;
        final String id;
        final boolean doesAllowScreenshot;
        final Map<String, Object> properties;

        Moment(String name, String id, boolean doesAllowScreenshot, Map<String, Object> properties) {
            this.name = name;
            this.id = id;
            this.doesAllowScreenshot = doesAllowScreenshot;
            this.properties = properties;
        }
    }
}
