package io.embrace.android.embracesdk;

import android.content.Context;
import android.telephony.PhoneStateListener;
import android.telephony.TelephonyManager;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.ArrayList;
import java.util.List;
import java.util.NavigableMap;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.CopyOnWriteArraySet;

import java9.util.stream.StreamSupport;

/**
 * Monitors changes in signal strength of the device, if enabled in {@link Config}.
 */
final class EmbraceSignalQualityService extends PhoneStateListener implements SignalQualityService, ConnectionQualityListener {
    /** Minimum time between samples in ms. */
    private static final long MIN_TIME_BETWEEN_SAMPLES = 10000L;

    private final NavigableMap<Long, SignalStrength> signalStrength;

    private final TelephonyManager telephonyManager;

    private final ConfigService configService;

    private final SignalStrengthUtils signalStrengthUtils;

    private final Set<ConnectionQualityListener> listeners = new CopyOnWriteArraySet<>();

    private volatile int signalLevel = SignalStrengthUtils.SIGNAL_STRENGTH_NONE_OR_UNKNOWN;

    private volatile ConnectionQuality lastQuality = ConnectionQuality.UNKNOWN;

    private volatile boolean estimatedQuality = true;

    private volatile int lastBandwidth = 0;

    /**
     * True if listening for signal strength changes is enabled, false otherwise.
     */
    private volatile boolean signalStrengthListening = false;

    EmbraceSignalQualityService(
            Context context,
            ConfigService configService,
            ActivityService activityService,
            ConnectionClassService connectionClassService) {

        signalStrength = new ConcurrentSkipListMap<>();
        Preconditions.checkNotNull(context, "context must not be null");
        this.telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
        this.configService = Preconditions.checkNotNull(configService, "configService must not be null");
        Preconditions.checkNotNull(connectionClassService, "connectionClassService must not be null");
        this.signalStrengthUtils = new SignalStrengthUtils(context);
        configService.addListener(this);
        activityService.addListener(this);
        connectionClassService.addListener(this);
    }



    @Override
    public List<SignalStrength> getSignalMeasurements(long startTime, long endTime) {
        return new ArrayList<>(this.signalStrength.subMap(startTime, endTime).values());
    }

    @Override
    public ConnectionQuality getEstimatedConnectionQuality() {
        switch (signalLevel) {
            case SignalStrengthUtils.SIGNAL_STRENGTH_NONE_OR_UNKNOWN:
                return ConnectionQuality.UNKNOWN;
            case SignalStrengthUtils.SIGNAL_STRENGTH_POOR:
                return ConnectionQuality.POOR;
            case SignalStrengthUtils.SIGNAL_STRENGTH_MODERATE:
                return ConnectionQuality.MODERATE;
            case SignalStrengthUtils.SIGNAL_STRENGTH_GOOD:
                return ConnectionQuality.GOOD;
            case SignalStrengthUtils.SIGNAL_STRENGTH_GREAT:
                return ConnectionQuality.EXCELLENT;
            default:
                return ConnectionQuality.UNKNOWN;
        }
    }

    @Override
    public void onSignalStrengthsChanged(android.telephony.SignalStrength signalStrength) {
        long now = System.currentTimeMillis();
        // Don't allow more than one sample every MIN_TIME_BETWEEN_SAMPLES milliseconds
        if (this.signalStrength.isEmpty() || (now - this.signalStrength.lastKey() >= MIN_TIME_BETWEEN_SAMPLES)) {
            SignalStrength result = SignalStrength.of(signalStrength);
            this.signalStrength.put(now, result);
        }
        this.signalLevel = signalStrengthUtils.getLevel(signalStrength);
        // Estimate connection quality from signal strength until it can be determined from throughput
        if (estimatedQuality && signalLevel != SignalStrengthUtils.SIGNAL_STRENGTH_NONE_OR_UNKNOWN) {
            ConnectionQuality newEstimate = getEstimatedConnectionQuality();
            if (!newEstimate.equals(lastQuality)) {
                notifyListeners(lastQuality, newEstimate, 0);
                this.lastQuality = newEstimate;
            }
        }
    }

    @Override
    public void onConfigChange(Config previousConfig, Config newConfig) {
        if (!previousConfig.getSignalStrengthEnabled().equals(newConfig.getSignalStrengthEnabled())) {
            if (newConfig.getSignalStrengthEnabled()) {
                registerPhoneStateListener();
                EmbraceLogger.logInfo("Starting signal strength listener as config was changed");
            } else {
                deregisterPhoneStateListener();
                EmbraceLogger.logInfo("Stopping signal strength listener as config was changed");

            }
        }
    }

    @Override
    public void close() {
        EmbraceLogger.logInfo("Stopping EmbraceSignalQualityService");
        deregisterPhoneStateListener();
    }

    @Override
    public void onBackground() {
        EmbraceLogger.logDebug("Pausing signal strength collection as app going to background");
        deregisterPhoneStateListener();
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime) {
        EmbraceLogger.logDebug("Resuming signal strength collection as app in foreground");
        registerPhoneStateListener();
    }


    @Override
    public void addListener(ConnectionQualityListener listener) {
        listeners.add(listener);
        try {
            // Notify listener of the current state upon registration
            listener.onConnectionQualityChange(lastQuality, lastQuality, lastBandwidth);
        } catch (Exception ex) {
            EmbraceLogger.logWarning("Failed to notify ConnectionQualityListener", ex);
        }
    }

    @Override
    public void removeListener(ConnectionQualityListener listener) {
        listeners.remove(listener);
    }


    @Override
    public void onConnectionQualityChange(
            ConnectionQuality previousConnectionQuality,
            ConnectionQuality newConnectionQuality,
            int currentBandwidth) {

        if (newConnectionQuality.equals(ConnectionQuality.UNKNOWN)) {
            // Estimate the quality from the signal strength if it cannot be determined from the network
            ConnectionQuality estimated = getEstimatedConnectionQuality();
            notifyListeners(previousConnectionQuality, estimated, currentBandwidth);
            this.lastQuality = estimated;
            this.estimatedQuality = true;
        } else {
            // Stop estimating quality
            this.estimatedQuality = false;
            notifyListeners(previousConnectionQuality, newConnectionQuality, currentBandwidth);
            this.lastQuality = newConnectionQuality;
        }
        this.lastBandwidth = currentBandwidth;
    }

    private void registerPhoneStateListener() {
        if (telephonyManager != null && configService.getConfig().getSignalStrengthEnabled() && !signalStrengthListening) {
            try {
                telephonyManager.listen(this, PhoneStateListener.LISTEN_SIGNAL_STRENGTHS);
                signalStrengthListening = true;
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to register signal strength listener", ex);
            }
        }
    }

    private void deregisterPhoneStateListener() {
        if (telephonyManager != null && signalStrengthListening) {
            try {
                telephonyManager.listen(this, PhoneStateListener.LISTEN_NONE);
                signalStrengthListening = false;
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to deregister signal strength listener", ex);
            }
        }
    }

    private void notifyListeners(
            ConnectionQuality previousQuality,
            ConnectionQuality newQuality,
            int bandwidth) {

        StreamSupport.stream(listeners).forEach(listener -> {
            try {
                listener.onConnectionQualityChange(previousQuality, newQuality, bandwidth);
            } catch (Exception ex) {
                EmbraceLogger.logWarning("Failed to notify ConnectionQualityListener", ex);
            }
        });
    }
}
