package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

/**
 * Version and environment information relating to the app which has embedded the Embrace SDK.
 */
final class AppInfo {

    /** The version of the app which has embedded the Embrace SDK. */
    @SerializedName("v")
    private final String appVersion;

    /**
     * A unique ID for the build which is generated at build time. This is written to a JSON file in
     * the build directory and read by {@link BuildInfo}.
     */
    @SerializedName("bi")
    private final String buildId;

    /** The name of the environment, i.e. dev or prod, determined by whether this is a debug build. */
    @SerializedName("e")
    private final String environment;

    /** Whether the app was updated since the previous launch. */
    @SerializedName("vu")
    private final boolean appUpdated;

    /** Whether the app was updated since the previous launch. */
    @SerializedName("vul")
    private final boolean appUpdatedThisLaunch;

    /** The app bundle version. */
    @SerializedName("bv")
    private final String bundleVersion;

    /** Whether the OS was updated since the last launch. */
    @SerializedName("ou")
    private final boolean osUpdated;

    /** Whether the OS was updated since the last launch. */
    @SerializedName("oul")
    private final boolean osUpdatedThisLaunch;

    /** The version number of the Embrace SDK. */
    @SerializedName("sdk")
    private final String sdkVersion;

    /** The simple version number of the Embrace SDK. */
    @SerializedName("sdc")
    private final String sdkSimpleVersion;

    private AppInfo(Builder builder) {
        appVersion = builder.appVersion;
        buildId = builder.buildId;
        environment = builder.environment;
        appUpdated = builder.appUpdated;
        appUpdatedThisLaunch = builder.appUpdatedThisLaunch;
        bundleVersion = builder.bundleVersion;
        osUpdated = builder.osUpdated;
        osUpdatedThisLaunch = builder.osUpdatedThisLaunch;
        sdkVersion = builder.sdkVersion;
        sdkSimpleVersion = builder.sdkSimpleVersion;
    }

    static Builder newBuilder() {
        return new Builder();
    }


    static final class Builder {
        private String appVersion;
        private String buildId;
        private String environment;
        private boolean appUpdated;
        private boolean appUpdatedThisLaunch;
        private String bundleVersion;
        private boolean osUpdated;
        private boolean osUpdatedThisLaunch;
        private String sdkVersion;
        private String sdkSimpleVersion;

        private Builder() {
        }

        public Builder withAppVersion(String appVersion) {
            this.appVersion = appVersion;
            return this;
        }

        public Builder withBuildId(String buildId) {
            this.buildId = buildId;
            return this;
        }

        public Builder withEnvironment(String environment) {
            this.environment = environment;
            return this;
        }

        public Builder withAppUpdated(boolean appUpdated) {
            this.appUpdated = appUpdated;
            return this;
        }

        public Builder withAppUpdatedThisLaunch(boolean appUpdatedThisLaunch) {
            this.appUpdatedThisLaunch = appUpdatedThisLaunch;
            return this;
        }

        public Builder withBundleVersion(String bundleVersion) {
            this.bundleVersion = bundleVersion;
            return this;
        }

        public Builder withOsUpdated(boolean osUpdated) {
            this.osUpdated = osUpdated;
            return this;
        }

        public Builder withOsUpdatedThisLaunch(boolean osUpdatedThisLaunch) {
            this.osUpdatedThisLaunch = osUpdatedThisLaunch;
            return this;
        }

        public Builder withSdkVersion(String sdkVersion) {
            this.sdkVersion = sdkVersion;
            return this;
        }

        public Builder withSdkSimpleVersion(String sdkSimpleVersion) {
            this.sdkSimpleVersion = sdkSimpleVersion;
            return this;
        }

        public AppInfo build() {
            return new AppInfo(this);
        }
    }
}
