package io.embrace.android.embracesdk;

import java.io.Closeable;
import java.util.List;

/**
 * Captures network calls and network interface changes from the device.
 */
interface NetworkService extends Closeable {
    /**
     * Gets the intervals during which the device was on WIFI, WAN, or not reachable;
     *
     * @param startTime the start time of the period to search
     * @param endTime the end time of the period to search
     * @return a list of intervals for each network type
     */
    List<Interval> getNetworkInterfaceIntervals(long startTime, long endTime);

    /**
     * Gets a list of network calls which started during the given interval.
     *
     * @param startTime the time to search from
     * @param endTime the time to search up until
     * @return the list of network calls within the given time period
     */
    List<NetworkCall> getNetworkCalls(long startTime, long endTime);

    NetworkSession getNetworkSession(long startTime, long lastKnownTime);

    /**
     * Logs a HTTP network call.
     *
     * @param url the URL being called
     * @param httpMethod the HTTP method
     * @param statusCode the status code from the response
     * @param startTime the start time of the request
     * @param endTime the end time of the request
     * @param bytesSent the number of bytes sent
     * @param bytesReceived the number of bytes received
     */
    void logNetworkCall(
            String url,
            String httpMethod,
            Integer statusCode,
            long startTime,
            long endTime,
            Long bytesSent,
            Long bytesReceived);

    /**
     * Logs an exception which occurred when attempting to make a network call.
     *
     * @param url the URL being called
     * @param httpMethod the HTTP method
     * @param startTime the start time of the request
     * @param endTime the end time of the request
     * @param errorType the type of error being thrown
     * @param errorMessage the error message
     */
    void logNetworkError(
            String url,
            String httpMethod,
            long startTime,
            long endTime,
            String errorType,
            String errorMessage);
}
