package io.embrace.android.embracesdk;

import android.app.usage.StorageStatsManager;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.view.WindowManager;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Provides information about the state of the device, retrieved from Android system services,
 * which is used as metadata with telemetry submitted to the Embrace API.
 */
final class EmbraceMetadataService implements MetadataService, ActivityListener {
    /**
     * Default string value for app info missing strings
     */
    private static final String UNKNOWN_VALUE = "UNKNOWN";

    private final TelephonyManager telephonyManager;

    private final WindowManager windowManager;

    private final PackageManager packageManager;

    private final StorageStatsManager storageStatsManager;

    private final ApplicationInfo applicationInfo;

    private final PreferencesService preferencesService;

    private final ActivityService activityService;

    private final BuildInfo buildInfo;

    private final String deviceId;

    private final String packageName;

    private final String appVersionName;

    private final String appVersionCode;

    private final StatFs statFs = new StatFs(Environment.getDataDirectory().getPath());

    /**
     * This field is defined during instantiation as by the end of the startup
     */
    private final boolean appUpdated;

    private final boolean osUpdated;

    private volatile String activeSessionId;

    private EmbraceMetadataService(
            TelephonyManager telephonyManager,
            WindowManager windowManager,
            PackageManager packageManager,
            StorageStatsManager storageStatsManager,
            BuildInfo buildInfo,
            ApplicationInfo applicationInfo,
            String deviceId,
            String packageName,
            String appVersionName,
            String appVersionCode,
            boolean appUpdated,
            boolean osUpdated,
            PreferencesService preferencesService,
            ActivityService activityService) {

        this.telephonyManager = telephonyManager;
        this.windowManager = windowManager;
        this.packageManager = packageManager;
        this.storageStatsManager = storageStatsManager;
        this.buildInfo = Preconditions.checkNotNull(buildInfo);
        this.applicationInfo = applicationInfo;
        this.deviceId = Preconditions.checkNotNull(deviceId);
        this.packageName = Preconditions.checkNotNull(packageName);
        this.appVersionName = Preconditions.checkNotNull(appVersionName);
        this.appVersionCode = Preconditions.checkNotNull(appVersionCode);
        this.appUpdated = appUpdated;
        this.osUpdated = osUpdated;
        this.preferencesService = Preconditions.checkNotNull(preferencesService);
        this.activityService = Preconditions.checkNotNull(activityService);
        activityService.addListener(this);
    }

    /**
     * Creates an instance of the {@link EmbraceMetadataService} from the device's {@link Context}
     * for creating Android system services.
     *
     * @param context            the {@link Context}
     * @param buildInfo          the build information
     * @param preferencesService the preferences service
     * @return an instance
     */
    static EmbraceMetadataService ofContext(
            Context context,
            BuildInfo buildInfo,
            PreferencesService preferencesService,
            ActivityService activityService) {

        Preconditions.checkNotNull(context, "Device context is null");

        StorageStatsManager storageStatsManager = null;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            storageStatsManager = (StorageStatsManager) context.getSystemService(Context.STORAGE_STATS_SERVICE);
        }

        PackageInfo packageInfo;
        String appVersionName;
        String appVersionCode;
        try {
            packageInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            appVersionName = String.valueOf(packageInfo.versionName);
            appVersionCode = String.valueOf(packageInfo.versionCode);
        } catch (Exception e) {
            appVersionName = UNKNOWN_VALUE;
            appVersionCode = UNKNOWN_VALUE;
        }

        Optional<String> lastKnownAppVersion = preferencesService.getAppVersion();
        boolean isAppUpdated = lastKnownAppVersion.isPresent()
                && !lastKnownAppVersion.get().equalsIgnoreCase(appVersionName);

        Optional<String> lastKnownOsVersion = preferencesService.getOsVersion();
        boolean isOsUpdated = lastKnownOsVersion.isPresent()
                && !lastKnownOsVersion.get().equalsIgnoreCase(String.valueOf(Build.VERSION.RELEASE));

        return new EmbraceMetadataService(
                (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE),
                (WindowManager) context.getSystemService(Context.WINDOW_SERVICE),
                context.getPackageManager(),
                storageStatsManager,
                buildInfo,
                context.getApplicationInfo(),
                MetadataUtils.createDeviceId(context.getContentResolver()),
                context.getPackageName(),
                appVersionName,
                appVersionCode,
                isAppUpdated,
                isOsUpdated,
                preferencesService,
                activityService);
    }

    @Override
    public TelephonyInfo getTelephonyInfo() {
        String mcc = null;
        String mnc = null;
        String carrierName = null;
        String carrierCountry = null;

        // Some devices may not support TelephonyManager as a system service.
        if (telephonyManager != null) {
            try {
                String networkOperator = telephonyManager.getNetworkOperator();
                carrierName = telephonyManager.getNetworkOperatorName();
                carrierCountry = telephonyManager.getNetworkCountryIso();
                if (!TextUtils.isEmpty(networkOperator) && (networkOperator.length() == 5 || networkOperator.length() == 6)) {
                    mcc = networkOperator.substring(0, 3);
                    mnc = networkOperator.substring(3);
                }
            } catch (Exception ex) {
                EmbraceLogger.logWarning("Could not retrieve telephony information", ex);
            }
        }
        return new TelephonyInfo(carrierName, carrierCountry, mcc, mnc);
    }

    @Override
    public Optional<String> getAppVersionCodeForRequest() {
        return MetadataUtils.getVersionCodeForRequest(packageManager, packageName);
    }

    @Override
    public String getDeviceId() {
        return deviceId;
    }

    @Override
    public String getAppVersionCode() {
        return appVersionCode;
    }

    @Override
    public String getAppVersionName() {
        return appVersionName;
    }

    @Override
    public DeviceInfo getDeviceInfo() {
        return DeviceInfo.newBuilder()
                .withManufacturer(MetadataUtils.getDeviceManufacturer())
                .withModel(MetadataUtils.getModel())
                .withArchitecture(MetadataUtils.getArchitecture())
                .withJailbroken(MetadataUtils.isJailbroken())
                .withLocale(MetadataUtils.getLocale())
                .withInternalStorageTotalCapacity(MetadataUtils.getInternalStorageTotalCapacity(statFs))
                .withOperatingSystemType(MetadataUtils.getOperatingSystemType())
                .withOperatingSystemVersion(MetadataUtils.getOperatingSystemVersion())
                .withOperatingSystemVersionCode(MetadataUtils.getOperatingSystemVersionCode())
                .withScreenResolution(MetadataUtils.getScreenResolution(windowManager).orNull())
                .withTimezoneDescription(MetadataUtils.getTimezoneId())
                .withUptime(MetadataUtils.getSystemUptime())
                .build();
    }

    @Override
    public AppInfo getAppInfo() {
        // TODO default these all to UNKNOWN_VALUE
        return AppInfo.newBuilder()
                .withSdkVersion(String.valueOf(BuildConfig.VERSION_NAME))
                .withSdkSimpleVersion(String.valueOf(BuildConfig.VERSION_CODE))
                .withBuildId(buildInfo.getBuildId())
                .withAppVersion(appVersionName)
                .withBundleVersion(appVersionCode)
                .withEnvironment(MetadataUtils.appEnvironment(applicationInfo))
                .withAppUpdated(appUpdated)
                .withOsUpdated(osUpdated)
                .withAppUpdatedThisLaunch(appUpdated)
                .withOsUpdatedThisLaunch(osUpdated)
                .build();
    }

    @Override
    public boolean isDebug() {
        return MetadataUtils.isDebug(applicationInfo);
    }

    @Override
    public String getAppId() {
        return buildInfo.getAppId();
    }

    @Override
    public boolean isAppUpdated() {
        return appUpdated;
    }

    @Override
    public boolean isOsUpdated() {
        return osUpdated;
    }

    @Override
    public void setActiveSessionId(String sessionId) {
        this.activeSessionId = sessionId;
    }

    @Override
    public Optional<String> getActiveSessionId() {
        return Optional.fromNullable(activeSessionId);
    }

    @Override
    public String getAppState() {
        if (activityService.isInBackground()) {
            return "background";
        } else {
            return "active";
        }
    }

    @Override
    public DiskUsage getDiskUsage() {
        long free = MetadataUtils.getInternalStorageFreeCapacity(statFs);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            Optional<Long> usage = MetadataUtils.getDeviceDiskAppUsage(
                    storageStatsManager,
                    packageManager,
                    packageName);
            if (usage.isPresent()) {
                return new DiskUsage(usage.get(), free);
            }
        }
        return new DiskUsage(free);
    }

    @Override
    public void applicationStartupComplete() {
        String appVersion = getAppVersionName();
        String osVersion = String.valueOf(Build.VERSION.RELEASE);
        String deviceId = getDeviceId();
        long installDate = System.currentTimeMillis();
        String log = String.format("Setting metadata on preferences service. " +
                        "App version: {}, OS version {}, device ID: {}, install date: {}",
                appVersion,
                osVersion,
                deviceId,
                installDate);
        EmbraceLogger.logDebug(log);
        preferencesService.setAppVersion(appVersion);
        preferencesService.setOsVersion(osVersion);
        preferencesService.setDeviceIdentifier(deviceId);
        if (!preferencesService.getInstallDate().isPresent()) {
            preferencesService.setInstallDate(installDate);
        }
    }
}
