package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.optional.Optional;

/**
 * Intercepts uncaught Java exceptions and forwards them to the Embrace API.
 */
final class EmbraceCrashService implements CrashService {
    private static final String CRASH_REPORT_EVENT_NAME = "_crash_report";

    private final SessionService sessionService;

    private final MetadataService metadataService;

    private final ApiClient apiClient;

    private final UserService userService;

    EmbraceCrashService(
            SessionService sessionService,
            MetadataService metadataService,
            ApiClient apiClient,
            UserService userService) {

        registerExceptionHandler();
        this.sessionService = Preconditions.checkNotNull(sessionService);
        this.metadataService = Preconditions.checkNotNull(metadataService);
        this.apiClient = Preconditions.checkNotNull(apiClient);
        this.userService = Preconditions.checkNotNull(userService);
    }

    /**
     * Handles a crash caught by the {@link EmbraceUncaughtExceptionHandler} by constructing a
     * JSON message containing a description of the crash, device, and context, and then sending
     * it to the Embrace API.
     *
     * @param thread    the crashing thread
     * @param exception the exception thrown by the thread
     */
    @Override
    public void handleCrash(Thread thread, Throwable exception) {

        Event.Builder builder = Event.newBuilder();
        Crash crash = Crash.ofThrowable(exception);
        builder = builder.withName(CRASH_REPORT_EVENT_NAME)
                .withType(EmbraceEvent.Type.CRASH)
                .withTimestamp(System.currentTimeMillis())
                .withAppState(metadataService.getAppState());
        Optional<String> optionalSessionId = metadataService.getActiveSessionId();
        if (optionalSessionId.isPresent()) {
            builder = builder.withSessionId(optionalSessionId.get());
        }
        EventMessage.Builder versionedEventBuilder = EventMessage.newBuilder()
                .withAppInfo(metadataService.getAppInfo())
                .withDeviceInfo(metadataService.getDeviceInfo())
                .withUserInfo(userService.getUserInfo())
                .withCrash(crash)
                .withEvent(builder.build());
        try {
            apiClient.sendEvent(versionedEventBuilder.build()).get();
        } catch (Exception ex) {
            EmbraceLogger.logWarning("Failed to report crash to the api", ex);
        }
        sessionService.handleCrash();
    }

    /**
     * Registers the Embrace {@link java.lang.Thread.UncaughtExceptionHandler} to intercept uncaught
     * exceptions and forward them to the Embrace API as crashes.
     */
    private void registerExceptionHandler() {
        Thread.UncaughtExceptionHandler defaultHandler = Thread.getDefaultUncaughtExceptionHandler();
        EmbraceUncaughtExceptionHandler embraceHandler =
                new EmbraceUncaughtExceptionHandler(defaultHandler, this);
        Thread.setDefaultUncaughtExceptionHandler(embraceHandler);
    }
}
