package io.embrace.android.embracesdk;

import android.content.Context;
import android.content.res.AssetManager;

import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;
import com.google.gson.stream.JsonReader;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

/**
 * Specifies the application ID and build ID.
 */
final class BuildInfo {

    /**
     * Build Info json file name to be retrieved from the assets.
     */
    private static final String FILE_BUILD_INFO = "build-info.json";

    /** The Embrace app ID. This is used to identify the app within the database. */
    @SerializedName("app-id")
    private final String appId;

    /** The ID of the particular build, generated at compile-time. */
    @SerializedName("build-id")
    private final String buildId;

    BuildInfo(String appId, String buildId) {
        this.appId = appId;
        this.buildId = buildId;
    }

    /**
     * Loads the build information from a JSON file packaged within the application by Gradle at
     * build-time.
     *
     * @return the build information
     */
    static BuildInfo fromFile(Context context) {

        InputStream buildInfoStream = null;
        InputStreamReader inputStreamReader = null;
        JsonReader reader = null;
        try {
            AssetManager assets = context.getApplicationContext().getAssets();
            buildInfoStream = assets.open(FILE_BUILD_INFO);
            inputStreamReader = new InputStreamReader(buildInfoStream);
            reader = new JsonReader(inputStreamReader);
            return new Gson().fromJson(reader, BuildInfo.class);
        } catch (Exception ex) {
            throw new RuntimeException("Failed to read build-info.json", ex);
        } finally {
            try {
                if (buildInfoStream != null) {
                    buildInfoStream.close();
                }
                if (reader != null) {
                    reader.close();
                }
                if (inputStreamReader != null) {
                    inputStreamReader.close();
                }
            } catch (IOException ex) {
                EmbraceLogger.logWarning("Failed to close build-info.json", ex);
            }
        }
    }

    String getAppId() {
        return appId;
    }

    String getBuildId() {
        return buildId;
    }
}
