package io.embrace.android.embracesdk;

import android.app.Application;
import android.content.Context;
import android.util.Pair;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import io.embrace.android.embracesdk.network.http.HttpMethod;

import static io.embrace.android.embracesdk.EmbraceEventService.STARTUP_EVENT_NAME;

/**
 * Entrypoint for the SDK. This class is part of the Embrace Public API.
 * <p>
 * Contains a singleton instance of itself, and is used for initializing the SDK.
 */
public final class Embrace {
    /**
     * Singleton instance of the Embrace SDK.
     */
    private static final Embrace embrace = new Embrace();

    /**
     * The application being instrumented by the SDK.
     */
    private volatile Application application;

    /**
     * The object caching service.
     */
    private volatile CacheService cacheService;

    /**
     * The crash handling service.
     */
    private volatile CrashService crashService;

    /**
     * The breadcrumbs service.
     */
    private volatile BreadcrumbService breadcrumbService;

    /**
     * The session handling service.
     */
    private volatile EmbraceSessionService sessionService;

    /**
     * The device metadata service.
     */
    private volatile MetadataService metadataService;

    /**
     * The device's performance information service.
     */
    private volatile PerformanceInfoService performanceInfoService;
    /**
     * The power service.
     */
    private volatile PowerService powerService;
    /**
     * The memory service;
     */
    private volatile MemoryService memoryService;
    /**
     * The Activity service.
     */
    private volatile EmbraceActivityService activityService;
    /**
     * The Networking service.
     */
    private volatile NetworkService networkService;
    /**
     * The CPU service.
     */
    private volatile CpuService cpuService;
    /**
     * The Build info data accessor.
     */
    private volatile BuildInfo buildInfo;
    /**
     * The ANR Monitoring service.
     */
    private volatile AnrService anrService;
    /**
     * The EmbraceRemoteLogger.
     */
    private volatile EmbraceRemoteLogger remoteLogger;
    /**
     * The Configuration service.
     */
    private volatile ConfigService configService;
    /**
     * The Api Client.
     */
    private volatile ApiClient apiClient;
    /**
     * The Embrace prefences service.
     */
    private volatile PreferencesService preferencesService;
    /**
     * The Embrace screenshot service.
     */
    private volatile ScreenshotService screenshotService;
    /**
     * The Embrace event service.
     */
    private volatile EventService eventService;
    /**
     * The User service.
     */
    private volatile UserService userService;

    /**
     * The Embrace signal quality service.
     */
    private volatile SignalQualityService signalQualityService;

    /**
     * The Embrace connection class service.
     */
    private volatile ConnectionClassService connectionClassService;

    /**
     * Whether the Embrace SDK has been started yet.
     */
    private volatile AtomicBoolean started = new AtomicBoolean(false);

    /**
     * Gets the singleton instance of the Embrace SDK.
     *
     * @return the instance of the Embrace SDK
     */
    public static Embrace getInstance() {
        return embrace;
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param application an instance of the application to instrument
     */
    public void start(Application application) {
        start(application, true);
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param application an instance of the application to instrument
     * @param enableIntegrationTesting if true, debug sessions (those which are not part of a
     *                                 release APK) will go to the live integration testing tab
     *                                 of the dashboard. If false, they will appear in 'recent
     *                                 sessions'.
     */
    public void start(Application application, boolean enableIntegrationTesting) {
        Preconditions.checkState(this.application == null, "Embrace SDK has already been initialized");
        this.application = Preconditions.checkNotNull(application, "application must not be null");
        Context context = this.application.getApplicationContext();

        // Bring up all services and inject dependencies
        try {
            /** ------------------------------------------------------------------------------------
             *  Device instrumentation (power, memory, CPU, network, preferences, CPU)            */
            this.powerService = new EmbracePowerService(context);
            this.memoryService = EmbraceMemoryService.ofContext(context);
            this.activityService =
                    new EmbraceActivityService(application, memoryService);
            this.preferencesService = new EmbracePreferencesService(context, activityService);
            this.connectionClassService = new EmbraceConnectionClassService();
            this.networkService = new EmbraceNetworkService(context, connectionClassService);
            this.cpuService = new EmbraceCpuService();
            this.anrService = new EmbraceAnrService();
            this.buildInfo = BuildInfo.fromFile(context);
            this.metadataService = EmbraceMetadataService.ofContext(
                    context,
                    buildInfo,
                    preferencesService,
                    activityService);

            /** ------------------------------------------------------------------------------------
             *  API services                                                                      */
            this.cacheService = new EmbraceCacheService(context);
            this.apiClient = new ApiClient(metadataService, cacheService, enableIntegrationTesting);
            this.configService = new EmbraceConfigService(
                    apiClient,
                    activityService,
                    cacheService,
                    metadataService);
            this.signalQualityService =
                    new EmbraceSignalQualityService(context, configService, activityService);
            this.breadcrumbService = new EmbraceBreadcrumbService(configService, activityService);
            if (configService.isSdkDisabled()) {
                stop();
                return;
            }
            this.userService = new EmbraceUserService(
                    activityService,
                    configService,
                    preferencesService,
                    apiClient);
            this.screenshotService = new EmbraceScreenshotService(
                    activityService,
                    configService,
                    apiClient);
            this.performanceInfoService = new EmbracePerformanceInfoService(
                    anrService,
                    networkService,
                    powerService,
                    cpuService,
                    memoryService,
                    signalQualityService,
                    connectionClassService,
                    metadataService);
            this.eventService = new EmbraceEventService(
                    apiClient,
                    configService,
                    metadataService,
                    performanceInfoService,
                    userService,
                    screenshotService,
                    activityService);
            this.remoteLogger = new EmbraceRemoteLogger(
                    metadataService,
                    screenshotService,
                    apiClient,
                    userService,
                    configService);
            this.sessionService = new EmbraceSessionService(
                    preferencesService,
                    performanceInfoService,
                    metadataService,
                    breadcrumbService,
                    powerService,
                    activityService,
                    apiClient,
                    eventService,
                    remoteLogger,
                    userService,
                    configService,
                    cacheService);
            this.crashService = new EmbraceCrashService(
                    sessionService,
                    metadataService,
                    apiClient,
                    userService);
        } catch (Exception ex) {
            EmbraceLogger.logError("Exception occurred whilst initializing the Embrace SDK. Instrumentation may be disabled.", ex);
        }

        // Intercept Android network calls
        StreamHandlerFactoryInstaller.registerFactory();
        started.set(true);
    }

    /**
     * Whether or not the SDK has been started.
     *
     * @return true if the SDK is started, false otherwise
     */
    public boolean isStarted() {
        return started.get();
    }

    /**
     * Shuts down the Embrace SDK.
     */
    void stop() {
        if (started.compareAndSet(true, false)) {
            EmbraceLogger.logInfo("Shutting down Embrace SDK.");
            try {
                // Keep the config service active to give the application the opportunity to
                // update the disabled flag, in case the SDK has been re-enabled.

                if (this.anrService != null) {
                    this.anrService.close();
                }

                if (this.powerService != null) {
                    this.powerService.close();
                }

                if (this.memoryService != null) {
                    this.memoryService.close();
                }

                if (this.cpuService != null) {
                    this.cpuService.close();
                }

                if (this.anrService != null) {
                    this.anrService.close();
                }

                if (this.activityService != null) {
                    this.activityService.close();
                }

                if (this.sessionService != null) {
                    this.sessionService.close();
                }

                if (this.eventService != null) {
                    this.eventService.close();
                }

                if (this.networkService != null) {
                    this.networkService.close();
                }

                if (this.signalQualityService != null) {
                    this.signalQualityService.close();
                }

                this.powerService = null;
                this.memoryService = null;
                this.breadcrumbService = null;
                this.activityService = null;
                this.preferencesService = null;
                this.networkService = null;
                this.cpuService = null;
                this.anrService = null;
                this.buildInfo = null;
                this.metadataService = null;
                this.performanceInfoService = null;
                this.cacheService = null;
                this.userService = null;
                this.screenshotService = null;
                this.eventService = null;
                this.remoteLogger = null;
                this.sessionService = null;
                this.crashService = null;
                this.signalQualityService = null;

                this.application = null;
            } catch (Exception ex) {
                EmbraceLogger.logError("Error whilst shutting down Embrace SDK", ex);
            }
        }
    }

    /**
     * This method sets a logging level, but this logging level is never used.
     *
     * @param severity the severity
     * @deprecated as the log level is never used. Use {@link EmbraceLogger}.
     */
    @Deprecated
    public void setLogLevel(EmbraceLogger.Severity severity) {
        // This method does not do anything and is purely retained for backwards-compatibility
    }

    /**
     * Sets the user ID. This would typically be some form of unique identifier such as a UUID or
     * database key for the user.
     *
     * @param userId the unique identifier for the user
     */
    public void setUserIdentifier(String userId) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring identifier update.");
                return;
            }
            userService.setUserIdentifier(userId);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot set user identifier");
        }
    }

    /**
     * Clears the currently set user ID. For example, if the user logs out.
     */
    public void clearUserIdentifier() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring identifier update.");
                return;
            }
            userService.clearUserIdentifier();
        }
    }

    /**
     * Sets the current user's email address.
     *
     * @param email the email address of the current user
     */
    public void setUserEmail(String email) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring email update.");
                return;
            }
            userService.setUserEmail(email);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear user identifier");
        }
    }

    /**
     * Clears the currently set user's email address.
     */
    public void clearUserEmail() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring email update.");
                return;
            }
            userService.clearUserEmail();
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear user email");
        }
    }

    /**
     * Sets this user as a paying user. This adds a persona to the user's identity.
     */
    public void setUserAsPayer() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring payer user update.");
                return;
            }
            userService.setUserAsPayer();
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot set user as payer");
        }
    }

    /**
     * Clears this user as a paying user. This would typically be called if a user is no longer
     * paying for the service and has reverted back to a basic user.
     */
    public void clearUserAsPayer() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring payer user update.");
                return;
            }
            userService.clearUserAsPayer();
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear user as payer");
        }
    }

    /**
     * Sets a custom user persona. A persona is a trait associated with a given user.
     *
     * @param persona the persona to set
     */
    public void setUserPersona(String persona) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring user persona update.");
                return;
            }
            userService.setUserPersona(persona);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot set user persona");
        }
    }

    /**
     * Clears the custom user persona, if it is set.
     *
     * @param persona the persona to clear
     */
    public void clearUserPersona(String persona) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring user persona update.");
                return;
            }
            userService.clearUserPersona(persona);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear user persona");
        }
    }

    /**
     * Sets the username of the currently logged in user.
     *
     * @param username the username to set
     */
    public void setUsername(String username) {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring username update.");
                return;
            }
            userService.setUsername(username);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot set username");
        }
    }

    /**
     * Clears the username of the currently logged in user, for example if the user has logged out.
     */
    public void clearUsername() {
        if (isStarted()) {
            if (configService.isMessageTypeDisabled(MessageType.USER)) {
                EmbraceLogger.logWarning("User updates are disabled, ignoring username update.");
                return;
            }
            userService.clearUsername();
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot clear username");
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name a name identifying the event
     */
    public void startEvent(String name) {
        if (isStarted()) {
            eventService.startEvent(name);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log event");
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name       a name identifying the event
     * @param identifier an identifier distinguishing between multiple events with the same name
     */
    public void startEvent(String name, String identifier) {
        if (isStarted()) {
            eventService.startEvent(name, identifier);
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name            a name identifying the event
     * @param identifier      an identifier distinguishing between multiple events with the same name
     * @param allowScreenshot true if a screenshot should be taken for a late event, false otherwise
     */
    public void startEvent(String name, String identifier, boolean allowScreenshot) {
        if (isStarted()) {
            eventService.startEvent(name, identifier, allowScreenshot);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log event");
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name            a name identifying the event
     * @param identifier      an identifier distinguishing between multiple events with the same name
     * @param allowScreenshot true if a screenshot should be taken for a late event, false otherwise
     * @param properties      custom key-value pairs to provide with the event
     */
    public void startEvent(
            String name,
            String identifier,
            boolean allowScreenshot,
            Map<String, Object> properties) {

        if (isStarted()) {
            Map<String, Object> normalizedProperties = new HashMap<>();
            if (properties != null) {
                normalizedProperties = PropertyUtils.sanitizeProperties(properties);
            }
            eventService.startEvent(name, identifier, allowScreenshot, normalizedProperties);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log event");
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name the name of the event to end
     */
    public void endEvent(String name) {
        if (isStarted()) {
            eventService.endEvent(name);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log event");
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param identifier the identifier of the event to end, distinguishing between events with the same name
     */
    public void endEvent(String name, String identifier) {
        if (isStarted()) {
            eventService.endEvent(name, identifier);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log event");
        }
    }

    public void endAppStartup() {
        endEvent(STARTUP_EVENT_NAME);
    }

    /**
     * Logs the fact that a network call occurred. These are recorded and sent to Embrace as part
     * of a particular session.
     *
     * @param url           the URL of the network call
     * @param httpMethod    the HTTP method of the network call
     * @param statusCode    the status code returned by the server
     * @param startTime     the time that the network call started
     * @param endTime       the time that the network call was completed
     * @param bytesSent     the number of bytes sent as part of the network call
     * @param bytesReceived the number of bytes returned by the server in response to the network call
     */
    public void logNetworkCall(
            String url,
            HttpMethod httpMethod,
            int statusCode,
            long startTime,
            long endTime,
            long bytesSent,
            long bytesReceived) {

        if (isStarted()) {
            if (configService.isUrlDisabled(url)) {
                EmbraceLogger.logWarning("Recording of network calls disabled for url: " + url);
                return;
            }
            networkService.logNetworkCall(
                    url,
                    httpMethod.name(),
                    statusCode,
                    startTime,
                    endTime,
                    bytesSent,
                    bytesReceived);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log network call");
        }
    }

    /**
     * Logs the fact that an exception was thrown when attempting to make a network call.
     * <p>
     * These are client-side exceptions and not server-side exceptions, such as a DNS error or
     * failure to connect to the remote server.
     *
     * @param url          the URL of the network call
     * @param httpMethod   the HTTP method of the network call
     * @param startTime    the time that the network call started
     * @param endTime      the time that the network call was completed
     * @param errorType    the type of the exception
     * @param errorMessage the message returned by the exception
     */
    public void logNetworkClientError(
            String url,
            HttpMethod httpMethod,
            long startTime,
            long endTime,
            String errorType,
            String errorMessage) {

        if (isStarted()) {
            if (configService.isUrlDisabled(url)) {
                EmbraceLogger.logWarning("Recording of network calls disabled for url: " + url);
                return;
            }
            networkService.logNetworkError(
                    url,
                    httpMethod.name(),
                    startTime,
                    endTime,
                    errorType,
                    errorMessage);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log network error");
        }
    }

    /**
     * Remotely logs a message at INFO level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message the message to remotely log
     */
    public void logInfo(String message) {
        logInfo(message, null);
    }

    /**
     * Remotely logs a message at INFO level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logInfo(String message, Map<String, Object> properties) {
        logMessage(EmbraceEvent.Type.INFO_LOG, message, properties, false);
    }

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message the message to remotely log
     */
    public void logWarning(String message) {
        logWarning(message, null);
    }

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logWarning(String message, Map<String, Object> properties) {
        logMessage(EmbraceEvent.Type.WARNING_LOG, message, properties, false);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message the message to remotely log
     */
    public void logError(String message) {
        logError(message, null);
    }


    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    public void logError(String message, Map<String, Object> properties) {
        logMessage(EmbraceEvent.Type.ERROR_LOG, message, properties, true);
    }

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message         the message to remotely log
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    public void logError(String message, Map<String, Object> properties, boolean allowScreenshot) {
        logMessage(EmbraceEvent.Type.ERROR_LOG, message, properties, allowScreenshot);
    }

    void logMessage(
            EmbraceEvent.Type type,
            String message,
            Map<String, Object> properties,
            boolean allowScreenshot) {

        if (isStarted()) {
            try {
                this.remoteLogger.log(message, type, allowScreenshot, properties);
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to log message using Embrace SDK.", ex);
            }
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log message.");
        }
    }

    /**
     * Logs a breadcrumb.
     * <p>
     * Breadcrumbs track a user's journey through the application and will be shown on the timeline.
     *
     * @param message the name of the breadcrumb to log
     */
    public void logBreadcrumb(String message) {
        if (isStarted()) {
            this.breadcrumbService.logCustom(message, System.currentTimeMillis());
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log breadcrumb.");
        }
    }

    /**
     * Registers a {@link ConnectionQualityListener}, notifying the listener each time that there is
     * a change in the connection quality.
     *
     * @param listener the listener to register
     */
    public void addConnectionQualityListener(ConnectionQualityListener listener) {
        if (isStarted()) {
            try {
                connectionClassService.addListener(listener);
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to add connection quality listener", ex);
            }
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot add listener.");

        }
    }

    /**
     * Removes a registered {@link ConnectionQualityListener}, suspending connection quality
     * notifications.
     *
     * @param listener the listener to remove
     */
    public void removeConnectionQualityListener(ConnectionQualityListener listener) {
        if (isStarted()) {
            try {
                connectionClassService.removeListener(listener);
            } catch (Exception ex) {
                EmbraceLogger.logError("Failed to remove connection quality listener", ex);
            }
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot remove listener.");

        }
    }

    /**
     * Logs the fact that a particular view was entered.
     * <p>
     * If the previously logged view has the same name, a duplicate view breadcrumb will not be
     * logged.
     *
     * @param screen the name of the view to log
     */
    void logView(String screen) {
        if (isStarted()) {
            this.breadcrumbService.logView(screen, System.currentTimeMillis());
        }
    }

    /**
     * Logs the fact that a particular view was entered.
     * <p>
     * If the previously logged view has the same name, a duplicate view breadcrumb will be
     * logged, and not treated as a duplicate.
     *
     * @param screen the name of the view to log
     */
    void forceLogView(String screen) {
        if (isStarted()) {
            this.breadcrumbService.forceLogView(screen, System.currentTimeMillis());
        }
    }

    /**
     * Logs a tap on a screen element.
     *
     * @param point       the coordinates of the screen tap
     * @param elementName the name of the element which was tapped
     * @param type        the type of tap that occurred
     */
    void logTap(Pair<Float, Float> point, String elementName, TapBreadcrumb.TapBreadcrumbType type) {
        if (isStarted()) {
            this.breadcrumbService.logTap(point, elementName, System.currentTimeMillis(), type);
        }
    }

    EventService getEventService() {
        return eventService;
    }

    EmbraceRemoteLogger getRemoteLogger() {
        return remoteLogger;
    }


}