package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import java9.util.Lists;

/**
 * Describes a particular Java exception. Where an exception has a cause, there will be an
 * {@link ExceptionInfo} for each nested cause.
 */
final class ExceptionInfo {
    /** The name of the class throwing the exception. */
    @SerializedName("n")
    private final String name;

    /** The exception message. */
    @SerializedName("m")
    private final String message;

    /** String representation of each line of the stack trace. */
    @SerializedName("tt")
    private final List<String> lines;

    ExceptionInfo(String name, String message, List<String> lines) {
        this.name = name;
        this.message = message;
        this.lines = Lists.copyOf(lines);
    }

    String getName() {
        return name;
    }

    String getMessage() {
        return message;
    }

    List<String> getLines() {
        return lines;
    }

    /**
     * Creates a {@link ExceptionInfo} from a {@link Throwable}, using the classname as the nane, the
     * exception message as the message, and each stacktrace element as each line.
     *
     * @param throwable the exception
     * @return the stacktrace instance
     */
    static ExceptionInfo ofThrowable(Throwable throwable) {
        String name = "";
        String message = "";
        List<String> lines = new ArrayList<>();

        if (throwable.getClass() != null && throwable.getClass().getName() != null) {
            name = throwable.getClass().getName();
        }

        if (throwable.getMessage() != null) {
            message = throwable.getMessage();
        }

        for (StackTraceElement line : throwable.getStackTrace()) {
            lines.add(line.toString());
        }

        return new ExceptionInfo(name, message, lines);
    }
}
