package io.embrace.android.embracesdk;

import java.util.HashSet;
import java.util.Set;

import java9.util.Sets;

class EmbraceUserService implements ActivityListener, UserService {
    private final PreferencesService preferencesService;

    private final ConfigService configService;

    private final ApiClient apiClient;

    private volatile UserInfo userInfo = UserInfo.ofEmpty();

    EmbraceUserService(
            ActivityService activityService,
            ConfigService configService,
            PreferencesService preferencesService,
            ApiClient apiClient) {

        this.configService = configService;
        this.preferencesService = preferencesService;
        this.apiClient = apiClient;
        activityService.addListener(this);
    }

    @Override
    public void applicationStartupComplete() {
        this.userInfo = UserInfo.ofStored(preferencesService, configService);
        apiClient.sendUser(userInfo);
    }

    @Override
    public UserInfo getUserInfo() {
        return userInfo;
    }

    @Override
    public void setUserIdentifier(String userId) {
        UserInfo newUserInfo = UserInfo.newBuilder(userInfo)
                .withUserId(userId)
                .build();
        this.userInfo = newUserInfo;
        preferencesService.setUserIdentifier(userId);
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void clearUserIdentifier() {
        setUserIdentifier(null);
    }

    @Override
    public void setUserEmail(String email) {
        UserInfo newUserInfo = UserInfo.newBuilder(userInfo)
                .withEmail(email)
                .build();
        this.userInfo = newUserInfo;
        preferencesService.setUserEmailAddress(email);
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void clearUserEmail() {
        setUserEmail(null);
    }

    @Override
    public void setUserAsPayer() {
        setUserPersona(UserInfo.PERSONA_PAYER);
    }

    @Override
    public void clearUserAsPayer() {
        clearUserPersona(UserInfo.PERSONA_PAYER);
    }

    @Override
    public void setUserPersona(String persona) {
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (builder.getPersonas() != null) {
            personas.addAll(builder.getPersonas());
        }
        UserInfo newUserInfo = builder.build();
        personas.add(persona);
        this.userInfo = newUserInfo;
        preferencesService.setUserPersonas(personas);
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void clearUserPersona(String persona) {
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (builder.getPersonas() != null) {
            personas.addAll(builder.getPersonas());
        }
        UserInfo newUserInfo = builder.build();
        this.userInfo = newUserInfo;
        personas.remove(persona);
        preferencesService.setUserPersonas(personas);
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void setUsername(String username) {
        UserInfo newUserInfo = UserInfo.newBuilder(userInfo)
                .withUsername(username)
                .build();
        this.userInfo = newUserInfo;
        preferencesService.setUsername(username);
        apiClient.sendUser(newUserInfo);
    }

    @Override
    public void clearUsername() {
        setUsername(null);
    }

}
