package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;
import com.fernandocejas.arrow.strings.Charsets;
import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Locale;
import java.util.concurrent.Future;
import java.util.zip.GZIPOutputStream;

import io.embrace.android.embracesdk.network.http.HttpMethod;
import io.embrace.android.embracesdk.utils.exceptions.Unchecked;
import java9.util.function.BiConsumer;

/**
 * Client for calling the Embrace API.
 */
class ApiClient implements Closeable {

    /** The version of the API message format. */
    static final int MESSAGE_VERSION = 13;

    private static final int CONFIG_API_VERSION = 2;

    private static final int API_VERSION = 1;

    private static final String DATA_DEV_PREFIX = "https://data-dev.emb-api.com";
    private static final String DATA_PROD_PREFIX = "https://data.emb-api.com";
    private static final String SCREENSHOT_URL_FORMAT = "%s/%s/%s/%s.jpg";

    private static final String URL_FORMAT = "%s/v%s/%s";

    private static final String CONFIG_URL_FORMAT = "%s?appId=%s&osVersion=%s&appVersion=%s&deviceId=%s";

    private static final String STORY_ID_HEADER_FORMAT = "%s:%s"; // [event type]:[story id]

    private static final String SCREENSHOT_URL = String.format(URL_FORMAT, "https://images.emb-api.com", API_VERSION, "screenshot");

    private final String dataUrlPrefix;

    private final String appId;

    private final String deviceId;

    private final String operatingSystemCode = String.format(Locale.US, "%d.0.0", MetadataUtils.getOperatingSystemVersionCode());

    private final String appVersion;

    private final BackgroundWorker worker = BackgroundWorker.ofSingleThread("API Client");

    private final Gson gson = new Gson();

    private final ScheduledWorker retryWorker = ScheduledWorker.ofSingleThread("API Retry");

    private final ApiClientRetryWorker<SessionMessage> sessionRetryWorker;

    private final ApiClientRetryWorker<EventMessage> eventRetryWorker;

    private final ApiClientRetryWorker<UserInfo> userInfoRetryWorker;


    ApiClient(MetadataService metadataService, CacheService cacheService) {
        Preconditions.checkNotNull(metadataService, "metadataService must not be null");
        dataUrlPrefix = metadataService.isDebug() ? DATA_DEV_PREFIX : DATA_PROD_PREFIX;
        appId = metadataService.getAppId();
        appVersion = metadataService.getAppVersionCodeForRequest().or("");
        deviceId = metadataService.getDeviceId();
        sessionRetryWorker =
                new ApiClientRetryWorker<>(SessionMessage.class, cacheService, this, retryWorker);
        eventRetryWorker =
                new ApiClientRetryWorker<>(EventMessage.class, cacheService, this, retryWorker);
        userInfoRetryWorker =
                new ApiClientRetryWorker<>(UserInfo.class, cacheService, this, retryWorker);

    }


    /**
     * Asynchronously gets the app's SDK configuration.
     * <p>
     * These settings define app-specific settings, such as disabled log patterns, whether
     * screenshots are enabled, as well as limits and thresholds.
     *
     * @return a future containing the configuration.
     */
    Future<Config> getConfig() {
        String base = String.format(URL_FORMAT, "https://config.emb-api.com", CONFIG_API_VERSION, "config");
        String url = String.format(CONFIG_URL_FORMAT, base, appId, operatingSystemCode, appVersion, deviceId);
        final HttpURLConnection connection = ApiRequest.newBuilder()
                .withHttpMethod(HttpMethod.GET)
                .withUrl(Unchecked.wrap(() -> new URL(url)))
                .build()
                .toConection();
        return worker.submit(() -> {
            connection.connect();
            return httpCall(connection, Config.class);
        });

    }

    /**
     * Asynchronously sends a screenshot corresponding to a log entry.
     *
     * @param screenshot the screenshot payload
     * @param logId the ID of the corresponding log
     * @return a future containing the path to the screenshot on the remote server
     */
    Future<String> sendLogScreenshot(byte[] screenshot, String logId) {
        URL url = Unchecked.wrap(() ->
                new URL(String.format(SCREENSHOT_URL_FORMAT, SCREENSHOT_URL, appId, "logs", logId)));
        ApiRequest request = screenshotBuilder(url)
                .withLogId(logId)
                .build();
        return rawPost(request, screenshot);
    }

    /**
     * Sends a screenshot corresponding to a moment.
     *
     * @param screenshot the screenshot payload
     * @param storyId the ID of the moment
     * @return a future containing the path to the screenshot on the remote server
     */
    Future<String> sendMomentScreenshot(byte[] screenshot, String storyId) {
        URL url = Unchecked.wrap(() ->
                new URL(String.format(SCREENSHOT_URL_FORMAT, SCREENSHOT_URL, appId, "moments", storyId)));
        ApiRequest request = screenshotBuilder(url)
                .withStoryId(storyId)
                .build();
        return rawPost(request, screenshot);
    }

    /**
     * Sends a session message to the API.
     *
     * @param sessionMessage the session to send
     * @return a future containing the response body from the server
     */
    Future<String> sendSession(SessionMessage sessionMessage) {
        URL url = Unchecked.wrap(() -> new URL(String.format(
                URL_FORMAT,
                dataUrlPrefix,
                API_VERSION,
                "log/sessions")));
        ApiRequest request = eventBuilder(url)
                .withDeviceId(deviceId)
                .withAppId(appId)
                .withUrl(url)
                .withHttpMethod(HttpMethod.POST)
                .withContentEncoding("gzip")
                .build();
        return jsonPost(request, sessionMessage, SessionMessage.class, sessionRetryWorker::retryCall);
    }

    /**
     * Sends a log message to the API.
     *
     * @param eventMessage the event message containing the log entry
     * @return a future containing the response body from the server
     */
    Future<String> sendLogs(EventMessage eventMessage) {
        Preconditions.checkNotNull(eventMessage.getEvent(), "event must be set");
        Event event = eventMessage.getEvent();
        Preconditions.checkNotNull(event.getType(), "event type must be set");
        Preconditions.checkNotNull(event.getStoryId(), "event storyId must be set");
        URL url = Unchecked.wrap(() -> new URL(String.format(
                URL_FORMAT,
                dataUrlPrefix,
                API_VERSION,
                "log/logging")));
        String abbreviation = event.getType().getAbbreviation();
        String storyIdentifier = String.format(STORY_ID_HEADER_FORMAT, abbreviation, event.getStoryId());
        ApiRequest request = eventBuilder(url)
                .withLogId(storyIdentifier)
                .build();
        return sendEvent(eventMessage, request);
    }

    /**
     * Sends an event to the API.
     *
     * @param eventMessage the event message containing the event
     * @return a future containing the response body from the server
     */
    Future<String> sendEvent(EventMessage eventMessage) {
        Preconditions.checkNotNull(eventMessage.getEvent(), "event must be set");
        Event event = eventMessage.getEvent();
        Preconditions.checkNotNull(event.getType(), "event type must be set");
        Preconditions.checkNotNull(event.getStoryId(), "event storyId must be set");
        URL url = Unchecked.wrap(() -> new URL(String.format(
                URL_FORMAT,
                dataUrlPrefix,
                API_VERSION,
                "log/events")));
        String abbreviation = event.getType().getAbbreviation();
        String storyIdentifier = String.format(
                STORY_ID_HEADER_FORMAT,
                abbreviation,
                event.getStoryId());
        ApiRequest request = eventBuilder(url)
                .withStoryId(storyIdentifier)
                .build();
        return sendEvent(eventMessage, request);
    }

    /**
     * Sends a user message to the API.
     *
     * @param user the user information
     * @return a future containing the response body from the server
     */
    Future<String> sendUser(UserInfo user) {
        URL url = Unchecked.wrap(() -> new URL(String.format(
                URL_FORMAT,
                dataUrlPrefix,
                API_VERSION,
                "log/users")));
        ApiRequest request = eventBuilder(url)
                .withDeviceId(deviceId)
                .withAppId(appId)
                .withUrl(url)
                .withHttpMethod(HttpMethod.POST)
                .withContentEncoding("gzip")
                .build();
        return jsonPost(request, user, UserInfo.class, userInfoRetryWorker::retryCall);
    }

    /**
     * Posts a JSON payload to the API.
     * <p>
     * If the request fails for any reason, such as a non-200 response, or an exception being thrown
     * during the connection, the failer handler will be invoked. This can be used to re-attempt
     * sending the message.
     *
     * @param request the details of the API request
     * @param payload the body of the API request
     * @param clazz the class of the payload
     * @param failureHandler callback if the request fails
     * @param <T> the type of the payload
     * @return a future containing the response body from the server
     */
    <T> Future<String> jsonPost(
            ApiRequest request,
            T payload,
            Class<T> clazz,
            BiConsumer<ApiRequest, T> failureHandler) {

        final byte[] bytes = gson.toJson(payload, clazz).getBytes(Charsets.UTF_8);
        return worker.submit(() -> {
            try {
                return sendBytes(request, gzip(bytes));
            } catch (Exception ex) {
                EmbraceLogger.logWarning("Failed to post Embrace API call. Will retry.", ex);
                failureHandler.accept(request, payload);
                throw Unchecked.propagate(ex);
            }
        });
    }

    private Future<String> sendEvent(EventMessage eventMessage, ApiRequest request) {
        return jsonPost(request, eventMessage, EventMessage.class, eventRetryWorker::retryCall);
    }

    private Future<String> rawPost(ApiRequest request, byte[] payload) {
        return worker.submit(() -> sendBytes(request, payload));
    }

    private String sendBytes(ApiRequest request, byte[] payload) {
        return Unchecked.wrap(() -> {
            HttpURLConnection connection = request.toConection();
            if (payload != null) {
                OutputStream outputStream = connection.getOutputStream();
                outputStream.write(payload);
                connection.connect();
            }
            return httpCall(connection);
        });
    }

    /**
     * Compresses a given byte array using the GZIP compression algorithm.
     *
     * @param bytes the byte array to compress
     * @return the compressed byte array
     */
    private static byte[] gzip(byte[] bytes) {
        ByteArrayOutputStream baos = null;
        GZIPOutputStream os = null;
        try {
            baos = new ByteArrayOutputStream();
            os = new GZIPOutputStream(baos);
            os.write(bytes);
            os.finish();
            return baos.toByteArray();
        } catch (IOException ex) {
            throw Unchecked.propagate(ex);
        } finally {
            try {
                if (baos != null) {
                    baos.close();
                }
                if (os != null) {
                    os.close();
                }
            } catch (IOException ex) {
                EmbraceLogger.logWarning("Failed to close streams when gzipping payload", ex);
            }
        }
    }

    /**
     * Executes a HTTP call using the specified connection, returning the JSON response from the
     * server as an object of the specified type.
     *
     * @param connection the HTTP connection to use
     * @param clazz the class representing the type of response
     * @param <T> the type of the object being returned
     * @return the Java representation of the JSON object
     */
    private <T> T httpCall(HttpURLConnection connection, Class<T> clazz) {
        handleHttpResponseCode(connection);
        return readJsonObject(connection, clazz);
    }

    /**
     * Executes a HTTP call using the specified connection, returning the response from the server
     * as a string.
     *
     * @param connection the HTTP connection to use
     * @return a string containing the response from the server to the request
     */
    private String httpCall(HttpURLConnection connection) {
        handleHttpResponseCode(connection);
        return Unchecked.wrap(() -> readString(connection.getInputStream()));
    }

    /**
     * Checks that the HTTP response was 200, otherwise throws an exception.
     * <p>
     * Attempts to read the error message payload from the response, if present, and writes it to
     * the logs at warning level.
     *
     * @param connection the connection containing the HTTP response
     */
    private void handleHttpResponseCode(HttpURLConnection connection) {
        Integer responseCode = null;
        try {
            responseCode = connection.getResponseCode();
        } catch (IOException ex) {
            // Connection failed or unexpected response code
        }

        if (responseCode == null || responseCode != HttpURLConnection.HTTP_OK) {
            if (responseCode != null) {
                String errorMessage = readString(connection.getErrorStream());
                EmbraceLogger.logWarning("Embrace API request failed. HTTP response code: " + responseCode + ", message: " + errorMessage);
            }
            throw new IllegalStateException("Failed to retrieve from Embrace server.");
        }
    }

    /**
     * Reads the JSON response payload of a {@link HttpURLConnection} and converts it to a Java
     * object of the specified type.
     *
     * @param connection the connection to read
     * @param clazz the class representing the type
     * @param <T> the type of object to return
     * @return the Java representation of the JSON object
     */
    private <T> T readJsonObject(HttpURLConnection connection, Class<T> clazz) {
        InputStreamReader inputStreamReader = null;
        JsonReader reader = null;
        try {
            inputStreamReader = new InputStreamReader(connection.getInputStream());
            reader = new JsonReader(inputStreamReader);
            return gson.fromJson(reader, clazz);
        } catch (IOException ex) {
            throw Unchecked.propagate(ex);
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
                if (inputStreamReader != null) {
                    inputStreamReader.close();
                }
            } catch (IOException ex) {
                EmbraceLogger.logWarning("Failed to close HTTP reader", ex);
            }
        }
    }

    /**
     * Reads an {@link InputStream} into a String.
     *
     * @param inputStream the input stream to read
     * @return the string representation
     */
    private String readString(InputStream inputStream) {
        InputStreamReader inputStreamReader = null;
        BufferedReader reader = null;
        try {
            inputStreamReader = new InputStreamReader(inputStream);
            reader = new BufferedReader(inputStreamReader);
            char[] buffer = new char[4096];
            StringBuilder sb = new StringBuilder();
            for (int len; (len = reader.read(buffer)) > 0;)
                sb.append(buffer, 0, len);
            return sb.toString();
        } catch (IOException ex) {
            throw Unchecked.propagate(ex);
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
                if (inputStreamReader != null) {
                    inputStreamReader.close();
                }
            } catch (IOException ex) {
                EmbraceLogger.logWarning("Failed to close HTTP reader", ex);
            }
        }
    }

    private ApiRequest.Builder screenshotBuilder(URL url) {
        return ApiRequest.newBuilder()
                .withContentType("application/octet-stream")
                .withHttpMethod(HttpMethod.POST)
                .withAppId(appId)
                .withDeviceId(deviceId)
                .withUrl(url);
    }

    private ApiRequest.Builder eventBuilder(URL url) {
        return ApiRequest.newBuilder()
                .withUrl(url)
                .withHttpMethod(HttpMethod.POST)
                .withAppId(appId)
                .withDeviceId(deviceId)
                .withContentEncoding("gzip");
    }

    @Override
    public void close()  {
        EmbraceLogger.logInfo("Shutting down ApiClient");
        retryWorker.close();
        worker.close();
    }
}
