/*
 * Decompiled with CFR 0.152.
 */
package org.owasp.esapi.crypto;

import java.io.UnsupportedEncodingException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.List;
import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.owasp.esapi.ESAPI;
import org.owasp.esapi.Logger;
import org.owasp.esapi.crypto.CipherText;
import org.owasp.esapi.errors.EncryptionException;

public class CryptoHelper {
    private static final Logger logger = ESAPI.getLogger("CryptoHelper");

    public static SecretKey generateSecretKey(String alg, int keySize) throws EncryptionException {
        assert (keySize > 0);
        String[] cipherSpec = alg.split("/");
        String cipherAlg = cipherSpec[0];
        try {
            if (cipherAlg.toUpperCase().startsWith("PBEWITH")) {
                cipherAlg = "PBE";
            }
            KeyGenerator kgen = KeyGenerator.getInstance(cipherAlg);
            kgen.init(keySize);
            return kgen.generateKey();
        }
        catch (NoSuchAlgorithmException e) {
            throw new EncryptionException("Failed to generate random secret key", "Failed to generate secret key for " + alg + " with size of " + keySize + " bits.", e);
        }
    }

    public static SecretKey computeDerivedKey(SecretKey keyDerivationKey, int keySize, String purpose) throws NoSuchAlgorithmException, InvalidKeyException, EncryptionException {
        byte[] inputBytes;
        assert (keyDerivationKey != null) : "Master key cannot be null.";
        assert (keySize >= 56) : "Master key has size of " + keySize + ", which is less than minimum of 56-bits.";
        assert (keySize % 8 == 0) : "Key size (" + keySize + ") must be a even multiple of 8-bits.";
        assert (purpose != null);
        assert (purpose.equals("encryption") || purpose.equals("authenticity")) : "Purpose must be \"encryption\" or \"authenticity\".";
        keySize = CryptoHelper.calcKeySize(keySize);
        byte[] derivedKey = new byte[keySize];
        byte[] tmpKey = null;
        try {
            inputBytes = purpose.getBytes("UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new EncryptionException("Encryption failure (internal encoding error: UTF-8)", "UTF-8 encoding is NOT supported as a standard byte encoding: " + e.getMessage(), e);
        }
        SecretKeySpec sk = new SecretKeySpec(keyDerivationKey.getEncoded(), "HmacSHA1");
        Mac mac = null;
        try {
            mac = Mac.getInstance("HmacSHA1");
            mac.init(sk);
        }
        catch (InvalidKeyException ex) {
            logger.error(Logger.SECURITY_FAILURE, "Created HmacSHA1 Mac but SecretKey sk has alg " + sk.getAlgorithm(), ex);
            throw ex;
        }
        int totalCopied = 0;
        int destPos = 0;
        int len = 0;
        do {
            len = (tmpKey = mac.doFinal(inputBytes)).length >= keySize ? keySize : Math.min(tmpKey.length, keySize - totalCopied);
            System.arraycopy(tmpKey, 0, derivedKey, destPos, len);
            inputBytes = tmpKey;
            destPos += len;
        } while ((totalCopied += tmpKey.length) < keySize);
        return new SecretKeySpec(derivedKey, keyDerivationKey.getAlgorithm());
    }

    public static boolean isCombinedCipherMode(String cipherMode) {
        assert (cipherMode != null) : "Cipher mode may not be null";
        assert (!cipherMode.equals("")) : "Cipher mode may not be empty string";
        List<String> combinedCipherModes = ESAPI.securityConfiguration().getCombinedCipherModes();
        return combinedCipherModes.contains(cipherMode);
    }

    public static boolean isAllowedCipherMode(String cipherMode) {
        if (CryptoHelper.isCombinedCipherMode(cipherMode)) {
            return true;
        }
        List<String> extraCipherModes = ESAPI.securityConfiguration().getAdditionalAllowedCipherModes();
        return extraCipherModes.contains(cipherMode);
    }

    public static boolean isMACRequired(CipherText ct) {
        boolean preferredCipherMode = CryptoHelper.isCombinedCipherMode(ct.getCipherMode());
        boolean wantsMAC = ESAPI.securityConfiguration().useMACforCipherText();
        return !preferredCipherMode && wantsMAC;
    }

    public static boolean isCipherTextMACvalid(SecretKey sk, CipherText ct) {
        if (CryptoHelper.isMACRequired(ct)) {
            try {
                SecretKey authKey = CryptoHelper.computeDerivedKey(sk, ct.getKeySize(), "authenticity");
                boolean validMAC = ct.validateMAC(authKey);
                return validMAC;
            }
            catch (Exception ex) {
                logger.warning(Logger.SECURITY_FAILURE, "Unable to validate MAC for ciphertext " + ct, ex);
                return false;
            }
        }
        return true;
    }

    public static void overwrite(byte[] bytes, byte x) {
        Arrays.fill(bytes, x);
    }

    public static void overwrite(byte[] bytes) {
        CryptoHelper.overwrite(bytes, (byte)42);
    }

    public static void copyByteArray(byte[] src, byte[] dest, int length) {
        System.arraycopy(src, 0, dest, 0, length);
    }

    public static void copyByteArray(byte[] src, byte[] dest) {
        CryptoHelper.copyByteArray(src, dest, src.length);
    }

    public static boolean arrayCompare(byte[] b1, byte[] b2) {
        if (b1 == b2) {
            return true;
        }
        if (b1 == null || b2 == null) {
            return b1 == b2;
        }
        if (b1.length != b2.length) {
            return false;
        }
        int result = 0;
        for (int i = 0; i < b1.length; ++i) {
            result |= b1[i] ^ b2[i];
        }
        return result == 0;
    }

    private static int calcKeySize(int ks) {
        assert (ks > 0) : "Key size must be > 0 bits.";
        int numBytes = 0;
        int n = ks / 8;
        int rem = ks % 8;
        numBytes = rem == 0 ? n : n + 1;
        return numBytes;
    }

    private CryptoHelper() {
    }
}

