package edu.chop.cbmi.dataExpress.backends

import edu.chop.cbmi.dataExpress.dataModels.sql._
import edu.chop.cbmi.dataExpress.dataModels.DataType
import java.sql.ResultSetMetaData

/**
 * SQL dialect for Oracle 
 * @see [[edu.chop.cbmi.dataExpress.backends.SqlDialect]]
 * 
 */
case object OracleSqlDialect extends SqlDialect {
  val identifierQuote = "\""
  val tableCreate = "CREATE TABLE"
  val tableTruncate = "TRUNCATE TABLE"
  val tableDrop = "DROP TABLE"
  val tableCascade = "CASCADE CONSTRAINTS"
  val insertStatement = "INSERT INTO"
  val insertValues = "VALUES"
  val transactionStart = "BEGIN TRANSACTION"
  val transactionEnd = "COMMIT"              //This should really return None for Oracle 11g + ?
  val transactionCommit = "COMMIT"
  val transactionRollback = "ROLLBACK"

  /*
   * Note that as per Oracle advice (http://docs.oracle.com/cd/E14072_01/server.112/e10592/sql_elements008.htm),
   * it is not advisable to quote all things. Under the principle of making things "just work" most of the time,
   * we will quote identifiers only when they clash with a reserved word
   */
  def quoteIdentifier(id: String) = {
   	if (reservedWords.contains(id.toUpperCase())){
   		"%s%s%s".format(identifierQuote, id, identifierQuote)
   	}
   	else {
   	      id
   	}
    
  }

  def createTable(name: String, columns: List[(String, DataType)], schemaName: Option[String]) = {
    val quotedSchemaPrefix = quoteSchemaName(schemaName)

    val createBlock = columns.map {
      (t: Tuple2[String, DataType]) => "%s %s".format(quoteIdentifier(t._1), toSqlString(t._2))
    }
    val createString = "%s %s%s (%s)".format(tableCreate, quotedSchemaPrefix, quoteIdentifier(name), createBlock.mkString(", "))
    createString
  }

  def dropTable(name: String, cascade: Boolean = false, schemaName: Option[String]) = {
    val quotedSchemaPrefix = quoteSchemaName(schemaName) 
   
    val fmtString = if (cascade) "%s %s%s " + tableCascade else "%s %s%s"
    fmtString.format(tableDrop, quotedSchemaPrefix, quoteIdentifier(name))
  }

  def truncate(table: String, schemaName: Option[String]) = {
    val quotedSchemaPrefix = quoteSchemaName(schemaName)

    "%s %s%s".format(tableTruncate, quotedSchemaPrefix, quoteIdentifier(table))
  }

  def commit() = transactionCommit

  def rollback() = transactionRollback

  def endTransaction() = transactionEnd

  def startTransaction() = transactionStart

  def insertRecord(tableName: String, columnNames: List[String], schemaName: Option[String] = None) = {
    val valuesList = "%s".format(List.fill(columnNames.size)("?").mkString(","))

    val quotedSchemaPrefix = quoteSchemaName(schemaName)

    val sqlString = "%s %s%s (%s) %s (%s)".format(insertStatement,
      quotedSchemaPrefix,
      quoteIdentifier(tableName),
      columnNames.map {
        n => quoteIdentifier(n)
      }.mkString(","),
      insertValues,
      valuesList)
    sqlString
  }

  //TODO: Create a "WHERE" clause object so we can do more than equality here
  def updateRecords(tableName: String, columnNames: List[String], filter: List[(String, Any)], schemaName: Option[String] = None) = {
    val quotedSchemaPrefix = quoteSchemaName(schemaName)

    val setString = columnNames.map {
      i => "%s = ?".format(quoteIdentifier(i))
    }.mkString(", ")
    val whereString = if (filter.size > 0) {
      "WHERE %s".format(filter.map {
        i => quoteIdentifier(i._1) + " = ?"
      }.mkString(" AND "))
    }
    else {
      "" //Empty string so we don't cause errors when doing bulk updates
    }
    val sqlString = "UPDATE %s%s SET %s %s".format(quotedSchemaPrefix, quoteIdentifier(tableName), setString, whereString)
    sqlString
  }


  def toSqlString(dataType: DataType): String = {
    dataType match {
      case IntegerDataType                                          => "INTEGER"
      case SmallIntegerDataType                                     => "SMALLINT"
      //TODO: Remove blatant Postgresql hack for lack of tinyint support below
      case TinyIntegerDataType                                      => "SMALLINT"
      case FloatDataType(p)                                           => "FLOAT"
      case DecimalDataType(p, s)                                      => "DECIMAL(%d,%d)".format(p, s)
      case DateDataType                                             => "DATE"
      case DateTimeDataType(true)                                     => "TIMESTAMP WITH TIME ZONE"
      case DateTimeDataType(false)                                    => "TIMESTAMP"
      case TimeDataType(true)                                         => "TIME WITH TIME ZONE"
      case TimeDataType(false)                                        => "TIME"
      case CharacterDataType(length, fixed)                           => {
        if (fixed) "CHAR(%d)".format(length)
        else "VARCHAR(%d)".format(length)
      }
      case TextDataType                                             => "CLOB"
      case BigBinaryDataType                                        => "BLOB"
      case BooleanDataType                                          => "NUMBER(1,0)"
      case BitDataType                                              => "NUMBER(1,0)"
      case _                                                          => ""
    }
  }

  def quoteSchemaName (schemaName: Option[String]) = {
    schemaName match {
        case None => ""
        case _ => quoteIdentifier(schemaName.get) + "."
    }
  }
  //Oracle reserved word list used for determining whether to quote an identifier.
  val reservedWords = List("!",
    "!=",
    "&",
    "(",
    ")",
    "*",
    "+",
    ",",
    "-",
    ".",
    "/",
    ":",
    ";",
    "<",
    "<<",
    "<=",
    "=",
    "=>",
    ">",
    ">=",
    "@",
    "A",
    "ABORT",
    "ABS",
    "ACCESS",
    "ACCESSED",
    "ACCOUNT",
    "ACOS",
    "ACTIVATE",
    "ACTIVE_COMPONENT",
    "ACTIVE_FUNCTION",
    "ACTIVE_TAG",
    "ADD",
    "ADD_COLUMN",
    "ADD_GROUP",
    "ADD_MONTHS",
    "ADJ_DATE",
    "ADMIN",
    "ADMINISTER",
    "ADMINISTRATOR",
    "ADVISE",
    "ADVISOR",
    "AFTER",
    "ALIAS",
    "ALL",
    "ALLOCATE",
    "ALLOW",
    "ALL_ROWS",
    "ALTER",
    "ALWAYS",
    "ANALYZE",
    "ANCILLARY",
    "AND",
    "AND_EQUAL",
    "ANTIJOIN",
    "ANY",
    "ANYSCHEMA",
    "APPEND",
    "APPENDCHILDXML",
    "APPEND_VALUES",
    "APPLY",
    "ARCHIVE",
    "ARCHIVELOG",
    "ARRAY",
    "AS",
    "ASC",
    "ASCII",
    "ASCIISTR",
    "ASIN",
    "ASSEMBLY",
    "ASSOCIATE",
    "ASYNC",
    "ASYNCHRONOUS",
    "AT",
    "ATAN",
    "ATAN2",
    "ATTRIBUTE",
    "ATTRIBUTES",
    "AUDIT",
    "AUTHENTICATED",
    "AUTHENTICATION",
    "AUTHID",
    "AUTHORIZATION",
    "AUTO",
    "AUTOALLOCATE",
    "AUTOEXTEND",
    "AUTOMATIC",
    "AVAILABILITY",
    "AVG",
    "BACKUP",
    "BASIC",
    "BASICFILE",
    "BATCH",
    "BECOME",
    "BEFORE",
    "BEGIN",
    "BEGIN_OUTLINE_DATA",
    "BEHALF",
    "BETWEEN",
    "BFILE",
    "BFILENAME",
    "BIGFILE",
    "BINARY",
    "BINARY_DOUBLE",
    "BINARY_DOUBLE_INFINITY",
    "BINARY_DOUBLE_NAN",
    "BINARY_FLOAT",
    "BINARY_FLOAT_INFINITY",
    "BINARY_FLOAT_NAN",
    "BINDING",
    "BIND_AWARE",
    "BIN_TO_NUM",
    "BITAND",
    "BITMAP",
    "BITMAPS",
    "BITMAP_TREE",
    "BITS",
    "BLOB",
    "BLOCK",
    "BLOCKS",
    "BLOCKSIZE",
    "BLOCK_RANGE",
    "BODY",
    "BOTH",
    "BOUND",
    "BRANCH",
    "BREADTH",
    "BROADCAST",
    "BUFFER",
    "BUFFER_CACHE",
    "BUFFER_POOL",
    "BUILD",
    "BULK",
    "BY",
    "BYPASS_RECURSIVE_CHECK",
    "BYPASS_UJVC",
    "BYTE",
    "CACHE",
    "CACHE_CB",
    "CACHE_INSTANCES",
    "CACHE_TEMP_TABLE",
    "CALL",
    "CANCEL",
    "CARDINALITY",
    "CASCADE",
    "CASE",
    "CAST",
    "CATEGORY",
    "CEIL",
    "CELL_FLASH_CACHE",
    "CERTIFICATE",
    "CFILE",
    "CHAINED",
    "CHANGE",
    "CHANGE_DUPKEY_ERROR_INDEX",
    "CHAR",
    "CHARACTER",
    "CHARTOROWID",
    "CHAR_CS",
    "CHECK",
    "CHECKPOINT",
    "CHECK_ACL_REWRITE",
    "CHILD",
    "CHOOSE",
    "CHR",
    "CHUNK",
    "CLASS",
    "CLEAR",
    "CLOB",
    "CLONE",
    "CLOSE",
    "CLOSE_CACHED_OPEN_CURSORS",
    "CLUSTER",
    "CLUSTERING_FACTOR",
    "CLUSTER_ID",
    "CLUSTER_PROBABILITY",
    "CLUSTER_SET",
    "COALESCE",
    "COALESCE_SQ",
    "COARSE",
    "COLD",
    "COLLECT",
    "COLUMN",
    "COLUMNAR",
    "COLUMNS",
    "COLUMN_AUTH_INDICATOR",
    "COLUMN_STATS",
    "COLUMN_VALUE",
    "COMMENT",
    "COMMIT",
    "COMMITTED",
    "COMPACT",
    "COMPATIBILITY",
    "COMPILE",
    "COMPLETE",
    "COMPLIANCE",
    "COMPOSE",
    "COMPOSITE",
    "COMPOSITE_LIMIT",
    "COMPOUND",
    "COMPRESS",
    "COMPUTE",
    "CONCAT",
    "CONFIRM",
    "CONFORMING",
    "CONNECT",
    "CONNECT_BY_CB_WHR_ONLY",
    "CONNECT_BY_COMBINE_SW",
    "CONNECT_BY_COST_BASED",
    "CONNECT_BY_ELIM_DUPS",
    "CONNECT_BY_FILTERING",
    "CONNECT_BY_ISCYCLE",
    "CONNECT_BY_ISLEAF",
    "CONNECT_BY_ROOT",
    "CONNECT_TIME",
    "CONSIDER",
    "CONSISTENT",
    "CONST",
    "CONSTANT",
    "CONSTRAINT",
    "CONSTRAINTS",
    "CONTAINER",
    "CONTENT",
    "CONTENTS",
    "CONTEXT",
    "CONTINUE",
    "CONTROLFILE",
    "CONVERT",
    "CORR",
    "CORRUPTION",
    "CORRUPT_XID",
    "CORRUPT_XID_ALL",
    "CORR_K",
    "CORR_S",
    "COS",
    "COSH",
    "COST",
    "COST_XML_QUERY_REWRITE",
    "COUNT",
    "COVAR_POP",
    "COVAR_SAMP",
    "CO_AUTH_IND",
    "CPU_COSTING",
    "CPU_PER_CALL",
    "CPU_PER_SESSION",
    "CRASH",
    "CREATE",
    "CREATE_STORED_OUTLINES",
    "CREATION",
    "CROSS",
    "CROSSEDITION",
    "CSCONVERT",
    "CUBE",
    "CUBE_GB",
    "CUME_DIST",
    "CUME_DISTM",
    "CURRENT",
    "CURRENTV",
    "CURRENT_DATE",
    "CURRENT_SCHEMA",
    "CURRENT_TIME",
    "CURRENT_TIMESTAMP",
    "CURRENT_USER",
    "CURSOR",
    "CURSOR_SHARING_EXACT",
    "CURSOR_SPECIFIC_SEGMENT",
    "CV",
    "CYCLE",
    "D",
    "DANGLING",
    "DATA",
    "DATABASE",
    "DATAFILE",
    "DATAFILES",
    "DATAOBJNO",
    "DATAOBJ_TO_PARTITION",
    "DATE",
    "DATE_MODE",
    "DAY",
    "DBA",
    "DBA_RECYCLEBIN",
    "DBMS_STATS",
    "DBTIMEZONE",
    "DB_ROLE_CHANGE",
    "DB_VERSION",
    "DDL",
    "DEALLOCATE",
    "DEBUG",
    "DEBUGGER",
    "DEC",
    "DECIMAL",
    "DECLARE",
    "DECODE",
    "DECOMPOSE",
    "DECR",
    "DECREMENT",
    "DECRYPT",
    "DEDUPLICATE",
    "DEFAULT",
    "DEFAULTS",
    "DEFERRABLE",
    "DEFERRED",
    "DEFINED",
    "DEFINER",
    "DEGREE",
    "DELAY",
    "DELETE",
    "DELETEXML",
    "DEMAND",
    "DENSE_RANK",
    "DENSE_RANKM",
    "DEPENDENT",
    "DEPTH",
    "DEQUEUE",
    "DEREF",
    "DEREF_NO_REWRITE",
    "DESC",
    "DETACHED",
    "DETERMINES",
    "DICTIONARY",
    "DIMENSION",
    "DIRECTORY",
    "DIRECT_LOAD",
    "DIRECT_PATH",
    "DISABLE",
    "DISABLE_PRESET",
    "DISABLE_RPKE",
    "DISALLOW",
    "DISASSOCIATE",
    "DISCONNECT",
    "DISK",
    "DISKGROUP",
    "DISKS",
    "DISMOUNT",
    "DISTINCT",
    "DISTINGUISHED",
    "DISTRIBUTED",
    "DML",
    "DML_UPDATE",
    "DOCFIDELITY",
    "DOCUMENT",
    "DOMAIN_INDEX_FILTER",
    "DOMAIN_INDEX_NO_SORT",
    "DOMAIN_INDEX_SORT",
    "DOUBLE",
    "DOWNGRADE",
    "DRIVING_SITE",
    "DROP",
    "DROP_COLUMN",
    "DROP_GROUP",
    "DST_UPGRADE_INSERT_CONV",
    "DUMP",
    "DYNAMIC",
    "DYNAMIC_SAMPLING",
    "DYNAMIC_SAMPLING_EST_CDN",
    "E",
    "EACH",
    "EDITION",
    "EDITIONING",
    "EDITIONS",
    "ELEMENT",
    "ELIMINATE_JOIN",
    "ELIMINATE_OBY",
    "ELIMINATE_OUTER_JOIN",
    "ELSE",
    "EMPTY",
    "EMPTY_BLOB",
    "EMPTY_CLOB",
    "ENABLE",
    "ENABLE_PRESET",
    "ENCODING",
    "ENCRYPT",
    "ENCRYPTION",
    "END",
    "END_OUTLINE_DATA",
    "ENFORCE",
    "ENFORCED",
    "ENQUEUE",
    "ENTERPRISE",
    "ENTITYESCAPING",
    "ENTRY",
    "ERROR",
    "ERRORS",
    "ERROR_ARGUMENT",
    "ERROR_ON_OVERLAP_TIME",
    "ESCAPE",
    "ESTIMATE",
    "EVALNAME",
    "EVALUATION",
    "EVENTS",
    "EVERY",
    "EXCEPT",
    "EXCEPTIONS",
    "EXCHANGE",
    "EXCLUDE",
    "EXCLUDING",
    "EXCLUSIVE",
    "EXECUTE",
    "EXEMPT",
    "EXISTS",
    "EXISTSNODE",
    "EXP",
    "EXPAND_GSET_TO_UNION",
    "EXPAND_TABLE",
    "EXPIRE",
    "EXPLAIN",
    "EXPLOSION",
    "EXPORT",
    "EXPR_CORR_CHECK",
    "EXTENDS",
    "EXTENT",
    "EXTENTS",
    "EXTERNAL",
    "EXTERNALLY",
    "EXTRA",
    "EXTRACT",
    "EXTRACTVALUE",
    "FACILITY",
    "FACT",
    "FACTORIZE_JOIN",
    "FAILED",
    "FAILED_LOGIN_ATTEMPTS",
    "FAILGROUP",
    "FALSE",
    "FAST",
    "FBTSCAN",
    "FEATURE_ID",
    "FEATURE_SET",
    "FEATURE_VALUE",
    "FILE",
    "FILESYSTEM_LIKE_LOGGING",
    "FILTER",
    "FINAL",
    "FINE",
    "FINISH",
    "FIRST",
    "FIRSTM",
    "FIRST_ROWS",
    "FIRST_VALUE",
    "FLAGGER",
    "FLASHBACK",
    "FLASH_CACHE",
    "FLOAT",
    "FLOB",
    "FLOOR",
    "FLUSH",
    "FOLDER",
    "FOLLOWING",
    "FOLLOWS",
    "FOR",
    "FORCE",
    "FORCE_XML_QUERY_REWRITE",
    "FOREIGN",
    "FOREVER",
    "FORWARD",
    "FRAGMENT_NUMBER",
    "FREELIST",
    "FREELISTS",
    "FREEPOOLS",
    "FRESH",
    "FROM",
    "FROM_TZ",
    "FULL",
    "FUNCTION",
    "FUNCTIONS",
    "G",
    "GATHER_PLAN_STATISTICS",
    "GBY_CONC_ROLLUP",
    "GBY_PUSHDOWN",
    "GENERATED",
    "GLOBAL",
    "GLOBALLY",
    "GLOBAL_NAME",
    "GLOBAL_TOPIC_ENABLED",
    "GRANT",
    "GREATEST",
    "GROUP",
    "GROUPING",
    "GROUPING_ID",
    "GROUPS",
    "GROUP_BY",
    "GROUP_ID",
    "GUARANTEE",
    "GUARANTEED",
    "GUARD",
    "H",
    "HASH",
    "HASHKEYS",
    "HASH_AJ",
    "HASH_SJ",
    "HAVING",
    "HEADER",
    "HEAP",
    "HELP",
    "HEXTORAW",
    "HEXTOREF",
    "HIDDEN",
    "HIDE",
    "HIERARCHY",
    "HIGH",
    "HINTSET_BEGIN",
    "HINTSET_END",
    "HOT",
    "HOUR",
    "HWM_BROKERED",
    "HYBRID",
    "ID",
    "IDENTIFIED",
    "IDENTIFIER",
    "IDENTITY",
    "IDGENERATORS",
    "IDLE_TIME",
    "IF",
    "IGNORE",
    "IGNORE_OPTIM_EMBEDDED_HINTS",
    "IGNORE_ROW_ON_DUPKEY_INDEX",
    "IGNORE_WHERE_CLAUSE",
    "IMMEDIATE",
    "IMPACT",
    "IMPORT",
    "IN",
    "INCLUDE",
    "INCLUDE_VERSION",
    "INCLUDING",
    "INCR",
    "INCREMENT",
    "INCREMENTAL",
    "INDENT",
    "INDEX",
    "INDEXED",
    "INDEXES",
    "INDEXTYPE",
    "INDEXTYPES",
    "INDEX_ASC",
    "INDEX_COMBINE",
    "INDEX_DESC",
    "INDEX_FFS",
    "INDEX_FILTER",
    "INDEX_JOIN",
    "INDEX_ROWS",
    "INDEX_RRS",
    "INDEX_RS",
    "INDEX_RS_ASC",
    "INDEX_RS_DESC",
    "INDEX_SCAN",
    "INDEX_SKIP_SCAN",
    "INDEX_SS",
    "INDEX_SS_ASC",
    "INDEX_SS_DESC",
    "INDEX_STATS",
    "INDICATOR",
    "INFINITE",
    "INFORMATIONAL",
    "INITCAP",
    "INITIAL",
    "INITIALIZED",
    "INITIALLY",
    "INITRANS",
    "INLINE",
    "INLINE_XMLTYPE_NT",
    "INNER",
    "INSERT",
    "INSERTCHILDXML",
    "INSERTCHILDXMLAFTER",
    "INSERTCHILDXMLBEFORE",
    "INSERTXMLAFTER",
    "INSERTXMLBEFORE",
    "INSTANCE",
    "INSTANCES",
    "INSTANTIABLE",
    "INSTANTLY",
    "INSTEAD",
    "INSTR",
    "INSTR2",
    "INSTR4",
    "INSTRB",
    "INSTRC",
    "INT",
    "INTEGER",
    "INTERMEDIATE",
    "INTERNAL_CONVERT",
    "INTERNAL_USE",
    "INTERPRETED",
    "INTERSECT",
    "INTERVAL",
    "INTO",
    "INVALIDATE",
    "INVISIBLE",
    "IN_MEMORY_METADATA",
    "IN_XQUERY",
    "IS",
    "ISOLATION",
    "ISOLATION_LEVEL",
    "ITERATE",
    "ITERATION_NUMBER",
    "JAVA",
    "JOB",
    "JOIN",
    "K",
    "KEEP",
    "KEEP_DUPLICATES",
    "KERBEROS",
    "KEY",
    "KEYS",
    "KEYSIZE",
    "KEY_LENGTH",
    "KILL",
    "LAG",
    "LAST",
    "LAST_DAY",
    "LAST_VALUE",
    "LATERAL",
    "LAYER",
    "LDAP_REGISTRATION",
    "LDAP_REGISTRATION_ENABLED",
    "LDAP_REG_SYNC_INTERVAL",
    "LEAD",
    "LEADING",
    "LEAST",
    "LEFT",
    "LENGTH",
    "LENGTH2",
    "LENGTH4",
    "LENGTHB",
    "LENGTHC",
    "LESS",
    "LEVEL",
    "LEVELS",
    "LIBRARY",
    "LIFE",
    "LIFETIME",
    "LIKE",
    "LIKE2",
    "LIKE4",
    "LIKEC",
    "LIKE_EXPAND",
    "LIMIT",
    "LINK",
    "LIST",
    "LISTAGG",
    "LN",
    "LNNVL",
    "LOAD",
    "LOB",
    "LOBNVL",
    "LOBS",
    "LOCAL",
    "LOCALTIME",
    "LOCALTIMESTAMP",
    "LOCAL_INDEXES",
    "LOCATION",
    "LOCATOR",
    "LOCK",
    "LOCKED",
    "LOG",
    "LOGFILE",
    "LOGFILES",
    "LOGGING",
    "LOGICAL",
    "LOGICAL_READS_PER_CALL",
    "LOGICAL_READS_PER_SESSION",
    "LOGOFF",
    "LOGON",
    "LOG_READ_ONLY_VIOLATIONS",
    "LONG",
    "LOW",
    "LOWER",
    "LPAD",
    "LTRIM",
    "M",
    "MAIN",
    "MAKE_REF",
    "MANAGE",
    "MANAGED",
    "MANAGEMENT",
    "MANAGER",
    "MANUAL",
    "MAPPING",
    "MASTER",
    "MATCHED",
    "MATERIALIZE",
    "MATERIALIZED",
    "MAX",
    "MAXARCHLOGS",
    "MAXDATAFILES",
    "MAXEXTENTS",
    "MAXIMIZE",
    "MAXINSTANCES",
    "MAXLOGFILES",
    "MAXLOGHISTORY",
    "MAXLOGMEMBERS",
    "MAXSIZE",
    "MAXTRANS",
    "MAXVALUE",
    "MEASURE",
    "MEASURES",
    "MEDIAN",
    "MEDIUM",
    "MEMBER",
    "MEMORY",
    "MERGE",
    "MERGE$ACTIONS",
    "MERGE_AJ",
    "MERGE_CONST_ON",
    "MERGE_SJ",
    "METHOD",
    "MIGRATE",
    "MIGRATION",
    "MIN",
    "MINEXTENTS",
    "MINIMIZE",
    "MINIMUM",
    "MINING",
    "MINUS",
    "MINUS_NULL",
    "MINUTE",
    "MINVALUE",
    "MIRROR",
    "MIRRORCOLD",
    "MIRRORHOT",
    "MLSLABEL",
    "MOD",
    "MODE",
    "MODEL",
    "MODEL_COMPILE_SUBQUERY",
    "MODEL_DONTVERIFY_UNIQUENESS",
    "MODEL_DYNAMIC_SUBQUERY",
    "MODEL_MIN_ANALYSIS",
    "MODEL_NO_ANALYSIS",
    "MODEL_PBY",
    "MODEL_PUSH_REF",
    "MODIFY",
    "MONITOR",
    "MONITORING",
    "MONTH",
    "MONTHS_BETWEEN",
    "MOUNT",
    "MOUNTPATH",
    "MOVE",
    "MOVEMENT",
    "MULTISET",
    "MV_MERGE",
    "NAME",
    "NAMED",
    "NAMESPACE",
    "NAN",
    "NANVL",
    "NATIONAL",
    "NATIVE",
    "NATIVE_FULL_OUTER_JOIN",
    "NATURAL",
    "NAV",
    "NCHAR",
    "NCHAR_CS",
    "NCHR",
    "NCLOB",
    "NEEDED",
    "NESTED",
    "NESTED_TABLE_FAST_INSERT",
    "NESTED_TABLE_GET_REFS",
    "NESTED_TABLE_ID",
    "NESTED_TABLE_SET_REFS",
    "NESTED_TABLE_SET_SETID",
    "NETWORK",
    "NEVER",
    "NEW",
    "NEW_TIME",
    "NEXT",
    "NEXT_DAY",
    "NLJ_BATCHING",
    "NLJ_INDEX_FILTER",
    "NLJ_INDEX_SCAN",
    "NLJ_PREFETCH",
    "NLSSORT",
    "NLS_CALENDAR",
    "NLS_CHARACTERSET",
    "NLS_CHARSET_DECL_LEN",
    "NLS_CHARSET_ID",
    "NLS_CHARSET_NAME",
    "NLS_COMP",
    "NLS_CURRENCY",
    "NLS_DATE_FORMAT",
    "NLS_DATE_LANGUAGE",
    "NLS_INITCAP",
    "NLS_ISO_CURRENCY",
    "NLS_LANG",
    "NLS_LANGUAGE",
    "NLS_LENGTH_SEMANTICS",
    "NLS_LOWER",
    "NLS_NCHAR_CONV_EXCP",
    "NLS_NUMERIC_CHARACTERS",
    "NLS_SORT",
    "NLS_SPECIAL_CHARS",
    "NLS_TERRITORY",
    "NLS_UPPER",
    "NL_AJ",
    "NL_SJ",
    "NO",
    "NOAPPEND",
    "NOARCHIVELOG",
    "NOAUDIT",
    "NOCACHE",
    "NOCOMPRESS",
    "NOCPU_COSTING",
    "NOCYCLE",
    "NODELAY",
    "NOENTITYESCAPING",
    "NOFORCE",
    "NOGUARANTEE",
    "NOLOCAL",
    "NOLOGGING",
    "NOMAPPING",
    "NOMAXVALUE",
    "NOMINIMIZE",
    "NOMINVALUE",
    "NOMONITORING",
    "NONE",
    "NONSCHEMA",
    "NOORDER",
    "NOOVERRIDE",
    "NOPARALLEL",
    "NOPARALLEL_INDEX",
    "NORELY",
    "NOREPAIR",
    "NORESETLOGS",
    "NOREVERSE",
    "NOREWRITE",
    "NORMAL",
    "NOROWDEPENDENCIES",
    "NOSCHEMACHECK",
    "NOSEGMENT",
    "NOSORT",
    "NOSTRICT",
    "NOSWITCH",
    "NOT",
    "NOTHING",
    "NOTIFICATION",
    "NOVALIDATE",
    "NOWAIT",
    "NO_ACCESS",
    "NO_BASETABLE_MULTIMV_REWRITE",
    "NO_BIND_AWARE",
    "NO_BUFFER",
    "NO_CARTESIAN",
    "NO_CHECK_ACL_REWRITE",
    "NO_COALESCE_SQ",
    "NO_CONNECT_BY_CB_WHR_ONLY",
    "NO_CONNECT_BY_COMBINE_SW",
    "NO_CONNECT_BY_COST_BASED",
    "NO_CONNECT_BY_ELIM_DUPS",
    "NO_CONNECT_BY_FILTERING",
    "NO_COST_XML_QUERY_REWRITE",
    "NO_CPU_COSTING",
    "NO_DOMAIN_INDEX_FILTER",
    "NO_DST_UPGRADE_INSERT_CONV",
    "NO_ELIMINATE_JOIN",
    "NO_ELIMINATE_OBY",
    "NO_ELIMINATE_OUTER_JOIN",
    "NO_EXPAND",
    "NO_EXPAND_GSET_TO_UNION",
    "NO_EXPAND_TABLE",
    "NO_FACT",
    "NO_FACTORIZE_JOIN",
    "NO_FILTERING",
    "NO_GBY_PUSHDOWN",
    "NO_INDEX",
    "NO_INDEX_FFS",
    "NO_INDEX_SS",
    "NO_LOAD",
    "NO_MERGE",
    "NO_MODEL_PUSH_REF",
    "NO_MONITOR",
    "NO_MONITORING",
    "NO_MULTIMV_REWRITE",
    "NO_NATIVE_FULL_OUTER_JOIN",
    "NO_NLJ_BATCHING",
    "NO_NLJ_PREFETCH",
    "NO_ORDER_ROLLUPS",
    "NO_OUTER_JOIN_TO_INNER",
    "NO_PARALLEL",
    "NO_PARALLEL_INDEX",
    "NO_PARTIAL_COMMIT",
    "NO_PLACE_DISTINCT",
    "NO_PLACE_GROUP_BY",
    "NO_PQ_MAP",
    "NO_PRUNE_GSETS",
    "NO_PULL_PRED",
    "NO_PUSH_PRED",
    "NO_PUSH_SUBQ",
    "NO_PX_JOIN_FILTER",
    "NO_QKN_BUFF",
    "NO_QUERY_TRANSFORMATION",
    "NO_REF_CASCADE",
    "NO_RESULT_CACHE",
    "NO_REWRITE",
    "NO_SEMIJOIN",
    "NO_SET_TO_JOIN",
    "NO_SQL_TUNE",
    "NO_STAR_TRANSFORMATION",
    "NO_STATEMENT_QUEUING",
    "NO_STATS_GSETS",
    "NO_SUBQUERY_PRUNING",
    "NO_SUBSTRB_PAD",
    "NO_SWAP_JOIN_INPUTS",
    "NO_TABLE_LOOKUP_BY_NL",
    "NO_TEMP_TABLE",
    "NO_TRANSFORM_DISTINCT_AGG",
    "NO_UNNEST",
    "NO_USE_HASH",
    "NO_USE_HASH_AGGREGATION",
    "NO_USE_HASH_GBY_FOR_PUSHDOWN",
    "NO_USE_INVISIBLE_INDEXES",
    "NO_USE_MERGE",
    "NO_USE_NL",
    "NO_XDB_FASTPATH_INSERT",
    "NO_XMLINDEX_REWRITE",
    "NO_XMLINDEX_REWRITE_IN_SELECT",
    "NO_XML_DML_REWRITE",
    "NO_XML_QUERY_REWRITE",
    "NTH_VALUE",
    "NTILE",
    "NULL",
    "NULLIF",
    "NULLS",
    "NUMBER",
    "NUMERIC",
    "NUMTODSINTERVAL",
    "NUMTOYMINTERVAL",
    "NUM_INDEX_KEYS",
    "NVARCHAR2",
    "NVL",
    "NVL2",
    "OBJECT",
    "OBJECTTOXML",
    "OBJNO",
    "OBJNO_REUSE",
    "OCCURENCES",
    "OF",
    "OFF",
    "OFFLINE",
    "OID",
    "OIDINDEX",
    "OLAP",
    "OLD",
    "OLD_PUSH_PRED",
    "OLTP",
    "ON",
    "ONLINE",
    "ONLY",
    "OPAQUE",
    "OPAQUE_TRANSFORM",
    "OPAQUE_XCANONICAL",
    "OPCODE",
    "OPEN",
    "OPERATIONS",
    "OPERATOR",
    "OPTIMAL",
    "OPTIMIZER_FEATURES_ENABLE",
    "OPTIMIZER_GOAL",
    "OPTION",
    "OPT_ESTIMATE",
    "OPT_PARAM",
    "OR",
    "ORADEBUG",
    "ORA_BRANCH",
    "ORA_CHECKACL",
    "ORA_DST_AFFECTED",
    "ORA_DST_CONVERT",
    "ORA_DST_ERROR",
    "ORA_GET_ACLIDS",
    "ORA_GET_PRIVILEGES",
    "ORA_HASH",
    "ORA_ROWSCN",
    "ORA_ROWSCN_RAW",
    "ORA_ROWVERSION",
    "ORA_TABVERSION",
    "ORDER",
    "ORDERED",
    "ORDERED_PREDICATES",
    "ORDINALITY",
    "ORGANIZATION",
    "OR_EXPAND",
    "OR_PREDICATES",
    "OTHER",
    "OUTER",
    "OUTER_JOIN_TO_INNER",
    "OUTLINE",
    "OUTLINE_LEAF",
    "OUT_OF_LINE",
    "OVER",
    "OVERFLOW",
    "OVERFLOW_NOMOVE",
    "OVERLAPS",
    "OWN",
    "OWNER",
    "OWNERSHIP",
    "P",
    "PACKAGE",
    "PACKAGES",
    "PARALLEL",
    "PARALLEL_INDEX",
    "PARAM",
    "PARAMETERS",
    "PARENT",
    "PARITY",
    "PART$NUM$INST",
    "PARTIALLY",
    "PARTITION",
    "PARTITIONS",
    "PARTITION_HASH",
    "PARTITION_LIST",
    "PARTITION_RANGE",
    "PASSING",
    "PASSWORD",
    "PASSWORD_GRACE_TIME",
    "PASSWORD_LIFE_TIME",
    "PASSWORD_LOCK_TIME",
    "PASSWORD_REUSE_MAX",
    "PASSWORD_REUSE_TIME",
    "PASSWORD_VERIFY_FUNCTION",
    "PATH",
    "PATHS",
    "PBL_HS_BEGIN",
    "PBL_HS_END",
    "PCTFREE",
    "PCTINCREASE",
    "PCTTHRESHOLD",
    "PCTUSED",
    "PCTVERSION",
    "PENDING",
    "PERCENT",
    "PERCENTILE_CONT",
    "PERCENTILE_DISC",
    "PERCENT_RANK",
    "PERCENT_RANKM",
    "PERFORMANCE",
    "PERMANENT",
    "PERMISSION",
    "PFILE",
    "PHYSICAL",
    "PIKEY",
    "PIVOT",
    "PIV_GB",
    "PIV_SSF",
    "PLACE_DISTINCT",
    "PLACE_GROUP_BY",
    "PLAN",
    "PLSCOPE_SETTINGS",
    "PLSQL_CCFLAGS",
    "PLSQL_CODE_TYPE",
    "PLSQL_DEBUG",
    "PLSQL_OPTIMIZE_LEVEL",
    "PLSQL_WARNINGS",
    "POINT",
    "POLICY",
    "POST_TRANSACTION",
    "POWER",
    "POWERMULTISET",
    "POWERMULTISET_BY_CARDINALITY",
    "PQ_DISTRIBUTE",
    "PQ_MAP",
    "PQ_NOMAP",
    "PREBUILT",
    "PRECEDES",
    "PRECEDING",
    "PRECISION",
    "PRECOMPUTE_SUBQUERY",
    "PREDICATE_REORDERS",
    "PREDICTION",
    "PREDICTION_BOUNDS",
    "PREDICTION_COST",
    "PREDICTION_DETAILS",
    "PREDICTION_PROBABILITY",
    "PREDICTION_SET",
    "PREPARE",
    "PRESENT",
    "PRESENTNNV",
    "PRESENTV",
    "PRESERVE",
    "PRESERVE_OID",
    "PREVIOUS",
    "PRIMARY",
    "PRIOR",
    "PRIVATE",
    "PRIVATE_SGA",
    "PRIVILEGE",
    "PRIVILEGES",
    "PROCEDURAL",
    "PROCEDURE",
    "PROCESS",
    "PROFILE",
    "PROGRAM",
    "PROJECT",
    "PROPAGATE",
    "PROTECTED",
    "PROTECTION",
    "PUBLIC",
    "PULL_PRED",
    "PURGE",
    "PUSH_PRED",
    "PUSH_SUBQ",
    "PX_GRANULE",
    "PX_JOIN_FILTER",
    "QB_NAME",
    "QUERY",
    "QUERY_BLOCK",
    "QUEUE",
    "QUEUE_CURR",
    "QUEUE_ROWP",
    "QUIESCE",
    "QUORUM",
    "QUOTA",
    "RANDOM",
    "RANDOM_LOCAL",
    "RANGE",
    "RANK",
    "RANKM",
    "RAPIDLY",
    "RATIO_TO_REPORT",
    "RAW",
    "RAWTOHEX",
    "RAWTONHEX",
    "RBA",
    "RBO_OUTLINE",
    "RDBA",
    "READ",
    "READS",
    "REAL",
    "REBALANCE",
    "REBUILD",
    "RECORDS_PER_BLOCK",
    "RECOVER",
    "RECOVERABLE",
    "RECOVERY",
    "RECYCLE",
    "RECYCLEBIN",
    "REDO",
    "REDUCED",
    "REDUNDANCY",
    "REF",
    "REFERENCE",
    "REFERENCED",
    "REFERENCES",
    "REFERENCING",
    "REFRESH",
    "REFTOHEX",
    "REF_CASCADE_CURSOR",
    "REGEXP_COUNT",
    "REGEXP_INSTR",
    "REGEXP_LIKE",
    "REGEXP_REPLACE",
    "REGEXP_SUBSTR",
    "REGISTER",
    "REGR_AVGX",
    "REGR_AVGY",
    "REGR_COUNT",
    "REGR_INTERCEPT",
    "REGR_R2",
    "REGR_SLOPE",
    "REGR_SXX",
    "REGR_SXY",
    "REGR_SYY",
    "REGULAR",
    "REJECT",
    "REKEY",
    "RELATIONAL",
    "RELY",
    "REMAINDER",
    "REMOTE_MAPPED",
    "REMOVE",
    "RENAME",
    "REPAIR",
    "REPEAT",
    "REPLACE",
    "REPLICATION",
    "REQUIRED",
    "RESET",
    "RESETLOGS",
    "RESIZE",
    "RESOLVE",
    "RESOLVER",
    "RESOURCE",
    "RESPECT",
    "RESTORE",
    "RESTORE_AS_INTERVALS",
    "RESTRICT",
    "RESTRICTED",
    "RESTRICT_ALL_REF_CONS",
    "RESULT_CACHE",
    "RESUMABLE",
    "RESUME",
    "RETENTION",
    "RETRY_ON_ROW_CHANGE",
    "RETURN",
    "RETURNING",
    "REUSE",
    "REVERSE",
    "REVOKE",
    "REWRITE",
    "REWRITE_OR_ERROR",
    "RIGHT",
    "ROLE",
    "ROLES",
    "ROLLBACK",
    "ROLLING",
    "ROLLUP",
    "ROUND",
    "ROW",
    "ROWDEPENDENCIES",
    "ROWID",
    "ROWIDTOCHAR",
    "ROWIDTONCHAR",
    "ROWNUM",
    "ROWS",
    "ROW_LENGTH",
    "ROW_NUMBER",
    "RPAD",
    "RTRIM",
    "RULE",
    "RULES",
    "SALT",
    "SAMPLE",
    "SAVEPOINT",
    "SAVE_AS_INTERVALS",
    "SB4",
    "SCALE",
    "SCALE_ROWS",
    "SCAN",
    "SCAN_INSTANCES",
    "SCHEDULER",
    "SCHEMA",
    "SCHEMACHECK",
    "SCN",
    "SCN_ASCENDING",
    "SCOPE",
    "SD_ALL",
    "SD_INHIBIT",
    "SD_SHOW",
    "SEARCH",
    "SECOND",
    "SECUREFILE",
    "SECUREFILE_DBA",
    "SECURITY",
    "SEED",
    "SEGMENT",
    "SEG_BLOCK",
    "SEG_FILE",
    "SELECT",
    "SELECTIVITY",
    "SEMIJOIN",
    "SEMIJOIN_DRIVER",
    "SEQUENCE",
    "SEQUENCED",
    "SEQUENTIAL",
    "SERIALIZABLE",
    "SERVERERROR",
    "SESSION",
    "SESSIONS_PER_USER",
    "SESSIONTIMEZONE",
    "SESSIONTZNAME",
    "SESSION_CACHED_CURSORS",
    "SET",
    "SETS",
    "SETTINGS",
    "SET_TO_JOIN",
    "SEVERE",
    "SHARE",
    "SHARED",
    "SHARED_POOL",
    "SHOW",
    "SHRINK",
    "SHUTDOWN",
    "SIBLINGS",
    "SID",
    "SIGN",
    "SIGNAL_COMPONENT",
    "SIGNAL_FUNCTION",
    "SIMPLE",
    "SIN",
    "SINGLE",
    "SINGLETASK",
    "SINH",
    "SIZE",
    "SKIP",
    "SKIP_EXT_OPTIMIZER",
    "SKIP_UNQ_UNUSABLE_IDX",
    "SKIP_UNUSABLE_INDEXES",
    "SMALLFILE",
    "SMALLINT",
    "SNAPSHOT",
    "SOME",
    "SORT",
    "SOUNDEX",
    "SOURCE",
    "SPACE",
    "SPECIFICATION",
    "SPFILE",
    "SPLIT",
    "SPREADSHEET",
    "SQL",
    "SQLLDR",
    "SQL_TRACE",
    "SQRT",
    "STALE",
    "STANDALONE",
    "STANDBY",
    "STANDBY_MAX_DATA_DELAY",
    "STAR",
    "START",
    "STARTUP",
    "STAR_TRANSFORMATION",
    "STATEMENTS",
    "STATEMENT_ID",
    "STATEMENT_QUEUING",
    "STATIC",
    "STATISTICS",
    "STATS_BINOMIAL_TEST",
    "STATS_CROSSTAB",
    "STATS_F_TEST",
    "STATS_KS_TEST",
    "STATS_MODE",
    "STATS_MW_TEST",
    "STATS_ONE_WAY_ANOVA",
    "STATS_T_TEST_INDEP",
    "STATS_T_TEST_INDEPU",
    "STATS_T_TEST_ONE",
    "STATS_T_TEST_PAIRED",
    "STATS_WSR_TEST",
    "STDDEV",
    "STDDEV_POP",
    "STDDEV_SAMP",
    "STOP",
    "STORAGE",
    "STORE",
    "STREAMS",
    "STRICT",
    "STRING",
    "STRIP",
    "STRIPE_COLUMNS",
    "STRIPE_WIDTH",
    "STRUCTURE",
    "SUBMULTISET",
    "SUBPARTITION",
    "SUBPARTITIONS",
    "SUBPARTITION_REL",
    "SUBQUERIES",
    "SUBQUERY_PRUNING",
    "SUBSTITUTABLE",
    "SUBSTR",
    "SUBSTR2",
    "SUBSTR4",
    "SUBSTRB",
    "SUBSTRC",
    "SUCCESSFUL",
    "SUM",
    "SUMMARY",
    "SUPPLEMENTAL",
    "SUSPEND",
    "SWAP_JOIN_INPUTS",
    "SWITCH",
    "SWITCHOVER",
    "SYNC",
    "SYNCHRONOUS",
    "SYNONYM",
    "SYSASM",
    "SYSAUX",
    "SYSDATE",
    "SYSDBA",
    "SYSOPER",
    "SYSTEM",
    "SYSTEM_DEFINED",
    "SYSTIMESTAMP",
    "SYS_AUDIT",
    "SYS_CHECKACL",
    "SYS_CONNECT_BY_PATH",
    "SYS_CONTEXT",
    "SYS_DBURIGEN",
    "SYS_DL_CURSOR",
    "SYS_DM_RXFORM_CHR",
    "SYS_DM_RXFORM_NUM",
    "SYS_DOM_COMPARE",
    "SYS_DST_PRIM2SEC",
    "SYS_DST_SEC2PRIM",
    "SYS_ET_BFILE_TO_RAW",
    "SYS_ET_BLOB_TO_IMAGE",
    "SYS_ET_IMAGE_TO_BLOB",
    "SYS_ET_RAW_TO_BFILE",
    "SYS_EXTPDTXT",
    "SYS_EXTRACT_UTC",
    "SYS_FBT_INSDEL",
    "SYS_FILTER_ACLS",
    "SYS_GETTOKENID",
    "SYS_GETXTIVAL",
    "SYS_GET_ACLIDS",
    "SYS_GET_PRIVILEGES",
    "SYS_GUID",
    "SYS_MAKEXML",
    "SYS_MAKE_XMLNODEID",
    "SYS_MKXMLATTR",
    "SYS_OPTLOBPRBSC",
    "SYS_OPTXICMP",
    "SYS_OPTXQCASTASNQ",
    "SYS_OP_ADT2BIN",
    "SYS_OP_ADTCONS",
    "SYS_OP_ALSCRVAL",
    "SYS_OP_ATG",
    "SYS_OP_BIN2ADT",
    "SYS_OP_BITVEC",
    "SYS_OP_BL2R",
    "SYS_OP_BLOOM_FILTER",
    "SYS_OP_BLOOM_FILTER_LIST",
    "SYS_OP_C2C",
    "SYS_OP_CAST",
    "SYS_OP_CEG",
    "SYS_OP_CL2C",
    "SYS_OP_COMBINED_HASH",
    "SYS_OP_COMP",
    "SYS_OP_CONVERT",
    "SYS_OP_COUNTCHG",
    "SYS_OP_CSCONV",
    "SYS_OP_CSCONVTEST",
    "SYS_OP_CSR",
    "SYS_OP_CSX_PATCH",
    "SYS_OP_DECOMP",
    "SYS_OP_DESCEND",
    "SYS_OP_DISTINCT",
    "SYS_OP_DRA",
    "SYS_OP_DUMP",
    "SYS_OP_DV_CHECK",
    "SYS_OP_ENFORCE_NOT_NULL$",
    "SYS_OP_EXTRACT",
    "SYS_OP_GROUPING",
    "SYS_OP_GUID",
    "SYS_OP_IIX",
    "SYS_OP_ITR",
    "SYS_OP_LBID",
    "SYS_OP_LOBLOC2BLOB",
    "SYS_OP_LOBLOC2CLOB",
    "SYS_OP_LOBLOC2ID",
    "SYS_OP_LOBLOC2NCLOB",
    "SYS_OP_LOBLOC2TYP",
    "SYS_OP_LSVI",
    "SYS_OP_LVL",
    "SYS_OP_MAKEOID",
    "SYS_OP_MAP_NONNULL",
    "SYS_OP_MSR",
    "SYS_OP_NICOMBINE",
    "SYS_OP_NIEXTRACT",
    "SYS_OP_NII",
    "SYS_OP_NIX",
    "SYS_OP_NOEXPAND",
    "SYS_OP_NTCIMG$",
    "SYS_OP_NUMTORAW",
    "SYS_OP_OIDVALUE",
    "SYS_OP_OPNSIZE",
    "SYS_OP_PAR",
    "SYS_OP_PARGID",
    "SYS_OP_PARGID_1",
    "SYS_OP_PAR_1",
    "SYS_OP_PIVOT",
    "SYS_OP_R2O",
    "SYS_OP_RAWTONUM",
    "SYS_OP_RDTM",
    "SYS_OP_REF",
    "SYS_OP_RMTD",
    "SYS_OP_ROWIDTOOBJ",
    "SYS_OP_RPB",
    "SYS_OP_TOSETID",
    "SYS_OP_TPR",
    "SYS_OP_TRTB",
    "SYS_OP_UNDESCEND",
    "SYS_OP_VECAND",
    "SYS_OP_VECBIT",
    "SYS_OP_VECOR",
    "SYS_OP_VECXOR",
    "SYS_OP_VERSION",
    "SYS_OP_VREF",
    "SYS_OP_VVD",
    "SYS_OP_XMLCONS_FOR_CSX",
    "SYS_OP_XPTHATG",
    "SYS_OP_XPTHIDX",
    "SYS_OP_XPTHOP",
    "SYS_OP_XTXT2SQLT",
    "SYS_ORDERKEY_DEPTH",
    "SYS_ORDERKEY_MAXCHILD",
    "SYS_ORDERKEY_PARENT",
    "SYS_PARALLEL_TXN",
    "SYS_PATHID_IS_ATTR",
    "SYS_PATHID_IS_NMSPC",
    "SYS_PATHID_LASTNAME",
    "SYS_PATHID_LASTNMSPC",
    "SYS_PATH_REVERSE",
    "SYS_PXQEXTRACT",
    "SYS_RID_ORDER",
    "SYS_ROW_DELTA",
    "SYS_SC_2_XMLT",
    "SYS_SYNRCIREDO",
    "SYS_TYPEID",
    "SYS_UMAKEXML",
    "SYS_XMLANALYZE",
    "SYS_XMLCONTAINS",
    "SYS_XMLCONV",
    "SYS_XMLEXNSURI",
    "SYS_XMLGEN",
    "SYS_XMLINSTR",
    "SYS_XMLI_LOC_ISNODE",
    "SYS_XMLI_LOC_ISTEXT",
    "SYS_XMLLOCATOR_GETSVAL",
    "SYS_XMLNODEID",
    "SYS_XMLNODEID_GETCID",
    "SYS_XMLNODEID_GETLOCATOR",
    "SYS_XMLNODEID_GETOKEY",
    "SYS_XMLNODEID_GETPATHID",
    "SYS_XMLNODEID_GETPTRID",
    "SYS_XMLNODEID_GETRID",
    "SYS_XMLNODEID_GETSVAL",
    "SYS_XMLNODEID_GETTID",
    "SYS_XMLTRANSLATE",
    "SYS_XMLTYPE2SQL",
    "SYS_XMLT_2_SC",
    "SYS_XQBASEURI",
    "SYS_XQCASTABLEERRH",
    "SYS_XQCODEP2STR",
    "SYS_XQCODEPEQ",
    "SYS_XQCON2SEQ",
    "SYS_XQCONCAT",
    "SYS_XQDELETE",
    "SYS_XQDFLTCOLATION",
    "SYS_XQDOC",
    "SYS_XQDOCURI",
    "SYS_XQDURDIV",
    "SYS_XQED4URI",
    "SYS_XQENDSWITH",
    "SYS_XQERR",
    "SYS_XQERRH",
    "SYS_XQESHTMLURI",
    "SYS_XQEXLOBVAL",
    "SYS_XQEXSTWRP",
    "SYS_XQEXTRACT",
    "SYS_XQEXTRREF",
    "SYS_XQEXVAL",
    "SYS_XQFB2STR",
    "SYS_XQFNBOOL",
    "SYS_XQFNCMP",
    "SYS_XQFNDATIM",
    "SYS_XQFNLNAME",
    "SYS_XQFNNM",
    "SYS_XQFNNSURI",
    "SYS_XQFNPREDTRUTH",
    "SYS_XQFNQNM",
    "SYS_XQFNROOT",
    "SYS_XQFORMATNUM",
    "SYS_XQFTCONTAIN",
    "SYS_XQFUNCR",
    "SYS_XQGETCONTENT",
    "SYS_XQINDXOF",
    "SYS_XQINSERT",
    "SYS_XQINSPFX",
    "SYS_XQIRI2URI",
    "SYS_XQLANG",
    "SYS_XQLLNMFRMQNM",
    "SYS_XQMKNODEREF",
    "SYS_XQNILLED",
    "SYS_XQNODENAME",
    "SYS_XQNORMSPACE",
    "SYS_XQNORMUCODE",
    "SYS_XQNSP4PFX",
    "SYS_XQNSPFRMQNM",
    "SYS_XQPFXFRMQNM",
    "SYS_XQPOLYABS",
    "SYS_XQPOLYADD",
    "SYS_XQPOLYCEL",
    "SYS_XQPOLYCST",
    "SYS_XQPOLYCSTBL",
    "SYS_XQPOLYDIV",
    "SYS_XQPOLYFLR",
    "SYS_XQPOLYMOD",
    "SYS_XQPOLYMUL",
    "SYS_XQPOLYRND",
    "SYS_XQPOLYSQRT",
    "SYS_XQPOLYSUB",
    "SYS_XQPOLYUMUS",
    "SYS_XQPOLYUPLS",
    "SYS_XQPOLYVEQ",
    "SYS_XQPOLYVGE",
    "SYS_XQPOLYVGT",
    "SYS_XQPOLYVLE",
    "SYS_XQPOLYVLT",
    "SYS_XQPOLYVNE",
    "SYS_XQREF2VAL",
    "SYS_XQRENAME",
    "SYS_XQREPLACE",
    "SYS_XQRESVURI",
    "SYS_XQRNDHALF2EVN",
    "SYS_XQRSLVQNM",
    "SYS_XQRYENVPGET",
    "SYS_XQRYVARGET",
    "SYS_XQRYWRP",
    "SYS_XQSEQ2CON",
    "SYS_XQSEQ2CON4XC",
    "SYS_XQSEQDEEPEQ",
    "SYS_XQSEQINSB",
    "SYS_XQSEQRM",
    "SYS_XQSEQRVS",
    "SYS_XQSEQSUB",
    "SYS_XQSEQTYPMATCH",
    "SYS_XQSTARTSWITH",
    "SYS_XQSTATBURI",
    "SYS_XQSTR2CODEP",
    "SYS_XQSTRJOIN",
    "SYS_XQSUBSTRAFT",
    "SYS_XQSUBSTRBEF",
    "SYS_XQTOKENIZE",
    "SYS_XQTREATAS",
    "SYS_XQXFORM",
    "SYS_XQ_ASQLCNV",
    "SYS_XQ_ATOMCNVCHK",
    "SYS_XQ_NRNG",
    "SYS_XQ_PKSQL2XML",
    "SYS_XQ_UPKXML2SQL",
    "T",
    "TABLE",
    "TABLES",
    "TABLESPACE",
    "TABLESPACE_NO",
    "TABLE_LOOKUP_BY_NL",
    "TABLE_STATS",
    "TABNO",
    "TAN",
    "TANH",
    "TBL$OR$IDX$PART$NUM",
    "TEMPFILE",
    "TEMPLATE",
    "TEMPORARY",
    "TEMP_TABLE",
    "TEST",
    "THAN",
    "THE",
    "THEN",
    "THREAD",
    "THROUGH",
    "TIME",
    "TIMEOUT",
    "TIMES",
    "TIMESTAMP",
    "TIMEZONE_ABBR",
    "TIMEZONE_HOUR",
    "TIMEZONE_MINUTE",
    "TIMEZONE_OFFSET",
    "TIMEZONE_REGION",
    "TIME_ZONE",
    "TIV_GB",
    "TIV_SSF",
    "TO",
    "TOPLEVEL",
    "TO_BINARY_DOUBLE",
    "TO_BINARY_FLOAT",
    "TO_BLOB",
    "TO_CHAR",
    "TO_CLOB",
    "TO_DATE",
    "TO_DSINTERVAL",
    "TO_LOB",
    "TO_MULTI_BYTE",
    "TO_NCHAR",
    "TO_NCLOB",
    "TO_NUMBER",
    "TO_SINGLE_BYTE",
    "TO_TIME",
    "TO_TIMESTAMP",
    "TO_TIMESTAMP_TZ",
    "TO_TIME_TZ",
    "TO_YMINTERVAL",
    "TRACE",
    "TRACING",
    "TRACKING",
    "TRAILING",
    "TRANSACTION",
    "TRANSFORM_DISTINCT_AGG",
    "TRANSITION",
    "TRANSITIONAL",
    "TRANSLATE",
    "TREAT",
    "TRIGGER",
    "TRIGGERS",
    "TRIM",
    "TRUE",
    "TRUNC",
    "TRUNCATE",
    "TRUSTED",
    "TUNING",
    "TX",
    "TYPE",
    "TYPES",
    "TZ_OFFSET",
    "U",
    "UB2",
    "UBA",
    "UID",
    "UNARCHIVED",
    "UNBOUND",
    "UNBOUNDED",
    "UNDER",
    "UNDO",
    "UNDROP",
    "UNIFORM",
    "UNION",
    "UNIQUE",
    "UNISTR",
    "UNLIMITED",
    "UNLOAD",
    "UNLOCK",
    "UNNEST",
    "UNPACKED",
    "UNPIVOT",
    "UNPROTECTED",
    "UNQUIESCE",
    "UNRECOVERABLE",
    "UNRESTRICTED",
    "UNTIL",
    "UNUSABLE",
    "UNUSED",
    "UPDATABLE",
    "UPDATE",
    "UPDATED",
    "UPDATEXML",
    "UPD_INDEXES",
    "UPD_JOININDEX",
    "UPGRADE",
    "UPPER",
    "UPSERT",
    "UROWID",
    "USAGE",
    "USE",
    "USER",
    "USERENV",
    "USERGROUP",
    "USERS",
    "USER_DEFINED",
    "USER_RECYCLEBIN",
    "USE_ANTI",
    "USE_CONCAT",
    "USE_HASH",
    "USE_HASH_AGGREGATION",
    "USE_HASH_GBY_FOR_PUSHDOWN",
    "USE_INVISIBLE_INDEXES",
    "USE_MERGE",
    "USE_MERGE_CARTESIAN",
    "USE_NL",
    "USE_NL_WITH_INDEX",
    "USE_PRIVATE_OUTLINES",
    "USE_SEMI",
    "USE_STORED_OUTLINES",
    "USE_TTT_FOR_GSETS",
    "USE_WEAK_NAME_RESL",
    "USING",
    "VALIDATE",
    "VALIDATION",
    "VALUE",
    "VALUES",
    "VARCHAR",
    "VARCHAR2",
    "VARIANCE",
    "VARRAY",
    "VARRAYS",
    "VARYING",
    "VAR_POP",
    "VAR_SAMP",
    "VECTOR_READ",
    "VECTOR_READ_TRACE",
    "VERIFY",
    "VERSION",
    "VERSIONING",
    "VERSIONS",
    "VERSIONS_ENDSCN",
    "VERSIONS_ENDTIME",
    "VERSIONS_OPERATION",
    "VERSIONS_STARTSCN",
    "VERSIONS_STARTTIME",
    "VERSIONS_XID",
    "VIEW",
    "VIRTUAL",
    "VISIBLE",
    "VOLUME",
    "VSIZE",
    "WAIT",
    "WALLET",
    "WELLFORMED",
    "WHEN",
    "WHENEVER",
    "WHERE",
    "WHITESPACE",
    "WIDTH_BUCKET",
    "WITH",
    "WITHIN",
    "WITHOUT",
    "WORK",
    "WRAPPED",
    "WRITE",
    "XDB_FASTPATH_INSERT",
    "XID",
    "XML",
    "XMLATTRIBUTES",
    "XMLCAST",
    "XMLCDATA",
    "XMLCOLATTVAL",
    "XMLCOMMENT",
    "XMLCONCAT",
    "XMLDIFF",
    "XMLELEMENT",
    "XMLEXISTS",
    "XMLEXISTS2",
    "XMLFOREST",
    "XMLINDEX_REWRITE",
    "XMLINDEX_REWRITE_IN_SELECT",
    "XMLINDEX_SEL_IDX_TBL",
    "XMLISNODE",
    "XMLISVALID",
    "XMLNAMESPACES",
    "XMLPARSE",
    "XMLPATCH",
    "XMLPI",
    "XMLQUERY",
    "XMLROOT",
    "XMLSCHEMA",
    "XMLSERIALIZE",
    "XMLTABLE",
    "XMLTOOBJECT",
    "XMLTRANSFORM",
    "XMLTRANSFORMBLOB",
    "XMLTYPE",
    "XML_DML_RWT_STMT",
    "XPATHTABLE",
    "XS_SYS_CONTEXT",
    "X_DYN_PRUNE",
    "YEAR",
    "YES",
    "ZONE",
    "[",
    "]",
    "^",
    "{",
    "|",
    "}")

  

}
