package eanative.android.view;

import android.app.Activity;
import android.content.Context;
import android.graphics.Rect;
import android.media.AudioManager;
import android.media.MediaPlayer;

import android.os.Handler;
import android.os.Message;
import android.util.DisplayMetrics;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.ImageButton;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.scriptedpapers.mediabutton.MaterialPlayPauseButton;

import org.nexage.sourcekit.util.HttpTools;
import org.nexage.sourcekit.vast.VASTPlayer;
import org.nexage.sourcekit.vast.model.TRACKING_EVENTS_TYPE;
import org.nexage.sourcekit.vast.model.VASTModel;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

import eanative.android.R;
import eanative.android.util.Logger;



public class EAVastPlayerView extends RelativeLayout implements MediaPlayer.OnCompletionListener,
        MediaPlayer.OnErrorListener, MediaPlayer.OnPreparedListener, MediaPlayer.OnVideoSizeChangedListener,
        SurfaceHolder.Callback{

    private static String TAG = "VastTeaserView";
    private static boolean isVideoPlaying = false;
    // timer delays
    private static final long TOOLBAR_HIDE_DELAY = 3000;
    private static final long QUARTILE_TIMER_INTERVAL = 250;
    private static final long VIDEO_PROGRESS_TIMER_INTERVAL = 250;


    // timers
    private Timer mToolBarTimer;
    private Timer mTrackingEventTimer;
    private Timer mStartVideoProgressTimer;

    private LinkedList<Integer> mVideoProgressTracker = null;
    private final int mMaxProgressTrackingPoints = 20;

    private Handler mHandler;
    private Handler mViewPositionHandler;


    private VASTModel mVastModel = null;
    private VASTPlayer mVastPlayer;
    private HashMap<TRACKING_EVENTS_TYPE, List<String>> mTrackingEventMap;

    private MediaPlayer mMediaPlayer;
    private SurfaceView mSurfaceView;
    private SurfaceHolder mSurfaceHolder;

    private MaterialPlayPauseButton mPlayPauseButton;
    private ImageButton mCloseButton;

    private int mVideoHeight;
    private int mVideoWidth;
    private int mScreenWidth;
    private int mScreenHeight;
    private boolean mIsVideoPaused = false;
    private boolean mIsPlayBackError = false;
    private boolean mIsProcessedImpressions = false;
    private boolean mIsCompleted = false;
    private boolean mIsToolbarVisible = true;
    private boolean mDoNotConsiderTouchUp = false;
    private boolean autoPlay = true;
    private boolean manualUserInteraction = false;

    private int mCurrentVideoPosition;
    private int mQuartile = 0;

    private ProgressBar mProgressBar;
    private int volume = 1;
    private double classId = Math.random();
    private boolean mediaSourceSet;


    public EAVastPlayerView(Context context, VASTModel vastModel){
        super(context);

        this.mVastModel = vastModel;

        mTrackingEventMap = mVastModel.getTrackingUrls();

        this.createUIComponents();

    }
    protected void onMediaStart(){

    }
    protected void onMediaComplete(){

    }
    protected void onLayout(boolean changed, int left, int top, int right, int bottom){
        super.onLayout(changed, left, top, right, bottom);
        if(this.isPlaying()){
            this.mPlayPauseButton.setToPause();
        }else {
            this.mPlayPauseButton.setToPlay();
        }
    }
    @Override
    public void onVideoSizeChanged(MediaPlayer mp, int width, int height) {
        Logger.debug(
                "entered onVideoSizeChanged -- (MediaPlayer callback)");
        mVideoWidth = width;
        mVideoHeight = height;
        Logger.debug("video size: " + mVideoWidth + "x" + mVideoHeight);
    }
    @Override
    public void onCompletion(MediaPlayer mediaPlayer) {
        Logger.debug("entered onCOMPLETION -- (MediaPlayer callback)");
        stopVideoProgressTimer();
        stopViewPositionTimer();
        stopToolBarTimer();
        isVideoPlaying = false;
        mPlayPauseButton.setToPlay();
        if ( !mIsPlayBackError && !mIsCompleted) {
            mIsCompleted = true;
            this.processEvent(TRACKING_EVENTS_TYPE.complete);

        }
        this.onMediaComplete();

    }
    @Override
    public void surfaceChanged(SurfaceHolder surfaceHolder, int arg1, int arg2,
                               int arg3) {
        Logger.debug(
                "entered surfaceChanged -- (SurfaceHolder callback) %s", surfaceHolder == this.mSurfaceHolder);
    }

    @Override
    public void surfaceDestroyed(SurfaceHolder surfaceHolder) {
        Logger.debug(
                "entered surfaceDestroyed -- (SurfaceHolder callback)");
        //cleanUpMediaPlayer();

    }

    @Override
    public void onPrepared(MediaPlayer mp) {
        Logger.debug(
                "entered onPrepared called --(MediaPlayer callback) ....about to play %s %s",classId,isVideoPlaying);

        calculateAspectRatio();

        if(autoPlay) {
            Logger.debug("on prepared is view visibile %s",this.isViewVisible());
            if(this.isViewVisible()) {
                this.startMediaPlayer();
                if (mIsVideoPaused) {
                    mMediaPlayer.pause();
                    this.mPlayPauseButton.setToPlay();
                } else {
                    this.startVideoProgressTimer();
                    this.mPlayPauseButton.setToPause();
                }
            }else{
                mMediaPlayer.seekTo(1);
            }
            this.startViewPositionTimer();
        }else{
            this.mPlayPauseButton.setToPlay();
        }

        this.hideProgressBar();


        Logger.debug("current location in video:"
                + mCurrentVideoPosition);
        if (mCurrentVideoPosition > 0) {
            Logger.debug("seeking to location:"
                    + mCurrentVideoPosition);
            mMediaPlayer.seekTo(mCurrentVideoPosition);
        }

        startQuartileTimer();

        this.mPlayPauseButton.setVisibility(VISIBLE);

    }
    public void cleanPlayerUp(){
        mHandler = null;
        Logger.debug("entered cleanPlayerUp");

        this.cleanUpMediaPlayer();
        this.stopQuartileTimer();
        this.stopVideoProgressTimer();
        this.stopViewPositionTimer();
    }
    public void play(){
        this.processPlaySteps();
    }
    public void pause(){
        this.manualUserInteraction = true;
        this.stopViewPositionTimer();
        this.processPauseSteps();
    }
    public int getCurrentPosition(){
        if(this.mMediaPlayer !=null) {
            return this.mMediaPlayer.getCurrentPosition();
        }
        return 0;
    }
    public int getDuration(){
        if(this.mMediaPlayer != null) {
            return this.mMediaPlayer.getDuration();
        }
        return 0;
    }
    public boolean isPlaying(){
        if(this.mMediaPlayer !=null) {
            try{
                return this.mMediaPlayer.isPlaying();
            }catch (IllegalStateException e){};
        }
        return  false;
    }
    public void start(){
        this.startMediaPlayer();
    }
    private void createUIComponents() {

        this.createSurface();
        this.createMediaPlayer();

        this.createPlayPauseButton();
        this.createProgressBar();
    }
    protected void startMediaPlayer(){
        if(!this.isPlaying()) {
            mMediaPlayer.start();
            isVideoPlaying = true;
            this.onMediaStart();
            this.mMediaPlayer.setVolume(this.volume, this.volume);
            if (!mIsProcessedImpressions) {
                this.processImpressions();
            }
        }
    }
    private void createProgressBar() {
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT,
                RelativeLayout.LayoutParams.WRAP_CONTENT);
        params.addRule(RelativeLayout.CENTER_IN_PARENT);

        mProgressBar = new ProgressBar(this.getContext());
        mProgressBar.setLayoutParams(params);

        this.addView(mProgressBar);
        mProgressBar.setVisibility(View.GONE);
    }

    private void showProgressBar() {
        mProgressBar.setVisibility(View.VISIBLE);
    }

    private void hideProgressBar() {
        mProgressBar.setVisibility(View.GONE);
    }

    private void createSurface() {

        mSurfaceView = new SurfaceView(this.getContext());

        mSurfaceHolder = mSurfaceView.getHolder();
        mSurfaceHolder.addCallback(this);
        this.addView(mSurfaceView);
    }

    private void createMediaPlayer() {

        mMediaPlayer = new MediaPlayer();
        mMediaPlayer.setOnCompletionListener(this);
        mMediaPlayer.setOnErrorListener(this);
        mMediaPlayer.setOnPreparedListener(this);
        mMediaPlayer.setOnVideoSizeChangedListener(this);
        mMediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);
        this.mediaSourceSet = false;

    }
    private void createPlayPauseButton() {

        mPlayPauseButton = new MaterialPlayPauseButton(this.getContext());
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                (int) getResources().getDimension(R.dimen.video_play_pause_width),
                (int) getResources().getDimension(R.dimen.video_play_pause_height));
        params.addRule(RelativeLayout.CENTER_IN_PARENT, RelativeLayout.TRUE);

        mPlayPauseButton.setLayoutParams(params);
        mPlayPauseButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                playPauseButtonClicked();
            }

        });
        this.addView(mPlayPauseButton);
    }
    public void activateButtons(boolean active) {
        mIsToolbarVisible = active;
        this.mPlayPauseButton.animate().alpha(active? 1:0);
    }
    public void closeStreaming() {

        Logger.debug("entered closestreaming()");
        this.cleanPlayerUp();

        if (!mIsPlayBackError) {
            this.processEvent(TRACKING_EVENTS_TYPE.close);
        }

        //finishVAST();

        Logger.debug("leaving closeClicked()");
    }

    private void playPauseButtonClicked() {
        this.manualUserInteraction = true;
        Logger.debug("entered playPauseClicked");
        if(mMediaPlayer==null) {
            Logger.debug( "mMediaPlayer is null when playPauseButton was clicked");
            return;
        }
        boolean isPlaying = mMediaPlayer.isPlaying();
        Logger.debug("isPlaying:" + isPlaying);

        if (isPlaying) {
            //pause
            this.pause();

        } else if (mIsVideoPaused) {
            //play
            this.play();
            if(!mIsCompleted) {
                this.processEvent(TRACKING_EVENTS_TYPE.resume);
            }
        } else {
            //replay
            this.processPlaySteps();
            mQuartile = 0;
            this.startQuartileTimer();
        }
    }

    private void processPauseSteps() {
        mIsVideoPaused = true;
        isVideoPlaying = false;
        if(mMediaPlayer != null) {
            mMediaPlayer.pause();
        }
        Logger.debug("entered process pause steps");

        this.stopVideoProgressTimer();
        this.stopToolBarTimer();
        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                mPlayPauseButton.setToPlay();
            }
        }, 10);

        if(!mIsCompleted) {
            this.processEvent(TRACKING_EVENTS_TYPE.pause);
        }
    }

    private void processPlaySteps() {
        mIsVideoPaused = false;
        isVideoPlaying = true;
        this.startMediaPlayer();
        mPlayPauseButton.setToPause();

        this.startVideoProgressTimer();
    }

    /*
    @Override
    public void onBackPressed() {
        Logger.debug("entered onBackPressed");
        this.closeClicked();

    }*/

    public void surfaceCreated(SurfaceHolder holder) {
        Logger.debug("surfaceCreated -- (SurfaceHolder callback)");
        try {
            if(mMediaPlayer==null) {
                this.showProgressBar();
                createMediaPlayer();
            }
            mMediaPlayer.setDisplay(mSurfaceHolder);

            if(this.mediaSourceSet != true) {
                String url = mVastModel.getPickedMediaFileURL();
                Logger.debug("URL for media file:" + url);
                mMediaPlayer.setDataSource(url);
                mMediaPlayer.prepareAsync();
                this.mediaSourceSet = true;
            }else{
                if(this.autoPlay == true){
                    this.play();
                }else {
                    this.activateButtons(true);
                }
            }
        } catch (Exception e) {
            Logger.debug("surface created error: %s", e.getMessage());
        }
    }


    private void calculateAspectRatio() {
        Logger.debug("entered calculateAspectRatio");

        if ( mVideoWidth == 0 || mVideoHeight == 0 ) {
            Logger.debug("mVideoWidth or mVideoHeight is 0, skipping calculateAspectRatio");
            return;
        }

        DisplayMetrics displayMetrics = this.getResources()
                .getDisplayMetrics();

        /*mScreenWidth = displayMetrics.widthPixels;
        mScreenHeight = displayMetrics.heightPixels;
*/
        mScreenWidth = this.getWidth();
        mScreenHeight = this.getHeight();

        Logger.debug("calculating aspect ratio");
        double widthRatio = 1.0 * mScreenWidth / mVideoWidth;
        double heightRatio = 1.0 * mScreenHeight / mVideoHeight;

        double scale = Math.min(widthRatio, heightRatio);

        int surfaceWidth = (int) (scale * mVideoWidth);
        int surfaceHeight = (int) (scale * mVideoHeight);

        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                surfaceWidth, surfaceHeight);
        params.addRule(RelativeLayout.CENTER_IN_PARENT);
        mSurfaceView.setLayoutParams(params);

        mSurfaceHolder.setFixedSize(surfaceWidth, surfaceHeight);

        Logger.debug(" screen size: " + mScreenWidth + "x" + mScreenHeight);
        Logger.debug(" video size:  " + mVideoWidth + "x" + mVideoHeight);
        Logger.debug(" widthRatio:   " + widthRatio);
        Logger.debug(" heightRatio:   " + heightRatio);

        Logger.debug("surface size: " + surfaceWidth + "x" + surfaceHeight);

    }

    private void cleanUpMediaPlayer() {

        Logger.debug("entered cleanUpMediaPlayer ");

        if (mMediaPlayer != null) {

            if (this.isPlaying()) {
                isVideoPlaying = false;
                mMediaPlayer.stop();
            }

            mMediaPlayer.setOnCompletionListener(null);
            mMediaPlayer.setOnErrorListener(null);
            mMediaPlayer.setOnPreparedListener(null);
            mMediaPlayer.setOnVideoSizeChangedListener(null);

            mMediaPlayer.release();
            mMediaPlayer = null;
        }
    }

    @Override
    public boolean onError(MediaPlayer mp, int what, int extra) {
        Logger.debug( "entered onError -- (MediaPlayer callback)");
        mIsPlayBackError = true;
        Logger.debug( "Shutting down Activity due to Media Player errors: WHAT:" + what +": EXTRA:" + extra+":");

        processErrorEvent();
        this.closeStreaming();

        return true;
    }

    private void processErrorEvent() {
        Logger.debug("entered processErrorEvent");

        List<String> errorUrls = mVastModel.getErrorUrl();
        fireUrls(errorUrls);

    }


    private void processImpressions() {
        Logger.debug("entered processImpressions");

        mIsProcessedImpressions = true;
        List<String> impressions = mVastModel.getImpressions();
        fireUrls(impressions);

    }

    private void fireUrls(List<String> urls) {
        Logger.debug("entered fireUrls");

        if (urls != null) {

            for (String url : urls) {
                Logger.debug("\tfiring url:%s", url);
                HttpTools.httpGetURL(url);
            }

        }else {
            Logger.debug("\turl list is null");
        }



    }


    private void startQuartileTimer() {
        Logger.debug("entered startQuartileTimer");
        stopQuartileTimer();

        if(mIsCompleted) {
            Logger.debug("ending quartileTimer because the video has been replayed");
            return;
        }

        final int videoDuration = mMediaPlayer.getDuration();

        mTrackingEventTimer = new Timer();
        mTrackingEventTimer.scheduleAtFixedRate(new TimerTask() {

            @Override
            public void run() {
                int percentage = 0;
                if(mMediaPlayer != null) {
                    try {
                        int curPos = mMediaPlayer.getCurrentPosition();
                        // wait for the video to really start
                        if (curPos == 0) {
                            return;
                        }
                        percentage = 100 * curPos / videoDuration;
                    } catch (Exception e) {
                        Logger.error(
                                "mediaPlayer.getCurrentPosition exception: %s", e.getMessage());
                        this.cancel();
                        return;
                    }
                }else{
                    return;
                }

                if (percentage >= 25 * mQuartile) {
                    if (mQuartile == 0) {
                        Logger.debug("Video at start: (%s%%)",percentage);
                        processEvent(TRACKING_EVENTS_TYPE.start);
                    } else if (mQuartile == 1) {
                        Logger.debug("Video at first quartile: (%s%%)",percentage );
                        processEvent(TRACKING_EVENTS_TYPE.firstQuartile);
                    } else if (mQuartile == 2) {
                        Logger.debug("Video at midpoint: (%s%%)",percentage);
                        processEvent(TRACKING_EVENTS_TYPE.midpoint);
                    } else if (mQuartile == 3) {
                        Logger.debug("Video at third quartile: (%s%%)",percentage);
                        processEvent(TRACKING_EVENTS_TYPE.thirdQuartile);
                        stopQuartileTimer();
                    }
                    mQuartile++;
                }
            }

        }, 0, QUARTILE_TIMER_INTERVAL);
    }

    private void stopQuartileTimer() {

        if (mTrackingEventTimer != null) {
            mTrackingEventTimer.cancel();
            mTrackingEventTimer = null;
        }
    }

    private void startVideoProgressTimer() {
        Logger.debug("entered startVideoProgressTimer");

        mStartVideoProgressTimer = new Timer();
        mVideoProgressTracker = new LinkedList<Integer>();

        mStartVideoProgressTimer.schedule(new TimerTask() {
            int maxAmountInList = mMaxProgressTrackingPoints - 1;

            @Override
            public void run() {

                if (mMediaPlayer == null) {
                    return;
                }
                if (mVideoProgressTracker.size() == maxAmountInList) {
                    int firstPosition = mVideoProgressTracker.getFirst();
                    int lastPosition = mVideoProgressTracker.getLast();

                    if (lastPosition > firstPosition) {
                        Logger.verbose("video progressing (position:" + lastPosition + ") %s",classId);
                        mVideoProgressTracker.removeFirst();
                    } else {
                        Logger.debug( "detected video hang");
                        mIsPlayBackError = true;
                        stopVideoProgressTimer();
                        processErrorEvent();
                        closeStreaming();
                        //finishVAST();
                        //finishVAST();
                    }
                }

                try {
                    int curPos = mMediaPlayer.getCurrentPosition();
                    mVideoProgressTracker.addLast(curPos);
                } catch (Exception e) {
                    // occasionally the timer is in the middle of processing and
                    // the media player is cleaned up
                }
            }

        }, 0, VIDEO_PROGRESS_TIMER_INTERVAL);

    }
    private void stopToolBarTimer() {
        Logger.debug("entered stopToolBarTimer");
        if (mToolBarTimer != null) {
            mToolBarTimer.cancel();
            mToolBarTimer = null;
        }
    }

    private void stopVideoProgressTimer() {
        Logger.debug("entered stopVideoProgressTimer");

        if (mStartVideoProgressTimer != null) {
            mStartVideoProgressTimer.cancel();
        }
    }
    private void startViewPositionTimer() {
        Logger.debug("entered startViewPositionTimer,%s",this.classId);

        mViewPositionHandler = new Handler();


        mViewPositionHandler.postDelayed(checkIfViewIsVisibleRunnable, VIDEO_PROGRESS_TIMER_INTERVAL);

    }
    Runnable checkIfViewIsVisibleRunnable = new Runnable() {
        @Override
        public void run() {
            mViewPositionHandler.postDelayed(this, VIDEO_PROGRESS_TIMER_INTERVAL);
            checkIfViewIsVisible();
        }
    };
    private void stopViewPositionTimer() {
        Logger.debug("entered stopViewPositionTimer");

        if (mViewPositionHandler != null) {
            mViewPositionHandler.removeCallbacks(checkIfViewIsVisibleRunnable);
        }
    }

    private boolean isViewVisible() {
        if(this.isShown()) {
            Rect visibleRect = new Rect();
            Boolean isVisible = this.getGlobalVisibleRect(visibleRect);
            if (!isVisible) {
                return false;
            }
            Rect viewRect = new Rect();
            this.getDrawingRect(viewRect);

            return visibleRect.width() * visibleRect.height() > viewRect.width() * viewRect.height() * .90;
        }
        return false;
    }
    private void checkIfViewIsVisible() {

        if(!this.isViewVisible()){
            if(this.isPlaying() == true) {
                processPauseSteps();

            }
        }else if(this.autoPlay == true &&
                this.isPlaying() == false &&
                this.manualUserInteraction == false && isVideoPlaying == false){
            processPlaySteps();
        }

    }

    private void processEvent(TRACKING_EVENTS_TYPE eventName) {
        Logger.debug("entered Processing Event: " + eventName);
        List<String> urls = (List<String>) mTrackingEventMap.get(eventName);

        fireUrls(urls);

    }

    public boolean isAutoPlay() {
        return autoPlay;
    }

    public void setAutoPlay(boolean autoPlay) {
        this.autoPlay = autoPlay;
    }

    public void setVolume(int volume) {
        this.volume = volume;
        this.mMediaPlayer.setVolume(volume,volume);
    }
}
