package eanative.android.model;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import com.android.volley.ParseError;
import com.android.volley.Request.Method;
import com.android.volley.RequestQueue;
import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.StringRequest;
import com.android.volley.toolbox.Volley;



import eanative.android.R;
import eanative.android.ad.AdData;
import eanative.android.events.EventDispatcher;
import eanative.android.util.EANSettings;
import eanative.android.util.GPSTracker;
import eanative.android.util.Logger;
import eanative.android.util.Utils;
import android.content.Context;
import android.content.res.XmlResourceParser;
import android.os.Handler;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.webkit.WebView;


public class EANativeServerClient extends EventDispatcher {
	private Context appContext;
	

	private RequestQueue requestQueue;
	private static final String LIVE_AD_SERVER_URL = "http://serve.elasticad.net/mserve.php?";
	private static final String DEV_AD_SERVER_URL = "http://dev.serve.elasticad.net/mserve.php?";
    private static final String PRE_LIVE_AD_SERVER_URL = "http://serve.elasticad.net/mservetest.php?";

	private boolean isActive;

	
	private ArrayList<String> testDevices;

	public EANativeServerClient(Context appContext) {

		this.appContext = appContext;
		this.testDevices = new ArrayList<String>();

		TelephonyManager tm = (TelephonyManager) appContext
				.getSystemService(Context.TELEPHONY_SERVICE);

		String networkOperator = tm.getNetworkOperator();
		if (networkOperator != null && networkOperator.length() >= 6) {
			EANSettings.getSettings().mcc = networkOperator.substring(0, 3);
			EANSettings.getSettings().mnc = networkOperator.substring(3);
		}
		EANSettings.getSettings().marketAppId = appContext.getPackageName();
		EANSettings.getSettings().marketAppVersion = EANSettings.getMarketAppVersion(appContext,
				EANSettings.getSettings().marketAppId);
		EANSettings.getSettings().sdkVersion = appContext.getResources().getString(
				R.string.eanative_version);
		EANSettings.getSettings().carrier = EANSettings.getCarrier(appContext);
		EANSettings.getSettings().ua = new WebView(appContext).getSettings().getUserAgentString();
		EANSettings.getSettings().currentLocation = new GPSTracker(this.appContext);
		EANSettings.getSettings().ip =  Utils.getIPAddress(true);
		EANSettings.getAAID(appContext);
				
		this.requestQueue = Volley.newRequestQueue(appContext);
		
		Logger.info(
				"The EANative SDK was initialized. Use EANative.setTestMode(true, \"%s\"); to enable Test mode.",
				this.getDeviceID());

	}

	public void setApplicationKey(String nativeAppKey) {
		EANSettings.getSettings().nativeAppKey = nativeAppKey;
	}

	public void setActive(boolean state) {
		this.isActive = state;
		if (state) {
			this.requestQueue.start();
		} else {
			this.requestQueue.stop();
		}

	}
	
	public JsonObjectRequest requestAds(
			final EANativePlacementConfig placement,
			final Response.Listener<AdData> listener,
			final Response.ErrorListener errorListener) {
		if (this.isActive == false) {
			return null;
		}
		final Handler handler = new Handler();
		
		Response.ErrorListener wrappedErrorListener = new Response.ErrorListener() {
			public void onErrorResponse(final VolleyError error) {
				handler.post(new Runnable() {
					public void run() {
						defaultErrorListener
								.onErrorResponse(error);
						errorListener.onErrorResponse(error);
					}
				});
			}
		};

		Response.Listener<JSONObject> wrappedListener = new Response.Listener<JSONObject>() {
			public void onResponse(JSONObject json) {
				AdData ad = null;
				try {
					ad = adResponseFromJson(json);
				} catch (ParseError parseError) {
					errorListener.onErrorResponse(parseError);
					return;
				} catch (JSONException e) {
					// errorListener.onErrorResponse();
				}
				if (ad != null){
					listener.onResponse(ad);
				}else{
					String error = "";
					try {
						error = json.getString("error");
					} catch (JSONException e) {
						
					}
					if(error !=""){
						Logger.warn("ERROR while trying to get ads: %s", error);
					}
					
				}
			}
		};

		// get used params
		ArrayList<String> tags = this
				.getTagsFromXML(placement.layoutResourceId);

		Map<String, String> params = new HashMap<String, String>();
		params.put("pid", placement.placementId);
		params.put("appid", urlEncodeParam(EANSettings.getSettings().marketAppId));
		params.put("nativeAppKey", urlEncodeParam(EANSettings.getSettings().nativeAppKey));
		params.put("marketAppVersion", EANSettings.getSettings().marketAppVersion);
		params.put("sdkVersion", EANSettings.getSettings().sdkVersion);
		params.put("ua", urlEncodeParam(EANSettings.getSettings().ua));
		params.put("carrier", urlEncodeParam(EANSettings.getSettings().carrier));
		params.put("loc", EANSettings.getSettings().getLocation());
		params.put("ip",EANSettings.getSettings().ip);
		params.put("os", urlEncodeParam("android " + android.os.Build.VERSION.SDK_INT));
		params.put("tags", TextUtils.join(",", tags));
		params.put("mcc", EANSettings.getSettings().mcc);
		params.put("mnc", EANSettings.getSettings().mnc);
		params.put("aaid", urlEncodeParam(EANSettings.getSettings().androidAdvertisingID));
		params.put("dnt", EANSettings.getSettings().isLimitAdTrackingEnabled?"true":"false");
		params.put("devmake", urlEncodeParam(EANSettings.getSettings().deviceMake));
		params.put("devmodel", urlEncodeParam(EANSettings.getSettings().deviceModel));
		params.put("test_hash", placement.testHash);
		if (this.isDeviceInTestMode()) {
			params.put("istest", "true");
		}

        String url = LIVE_AD_SERVER_URL;
        if(placement.environment == "DEVELOPMENT_MODE"){
            url = DEV_AD_SERVER_URL;
        }else if(placement.environment == "PRE_LIVE_MODE"){
            url = PRE_LIVE_AD_SERVER_URL;
        }

        for (String key : params.keySet()) {
			url += "&" + key + "=" + params.get(key);
		}
		if(placement.queryStringParams !=null && !placement.queryStringParams.isEmpty()) {
			JSONObject mJSONArray = new JSONObject(placement.queryStringParams);
			url += "&query-string=" + urlEncodeParam(mJSONArray.toString());
		}
		JsonObjectRequest request = new JsonObjectRequest(Method.GET, url,
				wrappedListener, wrappedErrorListener);

		this.requestQueue.add(request);
		
		Logger.debug("MSERVE URL %s", url);
		return request;
	}
	
	public void trackEvent(String url) {
		Response.Listener<String> wrappedListener = new Response.Listener<String>() {
			public void onResponse(String data) {

			}
		};
		//Logger.info("TRACK URL: %s", url);
		StringRequest request = new StringRequest(url, wrappedListener,
				defaultErrorListener);
		this.requestQueue.add(request);
	}

	private final Response.ErrorListener defaultErrorListener = new Response.ErrorListener() {
		public void onErrorResponse(VolleyError error) {
			Logger.error("Network error on request: %s " , error.getMessage());
		}
	};

	private String urlEncodeParam(String param) {
		if(param !=null) {
			try {
				return URLEncoder.encode(param, "UTF-8").replace("+", "%20");
			} catch (UnsupportedEncodingException e) {};

		}
		return param;
	}
	private AdData adResponseFromJson(JSONObject json) throws ParseError,
			JSONException {
		if (json.getString("status").indexOf("error") == -1)
			return adFromJson(json);
		return null;
	}

	private AdData adFromJson(JSONObject json) {
		return new AdData(json);
	}

	private ArrayList<String> getTagsFromXML(int layoutResourceId) {
		ArrayList<String> tags = new ArrayList<String>();

		try {
			XmlResourceParser xpp = this.appContext.getResources().getLayout(
					layoutResourceId);

			try {
				xpp.next();

				int eventType = xpp.getEventType();
				while (eventType != XmlPullParser.END_DOCUMENT) {
					if (eventType == XmlPullParser.START_TAG) {
						String tagName = xpp.getAttributeValue("http://schemas.android.com/apk/res/android", "tag");
						if (tagName != null && tagName!="" && tagName.indexOf("eanative_tag_") != -1) {
							tags.add(tagName.substring(13));
						}
					}
					eventType = xpp.next();
				}
			} catch (XmlPullParserException e) {
			}
			;
			xpp.close();
		} catch (IOException io) {

		}
		return tags;
	}

	public void setTestDevices(boolean testMode, String deviceId) {
		if (testMode) {
			testDevices.add(deviceId);
		} else {
			testDevices.remove(deviceId);
		}
	}

	protected boolean isDeviceInTestMode() {

		if (this.testDevices.indexOf(this.getDeviceID()) != -1) {
			return true;
		}
		return false;
	}

	protected String getDeviceID() {
		return Settings.Secure.getString(this.appContext.getContentResolver(),
				Settings.Secure.ANDROID_ID);
	}
	

}
