package eanative.android;

import android.app.Activity;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.content.res.Configuration;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.Handler;
import android.os.Bundle;
import android.util.DisplayMetrics;
import android.view.MotionEvent;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.ImageButton;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.scriptedpapers.mediabutton.MaterialPlayPauseButton;

import org.nexage.sourcekit.util.HttpTools;
import org.nexage.sourcekit.vast.VASTPlayer;
import org.nexage.sourcekit.vast.model.TRACKING_EVENTS_TYPE;
import org.nexage.sourcekit.vast.model.VASTModel;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;


import eanative.android.util.Logger;

import static android.view.View.GONE;
import static android.view.View.VISIBLE;

public class VastTeaserActivity extends Activity implements MediaPlayer.OnCompletionListener,
        MediaPlayer.OnErrorListener, MediaPlayer.OnPreparedListener, MediaPlayer.OnVideoSizeChangedListener,
        SurfaceHolder.Callback{

    private static String TAG = "VASTActivity";

    // timer delays
    private static final long TOOLBAR_HIDE_DELAY = 3000;
    private static final long QUARTILE_TIMER_INTERVAL = 250;
    private static final long VIDEO_PROGRESS_TIMER_INTERVAL = 250;


    // timers
    private Timer mToolBarTimer;
    private Timer mTrackingEventTimer;
    private Timer mStartVideoProgressTimer;
    private Handler durationHandler = new Handler();

    private LinkedList<Integer> mVideoProgressTracker = null;
    private final int mMaxProgressTrackingPoints = 20;

    private Handler mHandler;

    private VASTModel mVastModel = null;
    private HashMap<TRACKING_EVENTS_TYPE, List<String>> mTrackingEventMap;

    private MediaPlayer mMediaPlayer;
    private SurfaceView mSurfaceView;
    private SurfaceHolder mSurfaceHolder;
    private RelativeLayout mOverlay;
    private RelativeLayout mButtonsView;
    private RelativeLayout mRootLayout;

    private TextView mInfoButton;
    private TextView mCountdown;
    private MaterialPlayPauseButton mPlayPauseButton;
    private ImageButton mCloseButton;

    private int mVideoHeight;
    private int mVideoWidth;
    private int mScreenWidth;
    private int mScreenHeight;
    private boolean mIsVideoPaused = false;
    private boolean mIsPlayBackError = false;
    private boolean mIsProcessedImpressions = false;
    private boolean mIsCompleted = false;
    private boolean mIsToolbarVisible = true;
    private boolean mDoNotConsiderTouchUp = false;

    private int mCurrentVideoPosition;
    private int mQuartile = 0;

    private ProgressBar mProgressBar;
    private String elapsedTimeText;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        int currentOrientation = this.getResources().getConfiguration().orientation;
        Logger.info("currentOrientation:" + currentOrientation);

        if (currentOrientation != Configuration.ORIENTATION_LANDSCAPE) {
            Logger.info(
                    "Orientation is not landscape.....forcing landscape");
            setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE);

        }
        // hide title bar of application
        requestWindowFeature(Window.FEATURE_NO_TITLE);
        // hide status bar of Android
        getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,
                WindowManager.LayoutParams.FLAG_FULLSCREEN);

        setContentView(R.layout.activity_vast_teaser);

        Intent i = getIntent();
        mVastModel = (VASTModel) i
                .getSerializableExtra("eanative.android.vast.player.vastModel");

        elapsedTimeText = getIntent().getStringExtra("eanative.android.placement.ett");

        if (mVastModel == null) {
            Logger.info("vastModel is null. Stopping activity.");
            finishVAST();
        } else {

            mHandler = new Handler();

            mTrackingEventMap = mVastModel.getTrackingUrls();
            createUIComponents();

            hideSystemUI();
        }


    }

    @Override
    protected void onStart() {
        Logger.info("entered onStart --(life cycle event)");
        super.onStart();

    }

    @Override
    protected void onResume() {
        Logger.info("entered on onResume --(life cycle event)");
        super.onResume();

    }

    @Override
    protected void onStop() {
        Logger.info("entered on onStop --(life cycle event)");
        super.onStop();

    }

    @Override
    protected void onRestart() {
        Logger.info("entered on onRestart --(life cycle event)");
        super.onRestart();
        createMediaPlayer();

    }

    @Override
    protected void onPause() {
        Logger.info("entered on onPause --(life cycle event)");
        super.onPause();

        if (mMediaPlayer != null) {
            mCurrentVideoPosition = mMediaPlayer.getCurrentPosition();
        }
        cleanActivityUp();
    }

    @Override
    protected void onDestroy() {
        Logger.info("entered on onDestroy --(life cycle event)");
        super.onDestroy();

    }

    @Override
    public void onSaveInstanceState(Bundle savedInstanceState) {

        Logger.info("entered onSaveInstanceState ");
        super.onSaveInstanceState(savedInstanceState);
    }

    @Override
    public void onRestoreInstanceState(Bundle savedInstanceState) {
        Logger.info("in onRestoreInstanceState");
        super.onRestoreInstanceState(savedInstanceState);

    }
    public void onWindowFocusChanged(boolean hasFocus) {

        super.onWindowFocusChanged(hasFocus);

        if(hasFocus) {
            hideSystemUI();
        }
    }
    // This snippet hides the system bars.
    private void hideSystemUI() {
        // Set the IMMERSIVE flag.
        // Set the content to appear under the system bars so that the content
        // doesn't resize when the system bars hide and show.
        getWindow().getDecorView().setSystemUiVisibility(
                View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                        | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                        | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                        | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION // hide nav bar
                        | View.SYSTEM_UI_FLAG_FULLSCREEN // hide status bar
                        | View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY// hide status bar
                        | View.SYSTEM_UI_FLAG_IMMERSIVE);
    }

    // This snippet shows the system bars. It does this by removing all the flags
// except for the ones that make the content appear under the system bars.
    private void showSystemUI() {
        getWindow().getDecorView().setSystemUiVisibility(
                View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                        | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                        | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN);
    }
    private void createUIComponents() {

        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT,
                RelativeLayout.LayoutParams.MATCH_PARENT);

        this.mRootLayout = (RelativeLayout) findViewById(R.id.mRootLayout);

        this.createSurface();
        this.createMediaPlayer();
        this.createOverlay();


        this.createPlayPauseButton();

        this.createInfoButton();

        this.setContentView(mRootLayout);

        this.createCloseButton();
        //this.createProgressDialog();
        this.createProgressBar();

    }

    private void createProgressBar() {
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT,
                RelativeLayout.LayoutParams.WRAP_CONTENT);
        params.addRule(RelativeLayout.CENTER_IN_PARENT);

        mProgressBar = new ProgressBar(this);
        mProgressBar.setLayoutParams(params);

        mRootLayout.addView(mProgressBar);
        mProgressBar.setVisibility(View.GONE);
    }

    private void showProgressBar() {
        mProgressBar.setVisibility(View.VISIBLE);
    }

    private void hideProgressBar() {
        mProgressBar.setVisibility(View.GONE);
    }


    @SuppressWarnings("deprecation")
    private void createSurface() {

        mSurfaceView = (SurfaceView) findViewById(R.id.mSurfaceView);

        mSurfaceHolder = mSurfaceView.getHolder();
        mSurfaceHolder.addCallback(this);

        //mSurfaceHolder.setType(SurfaceHolder.SURFACE_TYPE_PUSH_BUFFERS);
    }

    private void createMediaPlayer() {

        mMediaPlayer = new MediaPlayer();
        mMediaPlayer.setOnCompletionListener(this);
        mMediaPlayer.setOnErrorListener(this);
        mMediaPlayer.setOnPreparedListener(this);
        mMediaPlayer.setOnVideoSizeChangedListener(this);
        mMediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);

    }
    private void createCloseButton() {

        mCloseButton = (ImageButton) findViewById(R.id.mCloseButton);
        mCloseButton.setImageResource(R.drawable.arrow_back);
        mCloseButton.setVisibility(VISIBLE);
        mCloseButton.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                if (event.getActionMasked() == MotionEvent.ACTION_UP){
                    closeClicked();
                }
                return true;
            }
        });
        mCloseButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                closeClicked();
            }
        });
    }

    private void createOverlay() {

        mOverlay = (RelativeLayout) findViewById(R.id.mOverlay);
        mOverlay.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {

                if (event.getActionMasked() == MotionEvent.ACTION_DOWN){
                    mDoNotConsiderTouchUp = !mIsToolbarVisible;
                    if(!mIsToolbarVisible){
                        showToolBar();
                    }
                }else{
                    if(!mDoNotConsiderTouchUp) {
                        hideToolBar();
                    }
                }
                return true;
            }
        });
        mButtonsView = (RelativeLayout) findViewById(R.id.mButtonsView);
    }

    private void createInfoButton() {

        String callToActionText = getIntent().getStringExtra("eanative.android.placement.cta");


        String clickThroughUrl = mVastModel.getVideoClicks().getClickThrough();

        mInfoButton = (TextView) findViewById(R.id.mInfoButton);
        mInfoButton.setText(callToActionText);
        if (clickThroughUrl != null && clickThroughUrl.length() > 0) {

            mInfoButton.setVisibility(VISIBLE);
            mInfoButton.setOnTouchListener(new View.OnTouchListener() {
                @Override
                public boolean onTouch(View v, MotionEvent event) {
                    if (event.getActionMasked() == MotionEvent.ACTION_UP){
                        infoClicked();
                    }
                    return true;
                }
            });
            mInfoButton.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    infoClicked();
                }
            });


        }else{
            mInfoButton.setVisibility(GONE);
        }

        mCountdown = (TextView) findViewById(R.id.mCountdown);
    }


    private void createPlayPauseButton() {

        mPlayPauseButton = (MaterialPlayPauseButton)findViewById(R.id.mPlayPauseButton);
        mPlayPauseButton.setToPause();
        mPlayPauseButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                playPauseButtonClicked();
            }

        });
    }

    private void infoClicked() {
        Logger.info("entered infoClicked:");

        activateButtons(false);

        boolean isPlaying = mMediaPlayer.isPlaying();

        if (isPlaying) {
            mMediaPlayer.pause();
            mCurrentVideoPosition = mMediaPlayer.getCurrentPosition();
        }

        processClickThroughEvent();

    }

    private void activateButtons(boolean active) {
        Logger.info("entered activateButtons:");

        mIsToolbarVisible = active;
        if (active) {
            mButtonsView.animate().alpha(1.0f);
        } else {
            mButtonsView.animate().alpha(0.0f);
        }


    }
    private void processClickThroughEvent() {
        Logger.info("entered processClickThroughEvent:");

        if(VASTPlayer.listener!=null) {
            VASTPlayer.listener.vastClick();
        }

        String clickThroughUrl = mVastModel.getVideoClicks().getClickThrough();
        Logger.info("clickThrough url: %s" , clickThroughUrl);


        // Before we send the app to the click through url, we will process ClickTracking URL's.
        List<String> urls = mVastModel.getVideoClicks().getClickTracking();
        fireUrls(urls);

        // Navigate to the click through url
        try {
            Uri uri = Uri.parse(clickThroughUrl);
            Intent intent = new Intent(Intent.ACTION_VIEW, uri);
            ResolveInfo resolvable = getPackageManager().resolveActivity(intent, PackageManager.GET_INTENT_FILTERS);
            if(resolvable == null) {
                Logger.info( "Clickthrough error occured, uri unresolvable");
                if (mCurrentVideoPosition>=mMediaPlayer.getCurrentPosition()*0.99) {
                    mMediaPlayer.start();
                }
                activateButtons(true);
                return;
            } else {
                startActivity(intent);
            }
        } catch (NullPointerException e) {
            Logger.info( e.getMessage(), e);
        }
    }

    private void closeClicked() {

        Logger.info("entered closeClicked()");
        cleanActivityUp();

        if (!mIsPlayBackError) {
            this.processEvent(TRACKING_EVENTS_TYPE.close);
        }

        finishVAST();

        Logger.info("leaving closeClicked()");
    }

    private void playPauseButtonClicked() {
        Logger.info("entered playPauseClicked");
        if(mMediaPlayer==null) {
            Logger.info( "mMediaPlayer is null when playPauseButton was clicked");
            return;
        }
        boolean isPlaying = mMediaPlayer.isPlaying();
        Logger.info("isPlaying:" + isPlaying);

        if (isPlaying) {
            //pause
            processPauseSteps();

        } else if (mIsVideoPaused) {
            //play
            this.processPlaySteps();
            if(!mIsCompleted) {
                this.processEvent(TRACKING_EVENTS_TYPE.resume);
            }
        } else {
            //replay
            this.processPlaySteps();
            mQuartile = 0;
            this.startQuartileTimer();
        }
    }

    private void processPauseSteps() {
        mIsVideoPaused = true;
        mMediaPlayer.pause();
        this.stopVideoProgressTimer();
        this.stopToolBarTimer();
        mPlayPauseButton.setToPlay();
        if(!mIsCompleted) {
            this.processEvent(TRACKING_EVENTS_TYPE.pause);
        }
    }


    private void processPlaySteps() {
        mIsVideoPaused = false;
        mMediaPlayer.start();
        mPlayPauseButton.setToPause();
        this.startToolBarTimer();
        this.startVideoProgressTimer();
    }


    @Override
    public void onBackPressed() {
        Logger.info("entered onBackPressed");
        this.closeClicked();

    }

    public void surfaceCreated(SurfaceHolder holder) {
        Logger.info("surfaceCreated -- (SurfaceHolder callback)");
        try {
            if(mMediaPlayer==null) {
                createMediaPlayer();
            }
            this.showProgressBar();
            mMediaPlayer.setDisplay(mSurfaceHolder);
            String url = mVastModel.getPickedMediaFileURL();

            Logger.info("URL for media file:" + url);
            mMediaPlayer.setDataSource(url);
            mMediaPlayer.prepareAsync();
        } catch (Exception e) {
            Logger.info( e.getMessage(), e);
        }
    }

    @Override
    public void surfaceChanged(SurfaceHolder surfaceHolder, int arg1, int arg2,
                               int arg3) {
        Logger.info(
                "entered surfaceChanged -- (SurfaceHolder callback)");
    }

    @Override
    public void surfaceDestroyed(SurfaceHolder surfaceHolder) {
        Logger.info(
                "entered surfaceDestroyed -- (SurfaceHolder callback)");
        cleanUpMediaPlayer();

    }

    @Override
    public void onVideoSizeChanged(MediaPlayer mp, int width, int height) {
        Logger.info(
                "entered onVideoSizeChanged -- (MediaPlayer callback)");
        mVideoWidth = width;
        mVideoHeight = height;
        Logger.info("video size: " + mVideoWidth + "x" + mVideoHeight);
    }

    @Override
    public void onPrepared(MediaPlayer mp) {
        Logger.info(
                "entered onPrepared called --(MediaPlayer callback) ....about to play");

        calculateAspectRatio();

        mMediaPlayer.start();

        this.hideProgressBar();

        if (mIsVideoPaused) {
            Logger.info("pausing video");
            mMediaPlayer.pause();
            this.mPlayPauseButton.setToPlay();
        } else {
            this.startVideoProgressTimer();
            this.mPlayPauseButton.setToPause();
        }

        Logger.info("current location in video:"
                + mCurrentVideoPosition);
        if (mCurrentVideoPosition > 0) {
            Logger.info("seeking to location:"
                    + mCurrentVideoPosition);
            mMediaPlayer.seekTo(mCurrentVideoPosition);

        }

        if (!mIsProcessedImpressions) {
            this.processImpressions();
        }

        startQuartileTimer();
        startToolBarTimer();

        if(!mMediaPlayer.isPlaying() && !mIsVideoPaused) {
            mMediaPlayer.start();

        }
        this.mPlayPauseButton.setVisibility(VISIBLE);
        durationHandler.postDelayed(updateRemainingTime, 100);
    }


    private void calculateAspectRatio() {
        Logger.info("entered calculateAspectRatio");

        if ( mVideoWidth == 0 || mVideoHeight == 0 ) {
            Logger.info("mVideoWidth or mVideoHeight is 0, skipping calculateAspectRatio");
            return;
        }

        DisplayMetrics displayMetrics = this.getResources()
                .getDisplayMetrics();

        mScreenWidth = displayMetrics.widthPixels;
        mScreenHeight = displayMetrics.heightPixels;

        Logger.info("calculating aspect ratio");
        double widthRatio = 1.0 * mScreenWidth / mVideoWidth;
        double heightRatio = 1.0 * mScreenHeight / mVideoHeight;

        double scale = Math.min(widthRatio, heightRatio);

        int surfaceWidth = (int) (scale * mVideoWidth);
        int surfaceHeight = (int) (scale * mVideoHeight);

        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                surfaceWidth, surfaceHeight);
        params.addRule(RelativeLayout.CENTER_IN_PARENT);
        mSurfaceView.setLayoutParams(params);

        mSurfaceHolder.setFixedSize(surfaceWidth, surfaceHeight);

        Logger.info(" screen size: " + mScreenWidth + "x" + mScreenHeight);
        Logger.info(" video size:  " + mVideoWidth + "x" + mVideoHeight);
        Logger.info(" widthRatio:   " + widthRatio);
        Logger.info(" heightRatio:   " + heightRatio);

        Logger.info("surface size: " + surfaceWidth + "x" + surfaceHeight);

    }

    private void cleanActivityUp() {

        this.cleanUpMediaPlayer();
        this.stopQuartileTimer();
        this.stopVideoProgressTimer();
        this.stopToolBarTimer();
        this.durationHandler.removeCallbacks(this.updateRemainingTime);
    }

    private void cleanUpMediaPlayer() {

        Logger.info("entered cleanUpMediaPlayer ");

        if (mMediaPlayer != null) {

            if (mMediaPlayer.isPlaying()) {
                mMediaPlayer.stop();
            }

            mMediaPlayer.setOnCompletionListener(null);
            mMediaPlayer.setOnErrorListener(null);
            mMediaPlayer.setOnPreparedListener(null);
            mMediaPlayer.setOnVideoSizeChangedListener(null);

            mMediaPlayer.release();
            mMediaPlayer = null;
        }

    }

    @Override
    public boolean onError(MediaPlayer mp, int what, int extra) {
        Logger.info( "entered onError -- (MediaPlayer callback)");
        mIsPlayBackError = true;
        Logger.info( "Shutting down Activity due to Media Player errors: WHAT:" + what +": EXTRA:" + extra+":");

        processErrorEvent();
        this.closeClicked();

        return true;
    }

    private void processErrorEvent() {
        Logger.info("entered processErrorEvent");

        List<String> errorUrls = mVastModel.getErrorUrl();
        fireUrls(errorUrls);

    }
    @Override
    public void onCompletion(MediaPlayer mediaPlayer) {
        Logger.info("entered onCOMPLETION -- (MediaPlayer callback)");
        stopVideoProgressTimer();
        stopToolBarTimer();
        mOverlay.setVisibility(VISIBLE);
        mPlayPauseButton.setToPlay();
        if ( !mIsPlayBackError && !mIsCompleted) {
            mIsCompleted = true;
            this.processEvent(TRACKING_EVENTS_TYPE.complete);

            if(VASTPlayer.listener!=null) {
                VASTPlayer.listener.vastComplete();
            }
        }

    }

    private void overlayClicked() {
        this.activateButtons(true);
        this.startToolBarTimer();
        Logger.info("overlay clicked");
    }

    private void processImpressions() {
        Logger.info("entered processImpressions");

        mIsProcessedImpressions = true;
        List<String> impressions = mVastModel.getImpressions();
        fireUrls(impressions);

    }

    private void fireUrls(List<String> urls) {
        Logger.info("entered fireUrls");

        if (urls != null) {

            for (String url : urls) {
                Logger.info("\tfiring url:%s", url);
                HttpTools.httpGetURL(url);
            }

        }else {
            Logger.info("\turl list is null");
        }



    }
    private void showToolBar(){
        this.activateButtons(true);
        this.startToolBarTimer();

    }
    private void hideToolBar(){
        activateButtons(false);
    }
    // Timers
    private void startToolBarTimer() {
        Logger.info("entered startToolBarTimer");
        if(mQuartile==4) {
            // we are at the end of the video, we dont want ot ever hide the toolbar now
            return;
        }

        if (mIsToolbarVisible && mMediaPlayer!= null && mMediaPlayer.isPlaying()) {
            stopToolBarTimer();
            mToolBarTimer = new Timer();
            mToolBarTimer.schedule(new TimerTask() {
                @Override
                public void run() {
                    mHandler.post(new Runnable() {
                        public void run() {
                            Logger.info("hiding buttons");
                            activateButtons(false);
                            mIsToolbarVisible = false;
                        }
                    });
                }
            }, TOOLBAR_HIDE_DELAY);
        }

        if (mIsVideoPaused) {
            activateButtons(true);
        }
    }

    private void stopToolBarTimer() {
        Logger.info("entered stopToolBarTimer");
        if (mToolBarTimer != null) {
            mToolBarTimer.cancel();
            mToolBarTimer = null;
        }
    }

    private void startQuartileTimer() {
        Logger.info("entered startQuartileTimer");
        stopQuartileTimer();

        if(mIsCompleted) {
            Logger.info("ending quartileTimer because the video has been replayed");
            return;
        }

        final int videoDuration = mMediaPlayer.getDuration();

        mTrackingEventTimer = new Timer();
        mTrackingEventTimer.scheduleAtFixedRate(new TimerTask() {

            @Override
            public void run() {
                int percentage = 0;
                try {
                    int curPos = mMediaPlayer.getCurrentPosition();
                    // wait for the video to really start
                    if (curPos == 0) {
                        return;
                    }
                    percentage = 100 * curPos / videoDuration;
                } catch (Exception e) {
                    Logger.error(
                            "mediaPlayer.getCurrentPosition exception: %s", e.getMessage());
                    this.cancel();
                    return;
                }

                if (percentage >= 25 * mQuartile) {
                    if (mQuartile == 0) {
                        Logger.info("Video at start: (%s%%)",percentage);
                        processEvent(TRACKING_EVENTS_TYPE.start);
                    } else if (mQuartile == 1) {
                        Logger.info("Video at first quartile: (%s%%)",percentage );
                        processEvent(TRACKING_EVENTS_TYPE.firstQuartile);
                    } else if (mQuartile == 2) {
                        Logger.info("Video at midpoint: (%s%%)",percentage);
                        processEvent(TRACKING_EVENTS_TYPE.midpoint);
                    } else if (mQuartile == 3) {
                        Logger.info("Video at third quartile: (%s%%)",percentage);
                        processEvent(TRACKING_EVENTS_TYPE.thirdQuartile);
                        stopQuartileTimer();
                    }
                    mQuartile++;
                }
            }

        }, 0, QUARTILE_TIMER_INTERVAL);
    }

    private void stopQuartileTimer() {

        if (mTrackingEventTimer != null) {
            mTrackingEventTimer.cancel();
            mTrackingEventTimer = null;
        }
    }

    private void startVideoProgressTimer() {
        Logger.info("entered startVideoProgressTimer");

        mStartVideoProgressTimer = new Timer();
        mVideoProgressTracker = new LinkedList<Integer>();

        mStartVideoProgressTimer.schedule(new TimerTask() {
            int maxAmountInList = mMaxProgressTrackingPoints - 1;

            @Override
            public void run() {

                if (mMediaPlayer == null) {
                    return;
                }

                if (mVideoProgressTracker.size() == maxAmountInList) {
                    int firstPosition = mVideoProgressTracker.getFirst();
                    int lastPosition = mVideoProgressTracker.getLast();

                    if (lastPosition > firstPosition) {
                        Logger.verbose("video progressing (position:" + lastPosition + ")");
                        mVideoProgressTracker.removeFirst();
                    } else {
                        Logger.info( "detected video hang");
                        mIsPlayBackError = true;
                        stopVideoProgressTimer();
                        processErrorEvent();
                        closeClicked();
                        finishVAST();
                    }
                }

                try {
                    int curPos = mMediaPlayer.getCurrentPosition();
                    mVideoProgressTracker.addLast(curPos);
                } catch (Exception e) {
                    // occasionally the timer is in the middle of processing and
                    // the media player is cleaned up
                }
            }

        }, 0, VIDEO_PROGRESS_TIMER_INTERVAL);

    }

    private void stopVideoProgressTimer() {
        Logger.info("entered stopVideoProgressTimer");

        if (mStartVideoProgressTimer != null) {

            mStartVideoProgressTimer.cancel();
        }
    }

    private void processEvent(TRACKING_EVENTS_TYPE eventName) {
        Logger.info("entered Processing Event: " + eventName);
        List<String> urls = (List<String>) mTrackingEventMap.get(eventName);

        fireUrls(urls);

    }

    private void finishVAST() {
        VASTPlayer.listener.vastDismiss();
        finish();
    }
    //handler to change seekBarTime
    private Runnable updateRemainingTime = new Runnable() {
        public void run() {
            if(mMediaPlayer!=null) {
                int mediaDuration = mMediaPlayer.getDuration();
                int mediaPosition = mMediaPlayer.getCurrentPosition();

                int duration = (mediaDuration/1000 - mediaPosition/1000);
                int minutes = duration/60;
                int seconds = duration % 60;

                String countdownText = elapsedTimeText.replaceAll("\\{m\\}",String.valueOf(minutes))
                        .replaceAll("\\{s\\}", String.valueOf(seconds));
                mCountdown.setText(countdownText);

                //repeat yourself that again in 100 miliseconds
                durationHandler.postDelayed(this, 100);
            }
        }
    };

}
