package eanative.android.model;

import eanative.android.ad.AdData;
import eanative.android.ad.EANativeAdPlacer;
import eanative.android.events.EANativeEvent;
import eanative.android.events.EANativeEventListener;
import eanative.android.util.Logger;
import android.content.Context;
import android.database.DataSetObserver;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;

public class EANativeListAdapter extends BaseAdapter {
	private final static int CALCULATE_CELLS_AHEAD = 4;
	private final Context context;
	private final BaseAdapter originalAdapter;
	private boolean canRequestAds = false;
	private EANativeAdPlacer adPlacer;
	private DataSetObserver originalDataObserver;

	
	public EANativeListAdapter(Context context, BaseAdapter originalAdapter,
			EANativeServerClient serverClient) {
		this.context = context;
		this.originalAdapter = originalAdapter;

		this.adPlacer = new EANativeAdPlacer(this.context, serverClient);
		this.originalDataObserver = new DataSetObserver() {
			public void onChanged() { 
				EANativeListAdapter.this.notifyDataSetChanged();
			}

			public void onInvalidated() {
				EANativeListAdapter.this.notifyDataSetInvalidated();
			}
		};
		originalAdapter.registerDataSetObserver(this.originalDataObserver);

		this.adPlacer.registerEventListener(EANativeEvent.ADS_LOADED,
				new EANativeEventListener() {
					public void handleMessage(String event, Object message) {
						EANativeListAdapter.this.notifyDataSetChanged();
					}
				});
	}
	public int getCount() {
		return this.originalAdapter.getCount() + this.adPlacer.getAdsCount();
	}
	/**
	 * Get the row id associated with the specified position in the list.
	 * @return The id of the item at the specified position.
	 */
	public long getItemId(int position) {
		return position;
	}

	/**
	 * Returns the number of types of Views that will be created by getView(int, View, ViewGroup). 
	 * This method returns 1+originalAdapter.getViewTypeCount()
	 * 
	 * @return The number of types of Views that will be created by this adapter
	 */
	@Override
	public int getViewTypeCount() {
		return this.adPlacer.getViewTpeCount()
				+ this.originalAdapter.getViewTypeCount();
	}

	/**
	 * 
	 */
	@Override
	public int getItemViewType(int position) {
		return this.adPlacer.isAd(position) ? this.getViewTypeCount() - 1
				: this.originalAdapter.getItemViewType(position);
	}

	public View getView(int position, View convertView, ViewGroup parent) {		
		EANativePlacementConfig config = this.adPlacer.getConfig();
		if(this.canRequestAds ==true){
			int maxAds = 0;
			int calculatedCell = position;
			if(config!=null && this.originalAdapter.getCount()>config.startAdPosition){
				calculatedCell = Math.min(this.originalAdapter.getCount(),this.adPlacer.getAdjustedPosition(position +CALCULATE_CELLS_AHEAD +1));
				maxAds = Math.round(1 + ( calculatedCell -config.startAdPosition )  / config.adFrequency);
				
			}
			if(maxAds > this.adPlacer.getTotalAdsToRequest()){
				this.adPlacer.setTotalAdsToRequest(maxAds);
			}
		}
		if (this.adPlacer.isAd(position)) {
			return this.adPlacer.getView(position, convertView, parent);
		}else{
			position = this.adPlacer.getAdjustedPosition(position);
		}	
		if(position<this.originalAdapter.getCount()){
			return this.originalAdapter.getView(position, convertView, parent);
		}
		return convertView;

	}

	public Object getItem(int position) {
		AdData ad = this.adPlacer.getAdData(position);
		return ad != null ? ad : this.originalAdapter.getItem(position);
	}
	
	public void notifyDataSetChanged() {
		this.adPlacer.clearPendingImpressions();
		super.notifyDataSetChanged();
	}
	public void initPlacement(String placementId, int layoutResId,EANativePlacementConfig placementConfig) {
		this.canRequestAds = true;
		this.adPlacer.initPlacement(placementId,layoutResId,placementConfig);
	}
	public void initPlacement(String placementId, int layoutResId) {
		this.canRequestAds = true;
		this.adPlacer.initPlacement(placementId,layoutResId);
	}

	public void initPlacement(EANativePlacementConfig placementConfig) {
		this.canRequestAds = true;
		this.adPlacer.initPlacement(placementConfig);

	}
	
	/**
	 * Clear all ads from the adapter
	 */
	public void clearAllAds(){
		if(this.adPlacer!=null){
			this.adPlacer.clearAllAds();
		}
		if(this.originalAdapter!=null){
			if(this.originalDataObserver!=null){
				this.originalAdapter.unregisterDataSetObserver(this.originalDataObserver);
			}
			this.originalDataObserver = null;
			this.notifyDataSetChanged();
		}
		this.canRequestAds = false;
	}

}
