package dev.micheleferretti.mapboxpluginoffline.model

import android.os.Bundle
import com.mapbox.mapboxsdk.offline.OfflineRegion
import com.mapbox.mapboxsdk.offline.OfflineRegionStatus
import dev.micheleferretti.mapboxpluginoffline.utils.requireBoolean
import dev.micheleferretti.mapboxpluginoffline.utils.requireLong

/**
 * This model represents the download managed by
 * the [OfflineService][dev.micheleferretti.mapboxpluginoffline.OfflineService].
 *
 * It holds the `options` with which the download has been started, the id (`regionId`) of the region generated by
 * Mapbox with `OfflineManager.createOfflineRegion()` and some of the `OfflineRegionStatus` data returned by
 * the `OfflineRegionObserver` attached to the region in order to monitor its download state.
 *
 * @property regionId `OfflineRegion` id.
 * @property options Options used to start the download.
 * @constructor Creates a `OfflineDownload` with the given values.
 */
class OfflineDownload(
    val regionId: Long,
    val options: OfflineDownloadOptions,
    private var completedResourceCount: Long,
    private var requiredResourceCount: Long,
    private var completedResourceSize: Long,
    private var isActive: Boolean
) {
    companion object {
        private const val EXTRA_REGION_ID = "extra.REGION_ID"
        private const val EXTRA_OPTIONS = "extra.OPTIONS"
        private const val EXTRA_COMPLETED_RESOURCE_COUNT = "extra.COMPLETED_RESOURCE_COUNT"
        private const val EXTRA_REQUIRED_RESOURCE_COUNT = "extra.REQUIRED_RESOURCE_COUNT"
        private const val EXTRA_COMPLETED_RESOURCE_SIZE = "extra.COMPLETED_RESOURCE_SIZE"
        private const val EXTRA_IS_ACTIVE = "extra.IS_ACTIVE"

        /**
         * Creates a `OfflineDownload` from a `Bundle`.
         * @param bundle the source `Bundle`.
         * @return a new `OfflineDownload`.
         * @see toBundle
         */
        @JvmStatic
        fun fromBundle(bundle: Bundle) = OfflineDownload(
            bundle.requireLong(EXTRA_REGION_ID),
            OfflineDownloadOptions.fromBundle(requireNotNull(bundle.getBundle(EXTRA_OPTIONS))),
            bundle.requireLong(EXTRA_COMPLETED_RESOURCE_COUNT),
            bundle.requireLong(EXTRA_REQUIRED_RESOURCE_COUNT),
            bundle.requireLong(EXTRA_COMPLETED_RESOURCE_SIZE),
            bundle.requireBoolean(EXTRA_IS_ACTIVE)
        )

        /**
         * Returns the completion percentage.
         * @param completedResourceCount number of resources that have been fully download.
         * @param requiredResourceCount number of resources that are known to be required.
         * @return the completion percentage.
         */
        @JvmStatic
        fun getPercentage(completedResourceCount: Long, requiredResourceCount: Long) =
            (if (requiredResourceCount > 0) 100.0 * completedResourceCount / requiredResourceCount else 0.0).toInt()
    }

    /**
     * Creates a `OfflineDownload` with the given values and no status data.
     */
    constructor(regionId: Long, options: OfflineDownloadOptions): this(regionId, options, 0L, 0L, 0L, false)

    /**
     * Creates a `Bundle` from this object.
     * @return a new `Bundle`.
     * @see fromBundle
     */
    fun toBundle() = Bundle().apply {
        putLong(EXTRA_REGION_ID, regionId)
        putBundle(EXTRA_OPTIONS, options.toBundle())
        putLong(EXTRA_COMPLETED_RESOURCE_COUNT, completedResourceCount)
        putLong(EXTRA_REQUIRED_RESOURCE_COUNT, requiredResourceCount)
        putLong(EXTRA_COMPLETED_RESOURCE_SIZE, completedResourceSize)
        putBoolean(EXTRA_IS_ACTIVE, isActive)
    }

    /**
     * As in `OfflineRegionStatus`, returns the number of resources that have been fully download.
     * @return the number of resources that have been fully download.
     */
    fun getCompletedResourceCount() = completedResourceCount

    /**
     * As in `OfflineRegionStatus`, returns the number of resources that are known to be required.
     * @return the number of resources that are known to be required.
     */
    fun getRequiredResourceCount() = requiredResourceCount

    /**
     * As in `OfflineRegionStatus`, returns the cumulative size, in bytes, of all resources that have been fully downloaded.
     * @return the cumulative size, in bytes, of all resources that have been fully downloaded.
     */
    fun getCompletedResourceSize() = completedResourceSize

    /**
     * Returns the download completion percentage.
     * @return the download completion percentage.
     */
    fun getPercentage() = getPercentage(completedResourceCount, requiredResourceCount)

    /**
     * Validates if `OfflineRegionStatus.getDownloadState()` equals `OfflineRegion.STATE_ACTIVE`.
     * @return `true` if download is active, `false` if not.
     */
    fun isActive() = isActive

    /**
     * As in `OfflineRegionStatus`, validates if the region download has completed.
     * @return `true` if download is complete, `false` if not.
     */
    fun isComplete() = (completedResourceCount == requiredResourceCount) && !isActive

    /**
     * Updates the download status properties
     */
    internal fun setStatus(status: OfflineRegionStatus) {
        completedResourceCount = status.completedResourceCount
        requiredResourceCount = status.requiredResourceCount
        completedResourceSize = status.completedResourceSize
        isActive = status.downloadState == OfflineRegion.STATE_ACTIVE
    }

    /**
     * Returns a `String` representation of this object.
     * @return a `String` representation of this object.
     */
    override fun toString() = "OfflineDownload(" +
            "regionId=$regionId, " +
            "options=$options, " +
            "completedResourceCount=$completedResourceCount, " +
            "requiredResourceCount=$requiredResourceCount, " +
            "completedResourceSize=$completedResourceSize, " +
            "isActive=$isActive" +
            ")"
}