/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.core.schema;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import dev.fitko.fitconnect.api.config.defaults.SchemaConfig;
import dev.fitko.fitconnect.api.domain.schema.SchemaResources;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectInitialisationException;
import dev.fitko.fitconnect.api.exceptions.internal.SchemaNotFoundException;
import dev.fitko.fitconnect.api.services.http.HttpClient;
import dev.fitko.fitconnect.api.services.schema.SchemaProvider;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SchemaResourceProvider
implements SchemaProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(SchemaResourceProvider.class);
    private static final Pattern VALID_SET_SCHEMA_PATTERN = Pattern.compile("1.\\d+.\\d+");
    public static final String VALID_METADATA_SCHEMA_PATTERN = "https://schema\\.fitko\\.de/fit-connect/metadata/[12]\\.\\d+\\.\\d+/metadata.schema.json";
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private final Map<URI, String> setSchemas = new HashMap<URI, String>();
    private final Map<URI, String> metadataSchemas = new HashMap<URI, String>();
    private final Map<URI, String> destinationSchemas = new HashMap<URI, String>();
    private final Map<URI, String> submissionDataSchemas = new HashMap<URI, String>();
    private final HttpClient httpClient;

    public SchemaResourceProvider(HttpClient httpClient, SchemaResources schemaResources) {
        this.httpClient = httpClient;
        schemaResources.getSetSchemas().forEach(this::addSetSchema);
        schemaResources.getMetadataSchemas().forEach(this::addMetadataSchema);
        schemaResources.getDestinationSchemas().forEach(this::addDestinationSchema);
        this.addCustomDataSchemas(schemaResources.getSubmissionDataSchemas());
        this.addDefaultDataSchemas();
        LOGGER.info("Initialised SDK schemas");
    }

    @Override
    public boolean isAllowedSetSchema(URI schemaUri) {
        return this.schemaVersionMatchesPattern(schemaUri, VALID_SET_SCHEMA_PATTERN);
    }

    @Override
    public boolean isAllowedMetadataSchema(URI schemaUri) {
        return this.metadataSchemas.containsKey(schemaUri) || schemaUri.toString().matches(VALID_METADATA_SCHEMA_PATTERN);
    }

    @Override
    public String loadSetSchema(URI schemaUri) throws SchemaNotFoundException {
        String schema = this.setSchemas.get(schemaUri);
        if (schema == null) {
            throw new SchemaNotFoundException("SET schema " + schemaUri.toString() + " is not available.");
        }
        return schema;
    }

    @Override
    public String loadMetadataSchema(URI schemaUri) throws SchemaNotFoundException {
        String schema = this.metadataSchemas.get(schemaUri);
        if (schema != null) {
            return schema;
        }
        return this.fetchSchemaFromRemote(schemaUri);
    }

    @Override
    public String loadDestinationSchema(URI schemaUri) throws SchemaNotFoundException {
        String schema = this.destinationSchemas.get(schemaUri);
        if (schema != null) {
            return schema;
        }
        return this.fetchSchemaFromRemote(schemaUri);
    }

    @Override
    public String loadSubmissionDataSchema(URI schemaUri) throws SchemaNotFoundException {
        String schema = this.submissionDataSchemas.get(schemaUri);
        if (schema != null) {
            return schema;
        }
        return this.fetchSchemaFromRemote(schemaUri);
    }

    private String fetchSchemaFromRemote(URI schemaUri) {
        LOGGER.debug("Schema {} is not available as local file, loading remote version.", (Object)schemaUri);
        if (!schemaUri.getScheme().equals("https")) {
            throw new SchemaNotFoundException("Fetching schema " + String.valueOf(schemaUri) + " from remote was skipped, since the URI does not support HTTPS.");
        }
        try {
            return this.httpClient.get(schemaUri.toString(), Map.of(), String.class).getBody();
        }
        catch (Exception exception) {
            throw new SchemaNotFoundException("Schema " + String.valueOf(schemaUri) + " is not available.", exception);
        }
    }

    private void addSetSchema(String schema) {
        this.readIdFromSchema(schema).ifPresentOrElse(id -> this.setSchemas.put((URI)id, schema), () -> LOGGER.warn("File '{}' does not provide a valid set schema and will be ignored.", (Object)schema));
    }

    private void addMetadataSchema(String schema) {
        this.readIdFromSchema(schema).ifPresentOrElse(id -> this.metadataSchemas.put((URI)id, schema), () -> LOGGER.warn("File '{}' does not provide a valid metadata schema and will be ignored.", (Object)schema));
    }

    private void addDestinationSchema(String schema) {
        this.readIdFromSchema(schema).ifPresentOrElse(id -> this.destinationSchemas.put((URI)id, schema), () -> LOGGER.warn("File '{}' does not provide a valid destination schema and will be ignored.", (Object)schema));
    }

    private void addDefaultDataSchemas() {
        this.submissionDataSchemas.put(SchemaConfig.ZBP_ADAPTER_SCHEMA.getSchemaUri(), SchemaResourceProvider.readPathToString(SchemaConfig.ZBP_ADAPTER_SCHEMA.getFileName()));
    }

    private void addCustomDataSchemas(Map<String, String> dataSchemas) {
        dataSchemas.forEach((schemaKey, schemaPath) -> this.submissionDataSchemas.putIfAbsent(SchemaResourceProvider.getUriFromKey(schemaKey), SchemaResourceProvider.readPathToString(schemaPath)));
    }

    private Optional<URI> readIdFromSchema(String schema) {
        try {
            return Optional.of(URI.create(MAPPER.readTree(schema).get("$id").asText()));
        }
        catch (JsonProcessingException | IllegalArgumentException | NullPointerException e) {
            return Optional.empty();
        }
    }

    private static String readPathToString(String schemaPath) {
        try {
            return Files.readString(Path.of(schemaPath, new String[0]));
        }
        catch (IOException | NullPointerException e) {
            try {
                return SchemaResourceProvider.readResourceToString(schemaPath);
            }
            catch (Exception ex) {
                throw new FitConnectInitialisationException("Reading data schema " + schemaPath + " failed", ex);
            }
        }
    }

    private boolean schemaVersionMatchesPattern(URI schemaUri, Pattern pattern) {
        String schemaVersion = schemaUri.getPath().split("/")[3];
        return pattern.matcher(schemaVersion).matches();
    }

    private static URI getUriFromKey(String schemaKey) {
        try {
            return URI.create(schemaKey);
        }
        catch (Exception e) {
            LOGGER.error("Could not map data schema {} to a valid URI", (Object)schemaKey);
            throw new FitConnectInitialisationException(e.getMessage(), e);
        }
    }

    public static String readResourceToString(String resourcePath) throws IOException {
        try (InputStream in = SchemaResourceProvider.class.getResourceAsStream(resourcePath);){
            String string = new String(Objects.requireNonNull(in).readAllBytes(), StandardCharsets.UTF_8);
            return string;
        }
    }
}

