/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.core.routing;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.JWSHeader;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.RSASSAVerifier;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.SignedJWT;
import dev.fitko.fitconnect.api.domain.model.route.Route;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.exceptions.internal.InvalidKeyException;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.exceptions.internal.ValidationException;
import dev.fitko.fitconnect.api.services.keys.KeyService;
import dev.fitko.fitconnect.api.services.routing.RoutingVerificationService;
import dev.fitko.fitconnect.api.services.validation.ValidationService;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RouteVerifier
implements RoutingVerificationService {
    private static final Logger LOGGER = LoggerFactory.getLogger(RouteVerifier.class);
    private final KeyService keyService;
    private final ValidationService validationService;
    private static final List<String> VALID_KEY_HOSTS = List.of(".fit-connect.fitko.net", ".fit-connect.fitko.dev");

    public RouteVerifier(KeyService keyService, ValidationService validationService) {
        this.keyService = keyService;
        this.validationService = validationService;
    }

    @Override
    public List<Route> validateRouteDestinations(List<Route> routes, String requestedServiceIdentifier, String requestedRegion) {
        return routes.stream().filter(route -> this.isValidRoute((Route)route, requestedServiceIdentifier, requestedRegion)).collect(Collectors.toList());
    }

    private boolean isValidRoute(Route route, String requestedServiceIdentifier, String requestedRegion) {
        try {
            SignedJWT signature = SignedJWT.parse((String)route.getDestinationSignature());
            JWSHeader header = signature.getHeader();
            JWTClaimsSet claims = signature.getJWTClaimsSet();
            this.checkHeaderAlgorithm(header);
            this.validatePayloadSchema(claims);
            this.checkExpectedServices(claims, requestedServiceIdentifier, requestedRegion);
            this.validateAgainstPublicKey(signature, header.getKeyID());
            return true;
        }
        catch (ValidationException e) {
            LOGGER.warn("Route validation failed for destination {}: {}", (Object)route.getDestinationId(), (Object)e.getMessage());
            return false;
        }
        catch (InvalidKeyException e) {
            LOGGER.warn("Route validation failed for destination {}: Public signature key is invalid: {}", (Object)route.getDestinationId(), (Object)e.getMessage());
            return false;
        }
        catch (RestApiException e) {
            LOGGER.warn("Route validation failed for destination {}: Could not retrieve public signature key: {}", (Object)route.getDestinationId(), (Object)e.getMessage());
            return false;
        }
        catch (JOSEException | ParseException e) {
            LOGGER.warn("Route validation failed for destination {}: Signature processing failed: {}", (Object)route.getDestinationId(), (Object)e.getMessage());
            return false;
        }
    }

    private void validatePayloadSchema(JWTClaimsSet claims) {
        ValidationResult validationResult = this.validationService.validateDestinationSchema(claims.toJSONObject());
        if (validationResult.hasError()) {
            throw new ValidationException(validationResult.getError().getMessage(), validationResult.getError());
        }
    }

    private void validateAgainstPublicKey(SignedJWT signature, String keyId) throws JOSEException, ParseException {
        RSAKey portalPublicKey = this.getSignatureValidationKey(signature, keyId);
        if (!signature.verify((JWSVerifier)new RSASSAVerifier(portalPublicKey))) {
            throw new ValidationException("Invalid destination signature for public key id " + keyId);
        }
    }

    private RSAKey getSignatureValidationKey(SignedJWT signature, String keyId) throws ParseException {
        String issuer = signature.getJWTClaimsSet().getIssuer();
        if (VALID_KEY_HOSTS.stream().anyMatch(issuer::endsWith)) {
            return this.keyService.getPublicSignatureWellKnownKey(keyId, issuer, signature.getJWTClaimsSet().getIssueTime());
        }
        throw new ValidationException("Requested signature validation key url '" + issuer + "' is no FIT-Connect host (.fit-connect.fitko.net or .fit-connect.fitko.dev)");
    }

    private void checkExpectedServices(JWTClaimsSet claims, String requestedServiceIdentifier, String requestedRegion) throws ParseException {
        List services = claims.getListClaim("services").stream().map(service -> new RouteService((Map)service)).collect(Collectors.toList());
        String serviceId = RouteVerifier.getIdFromIdentifier(requestedServiceIdentifier);
        if (services.stream().noneMatch(service -> service.hasMatchingService(serviceId))) {
            throw new ValidationException("Requested service identifier '" + requestedServiceIdentifier + "' is not supported by any of the destinations services");
        }
        if (requestedRegion != null) {
            String regionId = RouteVerifier.getIdFromIdentifier(requestedRegion);
            if (services.stream().noneMatch(service -> service.hasMatchingRegionAndService(regionId, serviceId))) {
                throw new ValidationException("Requested region '" + requestedRegion + "' does not match any service provided by the destination");
            }
        }
    }

    private static String getIdFromIdentifier(String identifier) {
        if (RouteVerifier.isNumericId(identifier)) {
            return identifier;
        }
        return Arrays.stream(identifier.split(":")).reduce((first, second) -> second).orElse(null);
    }

    private static boolean isNumericId(String identifier) {
        return Pattern.compile("\\d+").matcher(identifier).matches();
    }

    private void checkHeaderAlgorithm(JWSHeader header) {
        if (!header.getAlgorithm().equals((Object)JWSAlgorithm.PS512)) {
            throw new ValidationException("Algorithm in signature header is not " + String.valueOf(JWSAlgorithm.PS512));
        }
    }

    static class RouteService {
        private final List<String> regionIds;
        private final List<String> serviceIds;

        protected RouteService(Map<String, List<String>> service) {
            this.regionIds = service.getOrDefault("gebietIDs", Collections.emptyList()).stream().map(x$0 -> RouteVerifier.getIdFromIdentifier(x$0)).collect(Collectors.toList());
            this.serviceIds = service.getOrDefault("leistungIDs", Collections.emptyList()).stream().map(x$0 -> RouteVerifier.getIdFromIdentifier(x$0)).collect(Collectors.toList());
        }

        public boolean hasMatchingRegionAndService(String regionId, String serviceId) {
            return this.hasMatchingRegion(regionId) && this.hasMatchingService(serviceId);
        }

        public boolean hasMatchingRegion(String regionId) {
            return this.regionIds.stream().anyMatch(regionId::contains);
        }

        public boolean hasMatchingService(String serviceId) {
            return this.serviceIds.contains(serviceId);
        }
    }
}

