/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.core.io;

import dev.fitko.fitconnect.core.utils.Preconditions;
import dev.fitko.fitconnect.core.utils.StopWatch;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileChunker {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileChunker.class);
    private static final String CHUNK_INDEX_FORMAT = "%06d";
    private static final String CHUNK_FILE_ENDING = ".chunk";
    private static final String PART_SUFFIX = "_part_";
    public static final String FRAGMENT_PREFIX = "fragment_";

    public List<File> chunkFile(Path filePath, int chunkSize, Path outputDir) throws IOException {
        try (InputStream is = Files.newInputStream(filePath, new OpenOption[0]);){
            List<File> list = this.chunkStream(is, chunkSize, filePath.getFileName().toString(), outputDir);
            return list;
        }
    }

    public List<File> chunkStream(InputStream stream, int chunkSize, String fileName, Path outputDir) throws IOException {
        Preconditions.checkArgumentAndThrow(chunkSize <= 0, "chunk size must be larger than zero, but was " + chunkSize);
        byte[] buffer = new byte[chunkSize];
        ArrayList<File> fileChunks = new ArrayList<File>();
        String name = fileName == null ? UUID.randomUUID().toString() : fileName;
        int chunkCount = 0;
        int readBytes = stream.read(buffer);
        long start = StopWatch.start();
        while (readBytes > -1) {
            File fileChunk = this.writeFileChunk(FileChunker.generateFileName(name, chunkCount), buffer, readBytes, outputDir);
            LOGGER.debug("Wrote {} bytes of chunk #{}", (Object)readBytes, (Object)(++chunkCount));
            fileChunks.add(fileChunk);
            readBytes = stream.read(buffer);
        }
        LOGGER.info("Chunking to {} files took {}", (Object)chunkCount, (Object)StopWatch.stop(start));
        return fileChunks;
    }

    public Optional<File> concatChunks(Path path, String fileName) throws IOException {
        List chunkedFiles = Stream.of(Objects.requireNonNull(path.toFile().listFiles())).filter(file -> !file.isDirectory()).filter(file -> file.getName().contains(CHUNK_FILE_ENDING)).sorted(Comparator.comparing(File::getName)).collect(Collectors.toList());
        if (chunkedFiles.isEmpty()) {
            return Optional.empty();
        }
        File mergedFile = path.resolve(fileName).toFile();
        LOGGER.info("Merging {} chunks into file {}", (Object)chunkedFiles.size(), (Object)mergedFile.getPath());
        long start = StopWatch.start();
        try (FileOutputStream fos = new FileOutputStream(mergedFile);){
            for (File chunk : chunkedFiles) {
                InputStream is = Files.newInputStream(chunk.toPath(), new OpenOption[0]);
                try {
                    is.transferTo(fos);
                    fos.flush();
                    chunk.delete();
                }
                finally {
                    if (is == null) continue;
                    is.close();
                }
            }
        }
        LOGGER.info("Merging chunks took {}", (Object)StopWatch.stop(start));
        return Optional.of(mergedFile);
    }

    public File writeFileChunk(byte[] data, int chunkIndex, Path outputDir) throws IOException {
        String fileName = FileChunker.generateFileName(chunkIndex);
        return this.writeFileChunk(fileName, data, data.length, outputDir);
    }

    private File writeFileChunk(String fileName, byte[] buffer, int length, Path outputDir) throws IOException {
        File chunk = outputDir.resolve(fileName).toFile();
        try (FileOutputStream fos = new FileOutputStream(chunk);){
            fos.write(buffer, 0, length);
        }
        return chunk;
    }

    private static String generateFileName(String fileName, int chunkCount) {
        return fileName + PART_SUFFIX + String.format(CHUNK_INDEX_FORMAT, chunkCount) + CHUNK_FILE_ENDING;
    }

    private static String generateFileName(int chunkCount) {
        return FRAGMENT_PREFIX + String.format(CHUNK_INDEX_FORMAT, chunkCount) + CHUNK_FILE_ENDING;
    }
}

