/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.core.crypto;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWEDecrypter;
import com.nimbusds.jose.JWEEncrypter;
import com.nimbusds.jose.JWEHeader;
import com.nimbusds.jose.JWEObject;
import com.nimbusds.jose.Payload;
import com.nimbusds.jose.crypto.RSADecrypter;
import com.nimbusds.jose.crypto.RSAEncrypter;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.exceptions.internal.DecryptionException;
import dev.fitko.fitconnect.api.exceptions.internal.EncryptionException;
import dev.fitko.fitconnect.api.services.crypto.CryptoService;
import dev.fitko.fitconnect.api.services.crypto.MessageDigestService;
import dev.fitko.fitconnect.core.crypto.HashService;
import dev.fitko.fitconnect.core.crypto.constants.CryptoConstants;
import dev.fitko.fitconnect.core.utils.StopWatch;
import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JWECryptoService
implements CryptoService {
    private static final Logger LOGGER = LoggerFactory.getLogger(JWECryptoService.class);
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private final MessageDigestService messageDigestService;

    public JWECryptoService(MessageDigestService messageDigestService) {
        this.messageDigestService = messageDigestService;
    }

    public JWECryptoService() {
        this.messageDigestService = new HashService();
    }

    @Override
    public byte[] decryptToBytes(RSAKey privateKey, String encryptedData) throws DecryptionException {
        return this.decrypt(privateKey, encryptedData).toBytes();
    }

    @Override
    public String encryptObject(RSAKey encryptionKey, Object obj, String contentType) throws EncryptionException {
        try {
            Payload payload = new Payload(MAPPER.writeValueAsBytes(obj));
            return this.encrypt(encryptionKey, payload, contentType).serialize();
        }
        catch (JsonProcessingException e) {
            throw new EncryptionException(e.getMessage(), e);
        }
    }

    @Override
    public String encryptBytes(RSAKey publicKey, byte[] bytes, String contentType) throws EncryptionException {
        Payload payload = new Payload(bytes);
        return this.encrypt(publicKey, payload, contentType).serialize();
    }

    @Override
    public JWEObject encryptInputStream(RSAKey publicKey, InputStream inputStream, String contentType) throws EncryptionException {
        Payload payload;
        try {
            payload = new Payload(inputStream.readAllBytes());
        }
        catch (IOException e) {
            throw new EncryptionException(e.getMessage(), e);
        }
        return this.encrypt(publicKey, payload, contentType);
    }

    @Override
    public String hashBytes(byte[] data) {
        byte[] hash = this.messageDigestService.createHash(data);
        return this.messageDigestService.toHexString(hash);
    }

    @Override
    public String hashStream(InputStream inputStream) {
        byte[] hash = this.messageDigestService.createHash(inputStream);
        return this.messageDigestService.toHexString(hash);
    }

    private JWEObject encrypt(RSAKey publicKey, Payload payload, String contentType) throws EncryptionException {
        try {
            return this.encryptPayload(publicKey, payload, contentType);
        }
        catch (JOSEException e) {
            throw new EncryptionException(e.getMessage(), e);
        }
    }

    private Payload decrypt(RSAKey privateKey, String encData) throws DecryptionException {
        try {
            long start = StopWatch.start();
            this.printAvailableVmMemory();
            JWEObject jwe = JWEObject.parse((String)encData);
            jwe.decrypt((JWEDecrypter)new RSADecrypter(privateKey));
            LOGGER.trace("Decryption took {} ", (Object)StopWatch.stop(start));
            return jwe.getPayload();
        }
        catch (JOSEException | IllegalStateException | ParseException e) {
            throw new DecryptionException(e.getMessage(), e);
        }
    }

    private JWEHeader getJWEHeader(String keyID, String contentType) {
        return new JWEHeader.Builder(CryptoConstants.DEFAULT_JWE_ALGORITHM, CryptoConstants.DEFAULT_JWE_ENCRYPTION_METHOD).contentType(contentType).keyID(keyID).build();
    }

    private RSAEncrypter getEncrypter(RSAKey publicKey) {
        try {
            return new RSAEncrypter(publicKey.toRSAPublicKey());
        }
        catch (JOSEException e) {
            throw new EncryptionException("RSAEncrypter could not be initialized", e);
        }
    }

    private JWEObject encryptPayload(RSAKey publicKey, Payload payload, String contentType) throws JOSEException, EncryptionException {
        String keyID = this.getIdFromPublicKey(publicKey);
        JWEObject jwe = new JWEObject(this.getJWEHeader(keyID, contentType), payload);
        long start = StopWatch.start();
        jwe.encrypt((JWEEncrypter)this.getEncrypter(publicKey));
        this.printAvailableVmMemory();
        this.checkIfJWEObjectIsEncrypted(jwe);
        LOGGER.trace("Encryption took {} ", (Object)StopWatch.stop(start));
        return jwe;
    }

    private void checkIfJWEObjectIsEncrypted(JWEObject jwe) throws EncryptionException {
        if (!jwe.getState().equals((Object)JWEObject.State.ENCRYPTED)) {
            throw new EncryptionException("JWE object is not encrypted");
        }
    }

    private String getIdFromPublicKey(RSAKey publicKey) throws EncryptionException {
        String keyID = publicKey.getKeyID();
        if (keyID == null || keyID.isEmpty()) {
            throw new EncryptionException("public key has no keyID");
        }
        return keyID;
    }

    private void printAvailableVmMemory() {
        LOGGER.trace("Currently used VM memory {} MB", (Object)((Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) / 1024L / 1024L));
    }
}

