/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.core.cases;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.JWSHeader;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.RSASSAVerifier;
import com.nimbusds.jose.jwk.KeyOperation;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.SignedJWT;
import dev.fitko.fitconnect.api.domain.model.event.Event;
import dev.fitko.fitconnect.api.domain.model.event.EventIssuer;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.validation.ValidationContext;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.exceptions.internal.EventLogException;
import dev.fitko.fitconnect.api.services.events.EventLogVerificationService;
import dev.fitko.fitconnect.api.services.keys.KeyService;
import dev.fitko.fitconnect.api.services.validation.ValidationService;
import dev.fitko.fitconnect.core.utils.EventLogUtil;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

public class EventLogVerifier
implements EventLogVerificationService {
    private static final String UUID_V4_PATTERN = "[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-4[0-9a-fA-F]{3}-[89abAB][0-9a-fA-F]{3}-[0-9a-fA-F]{12}";
    private final KeyService keyService;
    private final ValidationService validationService;

    public EventLogVerifier(KeyService keyService, ValidationService validationService) {
        this.keyService = keyService;
        this.validationService = validationService;
    }

    @Override
    public List<ValidationResult> validateEventLogs(ValidationContext ctx, List<SignedJWT> eventLogs) {
        eventLogs.forEach(event -> this.validateEventLogEntry(ctx, (SignedJWT)event));
        return new ArrayList<ValidationResult>(ctx.getValidationResults());
    }

    private void validateEventLogEntry(ValidationContext ctx, SignedJWT signedJWT) {
        try {
            JWTClaimsSet payload = signedJWT.getJWTClaimsSet();
            String issuer = payload.getIssuer();
            JWSHeader header = signedJWT.getHeader();
            RSAKey verificationKey = this.getSignatureVerificationKey(issuer, header.getKeyID(), payload.getIssueTime());
            this.validateSchema(ctx, payload);
            this.validateHeader(ctx, header);
            this.validateHeaderKid(ctx, header.getKeyID());
            this.validatePayload(ctx, payload);
            this.validateIssuerClaim(ctx, payload);
            this.validateTransactionClaim(ctx, payload);
            this.validateSignatureKey(ctx, verificationKey);
            this.validateSignature(ctx, signedJWT, verificationKey);
            if (ctx.validateAuthTags() && EventLogVerifier.eventContainsAuthTags(payload)) {
                this.validateAuthenticationTags(ctx, payload);
            }
        }
        catch (JOSEException | RuntimeException | ParseException e) {
            ctx.addError((Exception)e);
        }
    }

    private void validateSchema(ValidationContext ctx, JWTClaimsSet payload) {
        if (EventLogVerifier.getSchemaClaim(payload) == null) {
            ctx.addError("Referenced schema must not be null");
        } else {
            ctx.addResult(this.validationService.validateSetEventSchema(payload.toString()));
        }
    }

    private void validateHeader(ValidationContext ctx, JWSHeader header) {
        if (header.getAlgorithm() == null) {
            ctx.addError("The provided alg in the SET header must not be null.");
        } else {
            ctx.addErrorIfTestFailed(header.getAlgorithm() == JWSAlgorithm.PS512, "The provided alg in the SET header is not allowed.");
        }
        if (header.getType() == null) {
            ctx.addError("The provided typ in the SET header must not be null.");
        } else {
            ctx.addErrorIfTestFailed(header.getType().toString().equals("secevent+jwt"), "The provided typ in the SET header is not secevent+jwt");
        }
    }

    private void validateHeaderKid(ValidationContext ctx, String keyId) {
        if (keyId == null) {
            ctx.addError("The kid the SET was signed with is not set and must not be null.");
        } else {
            int length = keyId.length();
            int minLength = 8;
            int maxLength = 64;
            if (length < 8 || length > 64) {
                ctx.addError("Invalid keyId path parameter. Must be >=8 and <=64.");
            }
        }
    }

    private void validatePayload(ValidationContext ctx, JWTClaimsSet claims) throws ParseException {
        ctx.addErrorIfTestFailed(claims.getClaim("iss") != null, "The claim iss is missing in the payload of the SET.");
        ctx.addErrorIfTestFailed(claims.getClaim("iat") != null, "The claim iat is missing in the payload of the SET.");
        ctx.addErrorIfTestFailed(claims.getClaim("jti") != null, "The claim jti is missing in the payload of the SET.");
        ctx.addErrorIfTestFailed(claims.getClaim("sub") != null, "The claim sub is missing in the payload of the SET.");
        ctx.addErrorIfTestFailed(claims.getClaim("txn") != null, "The claim txn is missing in the payload of the SET.");
        ctx.addErrorIfTestFailed(claims.getClaim("events") != null, "The claim events is missing in the payload of the SET.");
        ctx.addErrorIfTestFailed(claims.getJSONObjectClaim("events").keySet().size() == 1, "The events claims has must be exactly one event.");
        ctx.addErrorIfTestFailed(claims.getStringClaim("sub").matches("(submission|case|reply):[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-4[0-9a-fA-F]{3}-[89abAB][0-9a-fA-F]{3}-[0-9a-fA-F]{12}"), "The provided subject does not match the allowed pattern.");
        ctx.addErrorIfTestFailed(claims.getStringClaim("txn").matches("case:[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-4[0-9a-fA-F]{3}-[89abAB][0-9a-fA-F]{3}-[0-9a-fA-F]{12}"), "The provided txn does not match the allowed pattern.");
        EventLogVerifier.getEventClaim(claims).ifPresentOrElse(event -> ctx.addErrorIfTestFailed(Event.fromSchemaUri(event) != null, "The provided event is not a valid event supported by this instance."), () -> ctx.addError("No events in JWT"));
    }

    private void validateSignatureKey(ValidationContext ctx, RSAKey signatureKey) {
        if (signatureKey == null) {
            ctx.addError("The signature key cannot be validated, it must not be null.");
        } else {
            ctx.addResult(this.validationService.validatePublicKey(signatureKey, KeyOperation.VERIFY));
        }
    }

    private void validateSignature(ValidationContext ctx, SignedJWT signedJWT, RSAKey signatureKey) throws JOSEException {
        if (signatureKey == null) {
            ctx.addError("The signature cannot validated, signature key is unavailable.");
        } else {
            RSASSAVerifier jwsVerifier = new RSASSAVerifier(signatureKey);
            ctx.addErrorIfTestFailed(signedJWT.verify((JWSVerifier)jwsVerifier), "The signature of the token could not be verified with the specified key.");
        }
    }

    private static Object getSchemaClaim(JWTClaimsSet payload) {
        return payload.getClaim("$schema");
    }

    private static Optional<String> getEventClaim(JWTClaimsSet claims) throws ParseException {
        return claims.getJSONObjectClaim("events").keySet().stream().findFirst();
    }

    private RSAKey getSignatureVerificationKey(String issuer, String keyId, Date validationDate) throws EventLogException {
        EventIssuer issuerType = EventLogUtil.resolveIssuerType(issuer);
        if (issuerType == EventIssuer.SUBMISSION_SERVICE) {
            return this.keyService.getSubmissionServicePublicSignatureKey(keyId, validationDate);
        }
        return this.keyService.getDestinationPublicSignatureKey(EventLogUtil.getDestinationId(issuer), keyId, validationDate);
    }

    private void validateIssuerClaim(ValidationContext ctx, JWTClaimsSet payload) throws EventLogException {
        String issuer = payload.getIssuer();
        UUID destinationId = EventLogUtil.getDestinationId(issuer);
        Event event = EventLogUtil.getEventFromClaims(payload);
        if (destinationId == null && EventLogUtil.resolveIssuerType(issuer).equals((Object)EventIssuer.DESTINATION)) {
            ctx.addError("The event '" + event.getSchemaUri() + "' has to be created by the destination ('iss' claim must be an UUID)");
        }
        if (destinationId != null) {
            ctx.addErrorIfTestFailed(destinationId.equals(ctx.getDestinationId()), "The destination of the submission is not the issuer ('iss' claim must match submission.destinationId)");
        }
    }

    private void validateTransactionClaim(ValidationContext ctx, JWTClaimsSet payload) throws ParseException {
        String txn = payload.getStringClaim("txn");
        String transactionId = txn.substring(txn.indexOf(58) + 1);
        ctx.addErrorIfTestFailed(transactionId.equals(ctx.getCaseId().toString()), "The provided txn does not match with the case.");
    }

    private void validateAuthenticationTags(ValidationContext ctx, JWTClaimsSet payload) {
        AuthenticationTags eventAuthTags = EventLogUtil.getAuthTags(payload);
        AuthenticationTags submissionAuthTags = ctx.getAuthenticationTags();
        ctx.addErrorIfTestFailed(eventAuthTags != null, "AuthenticationTags of event must not be null.");
        ctx.addErrorIfTestFailed(submissionAuthTags != null, "AuthenticationTags of submission must not be null.");
        if (eventAuthTags != null && submissionAuthTags != null) {
            this.validateDataAuthTags(ctx, eventAuthTags.getData(), submissionAuthTags.getData());
            this.validateMetadataAuthTags(ctx, eventAuthTags.getMetadata(), submissionAuthTags.getMetadata());
            this.validateAttachmentsAuthTags(ctx, eventAuthTags.getAttachments(), submissionAuthTags.getAttachments());
        }
    }

    private void validateAttachmentsAuthTags(ValidationContext ctx, Map<UUID, String> eventAttachmentTags, Map<UUID, String> submissionAttachmentTags) {
        if (eventAttachmentTags != null && submissionAttachmentTags != null) {
            ctx.addErrorIfTestFailed(eventAttachmentTags.size() == submissionAttachmentTags.size(), "The events quantity of attachments does not match the submission.");
            eventAttachmentTags.forEach((key, eventTag) -> {
                String submissionTag = (String)submissionAttachmentTags.get(key);
                ctx.addErrorIfTestFailed(eventTag.equals(submissionTag), "The authentication-tag for the attachment " + String.valueOf(key) + " does not match the submission.");
            });
        }
    }

    private void validateDataAuthTags(ValidationContext ctx, String eventDataAuthTag, String submissionDataAuthTag) {
        ctx.addErrorIfTestFailed(eventDataAuthTag.equals(submissionDataAuthTag), "The events data authentication-tag does not match the submission.");
    }

    private void validateMetadataAuthTags(ValidationContext ctx, String eventMetadataAuthTag, String submissionMetadataAuthTag) {
        ctx.addErrorIfTestFailed(eventMetadataAuthTag.equals(submissionMetadataAuthTag), "The events metadata authentication-tag does not match the submission.");
    }

    private static boolean eventContainsAuthTags(JWTClaimsSet payload) throws ParseException {
        Optional<String> eventClaim = EventLogVerifier.getEventClaim(payload);
        return eventClaim.isPresent() && Event.fromSchemaUri(eventClaim.get()).hasAuthTags();
    }
}

