/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.client.subscriber;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.nimbusds.jose.jwk.KeyOperation;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.ApplicationConfig;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.event.EventPayload;
import dev.fitko.fitconnect.api.domain.model.event.Status;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.authtags.ValidatedAuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.event.problems.data.DataEncryptionIssue;
import dev.fitko.fitconnect.api.domain.model.event.problems.metadata.MetadataEncryptionIssue;
import dev.fitko.fitconnect.api.domain.model.event.problems.metadata.MetadataJsonSyntaxViolation;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.AttachmentForValidation;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.subscriber.ReceivedSubmission;
import dev.fitko.fitconnect.api.domain.subscriber.ReceivedSubmissionData;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectSubscriberException;
import dev.fitko.fitconnect.api.exceptions.internal.DecryptionException;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.exceptions.internal.SubmissionRequestException;
import dev.fitko.fitconnect.client.attachments.download.AttachmentDownloader;
import dev.fitko.fitconnect.client.util.AttachmentMapper;
import dev.fitko.fitconnect.client.util.MalwareScanner;
import dev.fitko.fitconnect.client.util.MetadataDeserializationHelper;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SubmissionReceiver {
    private static final Logger LOGGER = LoggerFactory.getLogger(SubmissionReceiver.class);
    private static final ObjectMapper MAPPER = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    private final FitConnectService fitConnectService;
    private final ApplicationConfig config;
    private final RSAKey privateKey;
    private final AttachmentDownloader<Submission> attachmentDownloader;
    private final MalwareScanner malwareScanner;

    public SubmissionReceiver(FitConnectService fitConnectService, RSAKey privateKey, ApplicationConfig config, AttachmentDownloader<Submission> attachmentDownloader) {
        this.config = config;
        this.privateKey = privateKey;
        this.fitConnectService = fitConnectService;
        this.attachmentDownloader = attachmentDownloader;
        this.malwareScanner = new MalwareScanner(config.getVirusScannerMode(), fitConnectService);
    }

    public ReceivedSubmission receiveSubmission(UUID submissionId) {
        LOGGER.info("Requesting submission ...");
        Submission submission = this.loadSubmission(submissionId);
        LOGGER.info("Loading authentication tags from event log ...");
        AuthenticationTags authenticationTags = this.loadAuthTagsForSubmitEvent(submission);
        LOGGER.info("Decrypting metadata ...");
        Metadata metadata = this.decryptMetadata(submission);
        this.validateMetadata(metadata, submission, authenticationTags);
        LOGGER.info("Loading and decrypting attachments ...");
        List<AttachmentForValidation> attachments = this.loadAttachments(submission, metadata);
        this.validateAttachments(attachments, submission, authenticationTags);
        LOGGER.info("Decrypting data ...");
        byte[] decryptedData = this.decryptData(submission, attachments);
        this.validateData(submission, metadata, decryptedData, authenticationTags.getData());
        LOGGER.info("SUCCESSFULLY RECEIVED SUBMISSION ! \n");
        return this.buildReceivedSubmission(submission, metadata, decryptedData, attachments);
    }

    private List<AttachmentForValidation> loadAttachments(Submission submission, Metadata metadata) {
        this.checkPrivateKey();
        return this.attachmentDownloader.loadAttachments(submission, this.privateKey, metadata.getContentStructure().getAttachments());
    }

    private Submission loadSubmission(UUID submissionId) {
        try {
            return this.fitConnectService.getSubmission(submissionId);
        }
        catch (RestApiException e) {
            throw new SubmissionRequestException(e.getMessage(), e);
        }
    }

    private AuthenticationTags loadAuthTagsForSubmitEvent(Submission submission) {
        ValidatedAuthenticationTags authTags = this.fitConnectService.getSubmissionAuthenticationTags(submission);
        if (authTags.causesRejection()) {
            LOGGER.error("Authentication Tag validation failed, due to {}", (Object)authTags.getErrorMessage());
            this.rejectSubmissionWithProblem(submission, authTags.getProblem());
            throw new SubmissionRequestException(authTags.getErrorMessage());
        }
        return authTags.getAuthenticationTags();
    }

    private void validateMetadata(Metadata metadata, Submission submission, AuthenticationTags authenticationTags) {
        VirusScanResult dataScanResult = this.malwareScanner.scanMetadata(metadata);
        if (dataScanResult.isInfected()) {
            LOGGER.error("Metadata is infected with {}", (Object)dataScanResult.getSignature());
            this.rejectSubmissionWithProblem(submission, new Problem[0]);
            throw new FitConnectSubscriberException("Metadata is infected with virus: " + dataScanResult.getSignature());
        }
        ValidationResult validationResult = this.fitConnectService.validateSubmissionMetadata(metadata, submission, authenticationTags);
        this.evaluateValidationResult(submission, validationResult);
    }

    private void validateAttachments(List<AttachmentForValidation> attachmentForValidation, Submission submission, AuthenticationTags authenticationTags) {
        ValidationResult validationResult = this.fitConnectService.validateAttachments(attachmentForValidation, authenticationTags);
        this.evaluateValidationResult(submission, validationResult);
        this.scanAttachmentsForMalware(attachmentForValidation, submission);
    }

    private void validateData(Submission submission, Metadata metadata, byte[] decryptedData, String authenticationTags) {
        ValidationResult validationResult = this.fitConnectService.validateData(decryptedData, submission.getEncryptedData(), metadata, authenticationTags);
        this.evaluateValidationResult(submission, validationResult);
        this.scanDataForMalware(submission, decryptedData);
    }

    private void scanAttachmentsForMalware(List<AttachmentForValidation> attachmentsForValidation, Submission submission) {
        List<Attachment> attachments = attachmentsForValidation.stream().map(AttachmentMapper::toApiAttachment).collect(Collectors.toList());
        VirusScanResult result = this.malwareScanner.scanAttachments(attachments);
        if (result.isInfected()) {
            LOGGER.error("Attachment is infected with virus {}", (Object)result.getSignature());
            this.rejectSubmissionWithProblem(submission, new Problem[0]);
            throw new SubmissionRequestException("Attachment is infected with virus: " + result.getSignature());
        }
    }

    private void scanDataForMalware(Submission submission, byte[] decryptedData) {
        VirusScanResult dataScanResult = this.malwareScanner.scanData(decryptedData);
        if (dataScanResult.isInfected()) {
            LOGGER.error("Submission data is infected with {}", (Object)dataScanResult.getResult());
            this.rejectSubmissionWithProblem(submission, new Problem[0]);
            throw new SubmissionRequestException("Submission data is infected with virus: " + dataScanResult.getSignature());
        }
    }

    private void evaluateValidationResult(Submission submission, ValidationResult validationResult) throws SubmissionRequestException {
        if (validationResult.hasProblems()) {
            this.rejectSubmissionWithProblem(submission, validationResult.getProblems().toArray(new Problem[0]));
            String errorMessage = validationResult.getProblems().stream().map(Problem::getDetail).collect(Collectors.joining());
            throw new SubmissionRequestException(errorMessage, validationResult.getError());
        }
        if (validationResult.hasError()) {
            LOGGER.error(validationResult.getError().getMessage(), (Throwable)validationResult.getError());
            throw new SubmissionRequestException(validationResult.getError().getMessage(), validationResult.getError());
        }
    }

    private ReceivedSubmission buildReceivedSubmission(Submission submission, Metadata metadata, byte[] decryptedData, List<AttachmentForValidation> attachments) {
        Map<UUID, String> attachmentAuthTags = AttachmentMapper.mapAttachmentIdsToAuthTags(attachments);
        Status submitState = this.fitConnectService.getSubmitState(submission);
        ReceivedSubmissionData receivedSubmissionData = ReceivedSubmissionData.builder().attachmentAuthTags(attachmentAuthTags).attachments(AttachmentMapper.getReceivedApiAttachments(attachments)).submissionStatus(submitState).submission(submission).metadata(metadata).data(decryptedData).build();
        return new ReceivedSubmission(this.fitConnectService, receivedSubmissionData);
    }

    private byte[] decryptData(Submission submission, List<AttachmentForValidation> attachments) {
        this.checkPrivateKey();
        try {
            return AttachmentMapper.getSubmissionDataFromAttachments(attachments).orElse(this.fitConnectService.decryptString(this.privateKey, submission.getEncryptedData()));
        }
        catch (DecryptionException e) {
            this.rejectSubmissionWithProblem(submission, new DataEncryptionIssue());
            throw new SubmissionRequestException(e.getMessage(), e);
        }
        catch (IOException e) {
            throw new SubmissionRequestException(e.getMessage(), e);
        }
    }

    private Metadata decryptMetadata(Submission submission) {
        this.checkPrivateKey();
        try {
            byte[] metadataBytes = this.fitConnectService.decryptString(this.privateKey, submission.getEncryptedMetadata());
            return MetadataDeserializationHelper.deserializeMetadata(MAPPER, metadataBytes);
        }
        catch (IOException e) {
            this.rejectSubmissionWithProblem(submission, new MetadataJsonSyntaxViolation());
            throw new SubmissionRequestException(e.getMessage(), e);
        }
        catch (DecryptionException e) {
            this.rejectSubmissionWithProblem(submission, new MetadataEncryptionIssue());
            throw new SubmissionRequestException(e.getMessage(), e);
        }
    }

    private void checkPrivateKey() {
        if (!SubmissionReceiver.isUnwrapKey(this.privateKey)) {
            throw new SubmissionRequestException("Private key is not suitable for decryption, could not find key_operation unwrapKey or key_use encryption");
        }
    }

    private static boolean isUnwrapKey(RSAKey privateKey) {
        return privateKey.getKeyOperations() != null && privateKey.getKeyOperations().stream().anyMatch(keyOp -> keyOp.identifier().equals(KeyOperation.UNWRAP_KEY.identifier()));
    }

    private void rejectSubmissionWithProblem(Submission submission, Problem ... problem) {
        this.reject(submission, List.of(problem));
    }

    private void reject(Submission submission, List<Problem> problems) {
        if (this.config.isAutoRejectEnabled()) {
            LOGGER.info("Auto-rejecting submission due the following problem(s): {}", (Object)problems.stream().map(Problem::getDetail).collect(Collectors.joining("\n")));
            this.fitConnectService.rejectSubmission(EventPayload.forRejectEvent(submission, problems));
        }
    }
}

