/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.client.sender;

import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.ApplicationConfig;
import dev.fitko.fitconnect.api.domain.limits.Limit;
import dev.fitko.fitconnect.api.domain.limits.submission.SubmissionLimits;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.attachment.AttachmentPayload;
import dev.fitko.fitconnect.api.domain.model.attachment.Fragment;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.domain.model.metadata.data.MimeType;
import dev.fitko.fitconnect.api.domain.model.submission.AnnounceSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.CreatedSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.SentSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.model.submission.SubmitSubmission;
import dev.fitko.fitconnect.api.domain.sender.SendableEncryptedSubmission;
import dev.fitko.fitconnect.api.domain.sender.SendableSubmission;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectSenderException;
import dev.fitko.fitconnect.client.attachments.AttachmentPayloadHandler;
import dev.fitko.fitconnect.client.attachments.upload.AttachmentUploader;
import dev.fitko.fitconnect.client.util.MalwareScanner;
import dev.fitko.fitconnect.client.util.MetadataBuilder;
import dev.fitko.fitconnect.client.util.SubmissionBuilder;
import dev.fitko.fitconnect.client.util.SubmissionValidator;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SubmissionSender {
    private static final Logger LOGGER = LoggerFactory.getLogger(SubmissionSender.class);
    private final ApplicationConfig config;
    private final FitConnectService fitConnectService;
    private final SubmissionValidator submissionValidator;
    private final AttachmentPayloadHandler attachmentPayloadHandler;
    private final AttachmentUploader attachmentUploader;
    private final MalwareScanner malwareScanner;

    public SubmissionSender(ApplicationConfig config, FitConnectService fitConnectService, SubmissionValidator submissionValidator, AttachmentPayloadHandler attachmentPayloadHandler, AttachmentUploader attachmentUploader) {
        this.config = config;
        this.fitConnectService = fitConnectService;
        this.submissionValidator = submissionValidator;
        this.attachmentPayloadHandler = attachmentPayloadHandler;
        this.attachmentUploader = attachmentUploader;
        this.malwareScanner = new MalwareScanner(config.getVirusScannerMode(), fitConnectService);
    }

    public SentSubmission sendSubmission(SendableSubmission sendableSubmission) {
        LOGGER.debug("Scanning data and attachments for malware");
        this.scanForMalware(sendableSubmission);
        PublicDestination destination = this.fitConnectService.getPublicDestination(sendableSubmission.getDestinationId());
        RSAKey encryptionKey = this.fitConnectService.getEncryptionKeyForDestination(destination.getDestinationId());
        this.submissionValidator.ensureValidDataPayload(sendableSubmission, destination);
        List<AttachmentPayload> attachmentPayloads = this.createAttachmentPayloads(sendableSubmission, destination);
        LOGGER.info("Announcing submission");
        CreatedSubmission announcedSubmission = this.fitConnectService.announceSubmission(SubmissionBuilder.buildCreateSubmission(sendableSubmission, attachmentPayloads));
        LOGGER.info("Uploading attachments");
        List<AttachmentPayload> uploadedAttachments = this.attachmentUploader.uploadAttachments(attachmentPayloads, announcedSubmission.getSubmissionId(), encryptionKey);
        LOGGER.info("Creating metadata with version {}", (Object)sendableSubmission.getMetadataVersion());
        Metadata metadata = MetadataBuilder.createMetadata(sendableSubmission, destination, uploadedAttachments);
        ValidationResult validatedMetadata = this.fitConnectService.validateMetadataSchema(metadata);
        if (validatedMetadata.hasError()) {
            LOGGER.error("Metadata does not match schema", (Throwable)validatedMetadata.getError());
            throw new FitConnectSenderException(validatedMetadata.getError().getMessage(), validatedMetadata.getError());
        }
        LOGGER.info("Encrypting submission data ...");
        String dataMimeType = sendableSubmission.getSubmissionSchema().getMimeType().value();
        String encryptedData = this.fitConnectService.encryptBytes(encryptionKey, sendableSubmission.getData(), dataMimeType);
        LOGGER.info("Encrypting metadata ...");
        String encryptedMetadata = this.fitConnectService.encryptObject(encryptionKey, metadata, MimeType.APPLICATION_JSON.value());
        Submission submission = this.fitConnectService.sendSubmission(SubmissionBuilder.buildSubmitSubmission(announcedSubmission.getSubmissionId(), encryptedData, encryptedMetadata));
        AuthenticationTags authenticationTags = this.buildAuthTags(encryptedData, encryptedMetadata, uploadedAttachments);
        LOGGER.info("SUCCESSFULLY HANDED IN SUBMISSION ! \n");
        return SubmissionBuilder.buildSentSubmission(submission, authenticationTags);
    }

    public SentSubmission sendEncryptedSubmission(SendableEncryptedSubmission sendableEncryptedSubmission) {
        PublicDestination destination = this.fitConnectService.getPublicDestination(sendableEncryptedSubmission.getDestinationId());
        this.submissionValidator.ensureValidDataPayload(sendableEncryptedSubmission, destination);
        Map<UUID, String> encryptedAttachments = sendableEncryptedSubmission.getAttachments();
        String encryptedData = sendableEncryptedSubmission.getData();
        String encryptedMetadata = sendableEncryptedSubmission.getMetadata();
        AnnounceSubmission submissionToAnnounce = SubmissionBuilder.buildCreateSubmission(sendableEncryptedSubmission, encryptedAttachments);
        UUID submissionId = this.fitConnectService.announceSubmission(submissionToAnnounce).getSubmissionId();
        SubmitSubmission submitSubmission = SubmissionBuilder.buildSubmitSubmission(submissionId, encryptedData, encryptedMetadata);
        this.uploadEncryptedAttachments(encryptedAttachments, submissionId);
        Submission submission = this.fitConnectService.sendSubmission(submitSubmission);
        LOGGER.info("SUCCESSFULLY HANDED IN SUBMISSION ! \n");
        AuthenticationTags authenticationTags = this.buildAuthTags(encryptedData, encryptedMetadata, encryptedAttachments);
        return SubmissionBuilder.buildSentSubmission(submission, authenticationTags);
    }

    private List<AttachmentPayload> createAttachmentPayloads(SendableSubmission sendableSubmission, PublicDestination destination) {
        List<Attachment> attachments = sendableSubmission.getAttachments();
        if (attachments.stream().anyMatch(a -> a.getPurpose().equals((Object)Purpose.DATA))) {
            LOGGER.info("Submission data exceeds allowed limit and will be sent as attachment");
        }
        if (this.submissionValidator.destinationSupportsAttachmentChunking(destination, attachments, this.config.getAttachmentChunkingConfig())) {
            SubmissionLimits attachmentLimits = this.fitConnectService.getDestinationAttachmentLimits(destination.getDestinationId());
            Limit submissionAttachmentLimit = attachmentLimits.getAttachments().getSubmissionLimits().getApplicable();
            return this.attachmentPayloadHandler.createChunkedPayloads(attachments, submissionAttachmentLimit);
        }
        return this.attachmentPayloadHandler.createPayloadsWithoutChunking(attachments);
    }

    public void uploadEncryptedAttachments(Map<UUID, String> encryptedAttachments, UUID submissionId) {
        if (encryptedAttachments.isEmpty()) {
            LOGGER.info("No attachments to upload");
        } else {
            LOGGER.info("Uploading {} attachment(s)", (Object)encryptedAttachments.size());
            encryptedAttachments.forEach((key, value) -> this.fitConnectService.uploadSubmissionAttachment(submissionId, (UUID)key, (String)value));
        }
    }

    private Map<UUID, String> mapAttachmentIdsToAuthTags(List<AttachmentPayload> attachmentPayloads) {
        if (attachmentPayloads == null || attachmentPayloads.isEmpty()) {
            return Collections.emptyMap();
        }
        LinkedHashMap<UUID, String> attachmentAuthTags = new LinkedHashMap<UUID, String>();
        for (AttachmentPayload attachment : attachmentPayloads) {
            if (attachment.hasFragmentedPayload()) {
                attachmentAuthTags.putAll(attachment.getFragments().stream().collect(Collectors.toMap(Fragment::getFragmentId, Fragment::getAuthTag)));
                continue;
            }
            attachmentAuthTags.put(attachment.getAttachmentId(), attachment.getAuthTag());
        }
        return attachmentAuthTags;
    }

    private Map<UUID, String> mapAttachmentIdsToAuthTags(Map<UUID, String> encryptedAttachments) {
        return encryptedAttachments.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> AuthenticationTags.getAuthTagFromJWT((String)e.getValue())));
    }

    private AuthenticationTags buildAuthTags(String encryptedData, String encryptedMetadata, List<AttachmentPayload> uploadedAttachments) {
        String dataAuthTag = AuthenticationTags.getAuthTagFromJWT(encryptedData);
        String metadataAuthTag = AuthenticationTags.getAuthTagFromJWT(encryptedMetadata);
        Map<UUID, String> attachmentAuthTags = this.mapAttachmentIdsToAuthTags(uploadedAttachments);
        return new AuthenticationTags(dataAuthTag, metadataAuthTag, attachmentAuthTags);
    }

    private AuthenticationTags buildAuthTags(String encryptedData, String encryptedMetadata, Map<UUID, String> encryptedAttachments) {
        String dataAuthTag = AuthenticationTags.getAuthTagFromJWT(encryptedData);
        String metadataAuthTag = AuthenticationTags.getAuthTagFromJWT(encryptedMetadata);
        Map<UUID, String> attachmentAuthTags = this.mapAttachmentIdsToAuthTags(encryptedAttachments);
        return new AuthenticationTags(dataAuthTag, metadataAuthTag, attachmentAuthTags);
    }

    private void scanForMalware(SendableSubmission sendableSubmission) {
        VirusScanResult dataScanResult = this.malwareScanner.scanData(sendableSubmission.getData());
        if (dataScanResult.isInfected()) {
            LOGGER.error("Data is infected with {}", (Object)dataScanResult.getSignature());
            throw new FitConnectSenderException("Data is infected with virus: " + dataScanResult.getSignature());
        }
        VirusScanResult attachmentsScanResult = this.malwareScanner.scanAttachments(sendableSubmission.getAttachments());
        if (attachmentsScanResult.isInfected()) {
            LOGGER.error("Attachment is infected with virus {}", (Object)attachmentsScanResult.getResult());
            throw new FitConnectSenderException("Attachment " + attachmentsScanResult.getResult() + " is infected with virus: " + attachmentsScanResult.getSignature());
        }
    }

    private void scanForMalware(Metadata metadata) {
        VirusScanResult dataScanResult = this.malwareScanner.scanMetadata(metadata);
        if (dataScanResult.isInfected()) {
            LOGGER.error("Metadata is infected with {}", (Object)dataScanResult.getSignature());
            throw new FitConnectSenderException("Metadata is infected with virus: " + dataScanResult.getSignature());
        }
    }
}

