/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.client.sender;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.ApplicationConfig;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.authtags.ValidatedAuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.event.problems.data.DataEncryptionIssue;
import dev.fitko.fitconnect.api.domain.model.event.problems.metadata.MetadataEncryptionIssue;
import dev.fitko.fitconnect.api.domain.model.event.problems.metadata.MetadataJsonSyntaxViolation;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.AttachmentForValidation;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.sender.ReceivedReply;
import dev.fitko.fitconnect.api.domain.sender.ReceivedReplyData;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectReplyException;
import dev.fitko.fitconnect.api.exceptions.internal.DecryptionException;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.exceptions.internal.SubmissionRequestException;
import dev.fitko.fitconnect.client.attachments.download.AttachmentDownloader;
import dev.fitko.fitconnect.client.util.AttachmentMapper;
import dev.fitko.fitconnect.client.util.MalwareScanner;
import dev.fitko.fitconnect.client.util.MetadataDeserializationHelper;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplyReceiver {
    private static final Logger LOGGER = LoggerFactory.getLogger(ReplyReceiver.class);
    private static final ObjectMapper MAPPER = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    private final ApplicationConfig config;
    private final FitConnectService fitConnectService;
    private final AttachmentDownloader<Reply> attachmentDownloader;
    private final MalwareScanner malwareScanner;

    public ReplyReceiver(ApplicationConfig config, FitConnectService fitConnectService, AttachmentDownloader<Reply> attachmentDownloader) {
        this.config = config;
        this.fitConnectService = fitConnectService;
        this.attachmentDownloader = attachmentDownloader;
        this.malwareScanner = new MalwareScanner(config.getVirusScannerMode(), fitConnectService);
    }

    public ReceivedReply receiveReply(UUID replyId, RSAKey replyDecryptionKey) {
        LOGGER.info("Requesting reply");
        Reply reply = this.loadReply(replyId);
        LOGGER.info("Loading authentication tags from event log");
        AuthenticationTags authenticationTagsFromEvent = this.loadAuthTagsForSubmitEvent(reply);
        LOGGER.info("Decrypting reply metadata");
        Metadata metadata = this.decryptMetadata(replyDecryptionKey, reply);
        this.validateMetadata(reply, authenticationTagsFromEvent, metadata);
        LOGGER.info("Loading reply attachments");
        List<ApiAttachment> apiAttachments = metadata.getContentStructure().getAttachments();
        List<AttachmentForValidation> attachments = this.attachmentDownloader.loadAttachments(reply, replyDecryptionKey, apiAttachments);
        AuthenticationTags authenticationTags = ReplyReceiver.buildAuthenticationTags(reply, attachments);
        this.validateAttachments(reply, attachments, authenticationTags);
        LOGGER.info("Decrypting reply data");
        byte[] decryptedData = this.decryptData(replyDecryptionKey, reply, attachments);
        this.validateData(reply, authenticationTagsFromEvent, metadata, decryptedData);
        return this.buildReceivedReply(reply, metadata, decryptedData, attachments, authenticationTags);
    }

    private void validateAttachments(Reply reply, List<AttachmentForValidation> attachments, AuthenticationTags authenticationTags) {
        ValidationResult validatedAttachments = this.fitConnectService.validateAttachments(attachments, authenticationTags);
        if (!validatedAttachments.isValid()) {
            LOGGER.error("Attachment validation failed", (Throwable)validatedAttachments.getError());
            this.evaluateValidationResult(reply, validatedAttachments);
        }
    }

    private void validateData(Reply reply, AuthenticationTags authenticationTagsFromEvent, Metadata metadata, byte[] decryptedData) {
        ValidationResult validatedData = this.fitConnectService.validateData(decryptedData, reply.getEncryptedData(), metadata, authenticationTagsFromEvent.getData());
        if (!validatedData.isValid()) {
            LOGGER.error("Data validation failed", (Throwable)validatedData.getError());
            this.evaluateValidationResult(reply, validatedData);
        }
    }

    private void validateMetadata(Reply reply, AuthenticationTags authenticationTagsFromEvent, Metadata metadata) {
        ValidationResult validatedMetadata = this.fitConnectService.validateReplyMetadata(metadata, reply, authenticationTagsFromEvent);
        if (!validatedMetadata.isValid()) {
            LOGGER.error("Metadata validation failed", (Throwable)validatedMetadata.getError());
            this.evaluateValidationResult(reply, validatedMetadata);
        }
    }

    private Reply loadReply(UUID replyId) {
        try {
            return this.fitConnectService.getReply(replyId);
        }
        catch (RestApiException e) {
            throw new FitConnectReplyException(e.getMessage(), e);
        }
    }

    private AuthenticationTags loadAuthTagsForSubmitEvent(Reply reply) {
        ValidatedAuthenticationTags validatedAuthenticationTags = this.fitConnectService.getReplyAuthenticationTags(reply);
        if (validatedAuthenticationTags.causesRejection()) {
            this.rejectReplyWithProblem(reply, validatedAuthenticationTags.getProblem());
            throw new FitConnectReplyException(validatedAuthenticationTags.getErrorMessage());
        }
        return validatedAuthenticationTags.getAuthenticationTags();
    }

    private static AuthenticationTags buildAuthenticationTags(Reply reply, List<AttachmentForValidation> attachments) {
        String dataAuthTag = AuthenticationTags.getAuthTagFromJWT(reply.getEncryptedData());
        String metadataAuthTag = AuthenticationTags.getAuthTagFromJWT(reply.getEncryptedMetadata());
        Map<UUID, String> attachmentAuthTags = AttachmentMapper.mapAttachmentIdsToAuthTags(attachments);
        return new AuthenticationTags(dataAuthTag, metadataAuthTag, attachmentAuthTags);
    }

    private ReceivedReply buildReceivedReply(Reply reply, Metadata metadata, byte[] decryptedData, List<AttachmentForValidation> attachments, AuthenticationTags authenticationTags) {
        List<Attachment> receivedAttachments = AttachmentMapper.getReceivedApiAttachments(attachments);
        ReceivedReplyData receivedData = new ReceivedReplyData(decryptedData, metadata, reply, receivedAttachments, authenticationTags);
        return new ReceivedReply(this.fitConnectService, receivedData);
    }

    private byte[] decryptData(RSAKey privateKey, Reply reply, List<AttachmentForValidation> attachments) {
        try {
            return AttachmentMapper.getSubmissionDataFromAttachments(attachments).orElse(this.fitConnectService.decryptString(privateKey, reply.getEncryptedData()));
        }
        catch (DecryptionException e) {
            this.reject(reply, List.of(new DataEncryptionIssue()));
            throw new FitConnectReplyException(e.getMessage(), e);
        }
        catch (IOException e) {
            throw new SubmissionRequestException(e.getMessage(), e);
        }
    }

    private Metadata decryptMetadata(RSAKey replyDecryptionKey, Reply reply) {
        try {
            byte[] metadataBytes = this.fitConnectService.decryptString(replyDecryptionKey, reply.getEncryptedMetadata());
            return MetadataDeserializationHelper.deserializeMetadata(MAPPER, metadataBytes);
        }
        catch (IOException e) {
            this.reject(reply, List.of(new MetadataJsonSyntaxViolation()));
            throw new FitConnectReplyException(e.getMessage(), e);
        }
        catch (DecryptionException e) {
            this.reject(reply, List.of(new MetadataEncryptionIssue()));
            throw new FitConnectReplyException(e.getMessage(), e);
        }
    }

    private void evaluateValidationResult(Reply reply, ValidationResult validationResult) throws SubmissionRequestException {
        if (validationResult.hasProblems()) {
            this.reject(reply, validationResult.getProblems());
            throw new FitConnectReplyException(validationResult.getProblems().stream().map(Problem::getDetail).collect(Collectors.joining()), validationResult.getError());
        }
        LOGGER.error(validationResult.getError().getMessage(), (Throwable)validationResult.getError());
        throw new FitConnectReplyException(validationResult.getError().getMessage(), validationResult.getError());
    }

    private void rejectReplyWithProblem(Reply reply, Problem ... problem) {
        this.reject(reply, List.of(problem));
    }

    private void reject(Reply reply, List<Problem> problems) {
        if (this.config.isAutoRejectEnabled()) {
            LOGGER.info("Auto-rejecting reply due the following problem(s): {}", (Object)problems.stream().map(Problem::getDetail).collect(Collectors.joining("\n")));
            this.fitConnectService.rejectReply(reply.getReplyId(), problems);
        }
    }
}

