/*
 * Decompiled with CFR 0.152.
 */
package dev.fitko.fitconnect.client.attachments;

import dev.fitko.fitconnect.api.config.chunking.AttachmentChunkingConfig;
import dev.fitko.fitconnect.api.config.chunking.ChunkSize;
import dev.fitko.fitconnect.api.domain.limits.Limit;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.attachment.AttachmentPayload;
import dev.fitko.fitconnect.api.domain.model.attachment.Fragment;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectAttachmentException;
import dev.fitko.fitconnect.api.services.crypto.MessageDigestService;
import dev.fitko.fitconnect.client.attachments.AttachmentStorageResolver;
import dev.fitko.fitconnect.core.io.FileChunker;
import dev.fitko.fitconnect.core.utils.Formatter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AttachmentPayloadHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(AttachmentPayloadHandler.class);
    public static final double ENCRYPTION_OVERHEAD_BUFFER_FACTOR = 0.6;
    private ChunkSize chunkSize;
    private final AttachmentChunkingConfig config;
    private final FileChunker chunker;
    private final MessageDigestService hashService;
    private final AttachmentStorageResolver attachmentStorageResolver;

    public AttachmentPayloadHandler(AttachmentChunkingConfig config, FileChunker chunker, AttachmentStorageResolver attachmentStorageResolver, MessageDigestService hashService) {
        this.config = config;
        this.chunker = chunker;
        this.hashService = hashService;
        this.chunkSize = config.getChunkSizeObject();
        this.attachmentStorageResolver = attachmentStorageResolver;
    }

    public List<AttachmentPayload> createChunkedPayloads(List<Attachment> attachments, Limit attachmentLimit) {
        long maxAllowedBytes = attachmentLimit.getMaxSizeIndividualBytes();
        if (this.chunkSize == null || (long)this.chunkSize.getSizeInBytes() > maxAllowedBytes) {
            this.chunkSize = ChunkSize.ofMB((int)((double)Formatter.toMegaBytes(maxAllowedBytes) * 0.6));
        }
        if (this.config.isChunkAllAttachments()) {
            LOGGER.info("Chunking in-memory and large attachments");
            return this.buildChunkedPayloads(attachments);
        }
        return this.buildChunkedPayloadsFromLargeAttachmentsOnly(attachments);
    }

    public List<AttachmentPayload> createPayloadsWithoutChunking(List<Attachment> attachments) {
        return attachments.stream().map(this::buildPayloadWithoutFragments).collect(Collectors.toList());
    }

    public File writeReceivedFragmentToFileSystem(UUID attachmentId, int fragmentIndex, byte[] attachmentData) {
        try {
            Path chunkedFilePath = this.attachmentStorageResolver.resolveIncomingAttachmentFolder(attachmentId);
            return this.chunker.writeFileChunk(attachmentData, fragmentIndex, chunkedFilePath);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public Optional<File> mergeChunks(ApiAttachment apiAttachment) {
        try {
            Path path = this.attachmentStorageResolver.resolveIncomingAttachmentFolder(apiAttachment.getAttachmentId());
            return this.chunker.concatChunks(path, apiAttachment.getFilename());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private List<AttachmentPayload> buildChunkedPayloads(List<Attachment> attachments) {
        return attachments.stream().map(this::buildChunkedPayload).collect(Collectors.toList());
    }

    private AttachmentPayload buildChunkedPayload(Attachment attachment) {
        return attachment.isInMemoryAttachment() ? this.chunkInMemoryData(attachment) : this.chunkLargeFileData(attachment);
    }

    private List<AttachmentPayload> buildChunkedPayloadsFromLargeAttachmentsOnly(List<Attachment> attachments) {
        List<AttachmentPayload> largePayloads = AttachmentPayloadHandler.mapByType(attachments, Attachment::isLargeAttachment, this::chunkLargeFileData);
        List<AttachmentPayload> inMemoryPayloads = AttachmentPayloadHandler.mapByType(attachments, Attachment::isInMemoryAttachment, this::buildPayloadWithoutFragments);
        return Stream.concat(largePayloads.stream(), inMemoryPayloads.stream()).collect(Collectors.toList());
    }

    private static List<AttachmentPayload> mapByType(List<Attachment> attachments, Predicate<Attachment> filter, Function<Attachment, AttachmentPayload> mapper) {
        return attachments.stream().filter(filter).map(mapper).collect(Collectors.toList());
    }

    private AttachmentPayload chunkInMemoryData(Attachment attachment) {
        try {
            byte[] inMemoryData = attachment.getDataAsBytes();
            LOGGER.info("Chunking in-memory data {} with a size of {}", (Object)attachment.getFileName(), (Object)Formatter.toHumanReadableSizePrefix(inMemoryData.length));
            if (this.dataSizeIsLessOrEqualThanChunkSize(inMemoryData.length)) {
                return this.buildPayloadWithoutFragments(attachment);
            }
            String originalDataHash = this.hashBytes(inMemoryData);
            List<Fragment> attachmentFragments = this.buildFragmentsFromInMemoryAttachment(attachment);
            AttachmentPayload chunkedAttachment = this.buildFragmentedAttachmentPayload(attachment, originalDataHash, attachmentFragments);
            LOGGER.info("Split data into {} chunks", (Object)attachmentFragments.size());
            return chunkedAttachment;
        }
        catch (IOException e) {
            throw new FitConnectAttachmentException(e.getMessage(), e);
        }
    }

    private AttachmentPayload chunkLargeFileData(Attachment attachment) {
        try {
            Path filePath = attachment.getLargeAttachmentFilePath();
            long fileSize = this.getFileSize(filePath);
            LOGGER.info("Chunking file {} with a size of {}", (Object)attachment.getFileName(), (Object)Formatter.toHumanReadableSizePrefix(fileSize));
            if (this.dataSizeIsLessOrEqualThanChunkSize(fileSize)) {
                return this.buildPayloadWithoutFragments(attachment);
            }
            String originalFileHash = this.hashFile(filePath);
            List<Fragment> attachmentFragments = this.buildFragmentsFromLargeAttachment(attachment);
            AttachmentPayload chunkedAttachment = this.buildFragmentedAttachmentPayload(attachment, originalFileHash, attachmentFragments);
            this.removeBufferedFile(attachment, filePath);
            LOGGER.info("Split file {} into {} chunks", (Object)filePath, (Object)attachmentFragments.size());
            return chunkedAttachment;
        }
        catch (IOException e) {
            throw new FitConnectAttachmentException(e.getMessage(), e);
        }
    }

    private List<Fragment> buildFragmentsFromLargeAttachment(Attachment attachment) throws IOException {
        try (InputStream inputStream = Files.newInputStream(attachment.getLargeAttachmentFilePath(), new OpenOption[0]);){
            List<Fragment> list = this.chunkDataToFragments(attachment.getFileName(), this.getOutgoingAttachmentPath(), inputStream);
            return list;
        }
    }

    private List<Fragment> buildFragmentsFromInMemoryAttachment(Attachment attachment) throws IOException {
        try (InputStream inputStream = attachment.getDataAsInputStream();){
            List<Fragment> list = this.chunkDataToFragments(attachment.getFileName(), this.getOutgoingAttachmentPath(), inputStream);
            return list;
        }
    }

    private List<Fragment> chunkDataToFragments(String fileName, Path path, InputStream inputStream) throws IOException {
        return this.chunker.chunkStream(inputStream, this.chunkSize.getSizeInBytes(), fileName, path).stream().map(file -> new Fragment(UUID.randomUUID(), (File)file)).collect(Collectors.toList());
    }

    private long getFileSize(Path dataFilePath) {
        long l;
        block8: {
            FileChannel channel = FileChannel.open(dataFilePath, new OpenOption[0]);
            try {
                l = channel.size();
                if (channel == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (channel != null) {
                        try {
                            channel.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new FitConnectAttachmentException(e.getMessage(), e);
                }
            }
            channel.close();
        }
        return l;
    }

    private boolean dataSizeIsLessOrEqualThanChunkSize(long dataLength) {
        if (dataLength > (long)this.chunkSize.getSizeInBytes()) {
            return false;
        }
        LOGGER.info("Attachment will not be chunked because its size is smaller than the chunk size of {} MB", (Object)this.chunkSize.getSizeInMB());
        return true;
    }

    private Path getOutgoingAttachmentPath() throws IOException {
        return this.attachmentStorageResolver.resolveOutgoingAttachmentFolder(UUID.randomUUID());
    }

    private String hashFile(Path filePath) throws IOException {
        try (InputStream fileStream = Files.newInputStream(filePath, new OpenOption[0]);){
            byte[] hash = this.hashService.createHash(fileStream);
            String string = this.hashService.toHexString(hash);
            return string;
        }
    }

    private String hashBytes(byte[] data) {
        byte[] hash = this.hashService.createHash(data);
        return this.hashService.toHexString(hash);
    }

    private void removeBufferedFile(Attachment attachment, Path filePath) {
        if (attachment.getPurpose().equals((Object)Purpose.DATA) || filePath.toString().contains("temp_fit_connect_attachment_")) {
            filePath.toFile().delete();
        }
    }

    private AttachmentPayload buildFragmentedAttachmentPayload(Attachment attachment, String hashedData, List<Fragment> fragments) {
        return AttachmentPayload.builder().mimeType(attachment.getMimeType()).fileName(attachment.getFileName()).description(attachment.getDescription()).purpose(attachment.getPurpose()).attachmentId(UUID.randomUUID()).hashedData(hashedData).fragments(fragments).build();
    }

    private AttachmentPayload buildPayloadWithoutFragments(Attachment attachment) {
        return AttachmentPayload.builder().mimeType(attachment.getMimeType()).fileName(attachment.getFileName()).description(attachment.getDescription()).purpose(attachment.getPurpose()).attachmentId(UUID.randomUUID()).data(attachment.getDataAsBytes()).build();
    }
}

