package dev.fitko.fitconnect.tools.keygen;

import static com.nimbusds.jose.jwk.KeyOperation.SIGN;
import static com.nimbusds.jose.jwk.KeyOperation.UNWRAP_KEY;
import static com.nimbusds.jose.jwk.KeyOperation.VERIFY;
import static com.nimbusds.jose.jwk.KeyOperation.WRAP_KEY;

import com.nimbusds.jose.JWEAlgorithm;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.util.Base64;
import dev.fitko.fitconnect.api.domain.crypto.JWKPair;
import dev.fitko.fitconnect.core.crypto.utils.CertUtils;
import dev.fitko.fitconnect.core.crypto.utils.KeyGenerator;
import java.security.KeyPair;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.List;
import java.util.UUID;

/**
 * JWK Test Key Generator.
 *
 * <p>Generates key pairs of public and private keys for encryption and signing.
 */
public final class TestKeyBuilder {

    public static final int DEFAULT_KEY_SIZE = 4096;
    public static final Duration CERT_VALIDITY = Duration.ofDays(365);

    private TestKeyBuilder() {}

    /**
     * Generate a set of public encryption key and private decryption key.
     *
     * @param keySize size of the RSA key in bits
     * @return JWKPair of a public and private key
     */
    public static JWKPair generateEncryptionKeyPair(final int keySize) {

        final String keyId = UUID.randomUUID().toString();
        final JWEAlgorithm encryptionAlgorithm = JWEAlgorithm.RSA_OAEP_256;

        final KeyPair keyPair = KeyGenerator.buildRSAKeyPair(keySize);
        final List<Base64> x509CertChain = buildX509CertificateChain(keyPair);

        final JWK publicEncryptionKey =
                KeyGenerator.buildJWK(keyPair, keyId, WRAP_KEY, encryptionAlgorithm, x509CertChain);
        final JWK privateDecryptionKey = KeyGenerator.buildJWK(keyPair, keyId, UNWRAP_KEY, encryptionAlgorithm);

        return new JWKPair(publicEncryptionKey.toPublicJWK(), privateDecryptionKey);
    }

    /**
     * Generate a set of public encryption key and private decryption key with a default key-length of
     * 4096 bit.
     *
     * @return JWKPair of a public and private key
     * @see TestKeyBuilder#DEFAULT_KEY_SIZE
     */
    public static JWKPair generateEncryptionKeyPair() {
        return generateEncryptionKeyPair(DEFAULT_KEY_SIZE);
    }

    /**
     * Generate a set of public signature verification key and private signature key.
     *
     * @param keySize size of the RSA key in bits
     * @return JWKPair of a signature and verification key
     */
    public static JWKPair generateSignatureKeyPair(final int keySize) {

        final String keyId = UUID.randomUUID().toString();
        final JWSAlgorithm signingAlgorithm = JWSAlgorithm.PS512;

        final KeyPair keyPair = KeyGenerator.buildRSAKeyPair(keySize);
        final List<Base64> x509CertChain = buildX509CertificateChain(keyPair);

        final JWK publicSignatureVerificationKey =
                KeyGenerator.buildJWK(keyPair, keyId, VERIFY, signingAlgorithm, x509CertChain);
        final JWK privateSignatureKey = KeyGenerator.buildJWK(keyPair, keyId, SIGN, signingAlgorithm);

        return new JWKPair(publicSignatureVerificationKey.toPublicJWK(), privateSignatureKey);
    }

    /**
     * Generate a set of public signature verification key and private signature key with default
     * key-length of 4096 bit.
     *
     * @return JWKPair of a signature and verification key
     * @see TestKeyBuilder#DEFAULT_KEY_SIZE
     */
    public static JWKPair generateSignatureKeyPair() {
        return generateSignatureKeyPair(DEFAULT_KEY_SIZE);
    }

    private static List<Base64> buildX509CertificateChain(KeyPair keyPair) {
        final X509Certificate certificate = CertUtils.generateX509Certificate(keyPair, CERT_VALIDITY);
        try {
            return List.of(Base64.encode(certificate.getEncoded()));
        } catch (CertificateEncodingException e) {
            throw new RuntimeException(e);
        }
    }
}
