package dev.fitko.fitconnect.tools.keygen;

import com.nimbusds.jose.jwk.JWK;
import dev.fitko.fitconnect.api.config.defaults.Environments;
import dev.fitko.fitconnect.api.domain.crypto.JWKPair;
import dev.fitko.fitconnect.api.exceptions.FitConnectException;
import java.io.IOException;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.Yaml;

@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class KeyWriter {

    private static final Logger LOGGER = LoggerFactory.getLogger(KeyWriter.class);

    static final String PUBLIC_ENCRYPTION_KEY_NAME = "publicKey_encryption.json";
    static final String PUBLIC_ENCRYPTION_KEY_DESC = "Encryption Public Key (key_use=wrapKey)";
    static final String PRIVATE_DECRYPTION_KEY_NAME = "privateKey_decryption.json";
    static final String PRIVATE_DECRYPTION_KEY_DESC = "Decryption Private Key (key_use=unwrapKey)";
    static final String PUBLIC_SIGNATURE_VERIFICATION_KEY_NAME = "publicKey_signature_verification.json";
    static final String PUBLIC_SIGNATURE_VERIFICATION_KEY_DESC = "Signature Verification Public Key (key_use=verify)";
    static final String PRIVATE_SIGNING_KEY_NAME = "privateKey_signing.json";
    static final String PRIVATE_SIGNING_KEY_DESC = "Signing Private Key (key_use=sign)";
    static final String CONFIG_YML_FILE_NAME = "config.yml";
    static final String TEMP_DIR_NAME = "testJWKs";

    public static void writeKeys(final KeyWriterSettings settings) {

        final Path dir = getKeyDirectory(settings.getOutputDir());
        LOGGER.info("Writing keys to directory {}", dir);

        final JWKPair encryptionKeyPair = settings.getEncryptionKeyPair();
        final JWKPair signatureKeyPair = settings.getSignatureKeyPair();

        writeKeyToFile(dir, PUBLIC_ENCRYPTION_KEY_NAME, PUBLIC_ENCRYPTION_KEY_DESC, encryptionKeyPair.getPublicKey());
        writeKeyToFile(
                dir, PRIVATE_DECRYPTION_KEY_NAME, PRIVATE_DECRYPTION_KEY_DESC, encryptionKeyPair.getPrivateKey());

        writeKeyToFile(
                dir,
                PUBLIC_SIGNATURE_VERIFICATION_KEY_NAME,
                PUBLIC_SIGNATURE_VERIFICATION_KEY_DESC,
                signatureKeyPair.getPublicKey());
        writeKeyToFile(dir, PRIVATE_SIGNING_KEY_NAME, PRIVATE_SIGNING_KEY_DESC, signatureKeyPair.getPrivateKey());

        if (settings.isCreateConfigYaml()) {
            writeConfigYamlToFile(dir);
        }
    }

    private static Path getKeyDirectory(final Path outputDir) {
        if (outputDir != null) {
            return outputDir.toAbsolutePath();
        }
        try {
            return Files.createTempDirectory(TEMP_DIR_NAME);
        } catch (final IOException e) {
            LOGGER.error(e.getMessage(), e);
            throw new FitConnectException(e.getMessage(), e);
        }
    }

    private static void writeConfigYamlToFile(final Path dir) {

        final Map<String, Object> data = new LinkedHashMap<>();
        data.put("senderConfig", Map.of("clientSecret", "", "clientId", ""));
        data.put(
                "subscriberConfig",
                Map.of(
                        "clientSecret",
                        "",
                        "clientId",
                        "",
                        "privateDecryptionKeyPaths",
                        List.of(dir.resolve(PRIVATE_DECRYPTION_KEY_NAME)
                                .toAbsolutePath()
                                .toString()),
                        "privateSigningKeyPath",
                        dir.resolve(PRIVATE_SIGNING_KEY_NAME).toAbsolutePath().toString()));
        data.put("activeEnvironment", Environments.TEST.getEnvironmentName().getName());

        final DumperOptions options = new DumperOptions();
        options.setDefaultScalarStyle(DumperOptions.ScalarStyle.PLAIN);
        options.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK);
        options.setPrettyFlow(true);

        final StringWriter writer = new StringWriter();
        final Yaml yaml = new Yaml(options);
        yaml.dump(data, writer);
        writer.flush();

        writeFile(dir.resolve(CONFIG_YML_FILE_NAME), writer.toString());

        LOGGER.info("Wrote config.yml");
    }

    private static void writeKeyToFile(final Path dir, final String filename, final String desc, final JWK jwk) {
        writeFile(dir.resolve(filename), jwk.toJSONString());
        LOGGER.info("Wrote {} as {}", desc, filename);
    }

    private static void writeFile(final Path path, final String content) {
        try {
            Files.write(path, content.getBytes(StandardCharsets.UTF_8));
        } catch (final IOException e) {
            LOGGER.error(e.getMessage(), e);
            throw new FitConnectException(e.getMessage(), e);
        }
    }
}
