package dev.fitko.fitconnect.core.validation.xml;

import static dev.fitko.fitconnect.core.validation.xml.AllowedSchemaPatterns.cannotBeValidated;
import static dev.fitko.fitconnect.core.validation.xml.AllowedSchemaPatterns.values;

import dev.fitko.fitconnect.api.domain.model.event.problems.data.DataSchemaViolation;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.services.schema.XMLSchemaLoader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.URI;
import java.util.Map;
import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

public class XmlSchemaValidator {

    private static final Logger LOGGER = LoggerFactory.getLogger(XmlSchemaValidator.class);
    private final SchemaFactory schemaFactory;
    private final XMLSchemaLoader xmlSchemaLoader;

    public XmlSchemaValidator(XMLSchemaLoader xmlSchemaLoader) {
        this.xmlSchemaLoader = xmlSchemaLoader;
        this.schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
    }

    public ValidationResult validate(byte[] xmlData, URI schemaURI) {

        if (cannotBeValidated(schemaURI)) {
            LOGGER.info("XML schema validation was skipped, no valid urn-schema was found. Allowed is {}", values());
            return ValidationResult.ok();
        }

        try {

            LOGGER.info("Validating XML against schema {}", schemaURI);
            Map<String, byte[]> entries = xmlSchemaLoader.loadSchemas(schemaURI);
            entries.keySet().forEach(s -> LOGGER.debug("Loaded schema {}", s));

            var resolver = new InMemoryResourceResolver(entries);

            // systemId is used to resolve includes
            var systemId = SchemaSystemIdExtractor.extractSystemId(xmlData);
            byte[] rootBytes = resolver.getBytesForSystemId(systemId);
            if (rootBytes == null) {
                return ValidationResult.error("Root XSD '" + systemId + "' could not be resolved in xml validation");
            }

            Source rootSource = new StreamSource(new ByteArrayInputStream(rootBytes));
            rootSource.setSystemId(systemId);

            schemaFactory.setResourceResolver(resolver);

            Schema schema = schemaFactory.newSchema(rootSource);
            Validator validator = schema.newValidator();
            validator.validate(new StreamSource(new ByteArrayInputStream(xmlData)));

        } catch (RuntimeException | SAXException | IOException e) {
            LOGGER.error(e.getMessage(), e);
            return ValidationResult.withErrorAndProblem(e, new DataSchemaViolation());
        }

        return ValidationResult.ok();
    }
}
