package dev.fitko.fitconnect.core.validation.xml;

import static dev.fitko.fitconnect.api.config.ApplicationConfig.LEIKA_KEY_PATTERN;

import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.services.http.HttpClient;
import dev.fitko.fitconnect.api.services.schema.XMLSchemaLoader;
import java.io.*;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Schema-loader that fetches zipped XML schema definitions from www.xrepository.de
 *
 * @see <a href="https://www.xrepository.de/cms/hilfe.html#rest-api">XRepo RestAPI</a>
 */
public class XRepoSchemaLoader implements XMLSchemaLoader {

    private static final Logger LOGGER = LoggerFactory.getLogger(XRepoSchemaLoader.class);
    private static final String XOEV_SCHEMA_URL = "https://www.xrepository.de/api/version_standard/%s/xmlschema";
    private static final Map<String, String> REQUEST_HEADER = Map.of("Accept", "application/zip");
    private final HttpClient httpClient;

    public XRepoSchemaLoader(HttpClient httpClient) {
        this.httpClient = httpClient;
    }

    @Override
    public Map<String, byte[]> loadSchemas(URI schemaUri) throws RestApiException {
        validateURIPattern(schemaUri);
        var schemaUrl = String.format(XOEV_SCHEMA_URL, schemaUri);
        LOGGER.info("Requesting XML schemas from {} ", schemaUrl);
        var httpResponse = httpClient.get(schemaUrl, REQUEST_HEADER, InputStream.class);
        return extractZipEntries(httpResponse.getBody());
    }

    /**
     * Extract schema files from a zipped input-stream.
     *
     * @param zippedFiles the zipped input-stream
     * @return map of zip entry name → byte[] of the schema data
     */
    Map<String, byte[]> extractZipEntries(final InputStream zippedFiles) {
        try (var zipInputStream = new ZipInputStream(zippedFiles)) {
            Map<String, byte[]> schemaDefinitions = new HashMap<>();
            ZipEntry zipEntry;

            while ((zipEntry = zipInputStream.getNextEntry()) != null) {
                if (zipEntry.isDirectory()) {
                    LOGGER.debug("Skipping directory entry {}", zipEntry.getName());
                    continue;
                }

                try (var schemaStream = new ByteArrayOutputStream()) {
                    zipInputStream.transferTo(schemaStream);
                    byte[] bytes = schemaStream.toByteArray();
                    LOGGER.debug("Unzipped {}", zipEntry.getName());
                    schemaDefinitions.put(zipEntry.getName(), bytes);
                }
            }

            return schemaDefinitions;
        } catch (IOException e) {
            LOGGER.error("Extracting zipped schema files failed", e);
            throw new UncheckedIOException("Extracting zipped schema files failed", e);
        }
    }

    private void validateURIPattern(URI schemaUri) {
        if (!LEIKA_KEY_PATTERN.matcher(schemaUri.toString()).matches()) {
            throw new RuntimeException(
                    "URI '" + schemaUri + "' does not match required pattern. See " + LEIKA_KEY_PATTERN);
        }
    }
}
